﻿using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Linq;
using Curse.Friends.TwitchService.Chat.Parsing;
using Curse.Friends.TwitchService.Configuration;
using Curse.Logging;

namespace Curse.Friends.TwitchService.Chat.Irc
{
    public class TwitchIrcConnection : BaseTwitchConnection
    {
        private ConcurrentDictionary<string, ChatRoomInfo> _joinedChannels = new ConcurrentDictionary<string, ChatRoomInfo>();
        private DateTime _lastSentMessage;
        private readonly int _maxQueue;
        private readonly int _messageWindowSeconds;

        public bool IsExpired
        {
            get { return DateTime.UtcNow - _lastSentMessage > TimeSpan.FromMinutes(TwitchServiceConfiguration.Instance.UserSessionWindowMinutes); }
        }

        public TwitchIrcConnection(string login, string authToken, int maxQueue = 19, int messageWindowSeconds = 29)
            : base(new TwitchLoginInfo(login, authToken), new LogCategory("IRC:" + login) {Throttle = TimeSpan.FromSeconds(30)})
        {
            _maxQueue = maxQueue;
            _messageWindowSeconds = messageWindowSeconds;
        }

        public MessageFailureReason TrySendMessage(string channel, string message, out long? retryAfter)
        {
            if (!IsConnected)
            {
                var connected = Connect();
                if (!connected)
                {
                    retryAfter = null;
                    return MessageFailureReason.Connection;
                }
            }

            if (_maxQueue > 0)
            {
                lock (_sent)
                {
                    var now = DateTime.UtcNow;
                    _sent.RemoveAll(m => m.Item2 < now.AddSeconds(-_messageWindowSeconds));

                    if (_sent.Count > _maxQueue)
                    {
                        var sendAfterTime = _sent.First().Item2 + TimeSpan.FromSeconds(30);
                        retryAfter = (long) (sendAfterTime - now).TotalMilliseconds;
                        return MessageFailureReason.Throttled;
                    }
                    retryAfter = null;

                    if (DoSend(channel, message))
                    {
                        _sent.Add(Tuple.Create(channel, DateTime.UtcNow));
                        return MessageFailureReason.Success;
                    }
                    return MessageFailureReason.Error;
                }
            }

            retryAfter = null;
            return DoSend(channel, message) ? MessageFailureReason.Success : MessageFailureReason.Error;
        }

        private bool DoSend(string channel, string message)
        {
            var success = _client.SendMessage(channel, message);
            if (success)
            {
                _lastSentMessage = DateTime.UtcNow;
            }
            return success;
        }

        public bool JoinChannel(string channelName)
        {
            return _client.JoinChannel(channelName);
        }

        public bool LeaveChannel(string channelName)
        {
            return _client.LeaveChannel(channelName);
        }

        private readonly List<Tuple<string, DateTime>> _sent = new List<Tuple<string, DateTime>>();

        protected override void CustomConnectedCallback()
        {
            _client.RequestCapabilities("twitch.tv/tags", "twitch.tv/commands");
        }

        public event EventHandler<IrcDisconnectedEventArgs> Disconnected;
 
        protected override void CustomDisconnectedCallback(bool isShuttingDown)
        {
            _joinedChannels = new ConcurrentDictionary<string, ChatRoomInfo>();
            Disconnected?.Invoke(this, new IrcDisconnectedEventArgs(isShuttingDown));
        }

        protected override void CustomTwitchMessageCallback(TwitchMessage message)
        {
            switch (message.MessageType)
            {
                case IrcMessageType.Notice:
                case IrcMessageType.PrivMsg:
                case IrcMessageType.ClearChat:
                case IrcMessageType.HostTarget:
                case IrcMessageType.UserNotice:
                    OnMessageReceived(message);
                    break;
                case IrcMessageType.Roomstate:
                    HandleRoomState(message);
                    break;
                case IrcMessageType.Userstate:
                    HandleUserState(message);
                    break;
                case IrcMessageType.Join:
                    _joinedChannels.TryAdd(message.ChannelName, new ChatRoomInfo());
                    break;
                case IrcMessageType.Part:
                    ChatRoomInfo roomInfo;
                    _joinedChannels.TryRemove(message.ChannelName, out roomInfo);
                    break;
            }
        }

        public event EventHandler<TwitchMessageEventArgs> MessageReceived;

        private void OnMessageReceived(TwitchMessage message)
        {
            MessageReceived?.Invoke(this, new TwitchMessageEventArgs(message));
        }

        private void HandleRoomState(TwitchMessage message)
        {
            ChatRoomInfo roomInfo;
            if (!_joinedChannels.TryGetValue(message.ChannelName, out roomInfo))
            {
                return;
            }

            object tagValue;
            if (message.Tags.TryGetValue(TwitchChatTag.Slow, out tagValue))
            {
                roomInfo.SlowModeThrottle = (int) tagValue;
            }

            if (message.Tags.TryGetValue(TwitchChatTag.SubsOnly, out tagValue))
            {
                roomInfo.SubscribersOnlyMode = (bool) tagValue;
            }
        }

        private void HandleUserState(TwitchMessage message)
        {
            ChatRoomInfo roomInfo;
            if (!_joinedChannels.TryGetValue(message.ChannelName, out roomInfo))
            {
                return;
            }

            roomInfo.UserCategory = UserCategory.None;

            object tagValue;
            if(message.Tags.TryGetValue(TwitchChatTag.Subscriber, out tagValue))
            {
                roomInfo.UserCategory |= UserCategory.Subscriber;
            }

            if (message.Tags.TryGetValue(TwitchChatTag.Mod, out tagValue))
            {
                roomInfo.UserCategory |= UserCategory.Moderator;
            }

            if (message.Tags.TryGetValue(TwitchChatTag.Turbo, out tagValue))
            {
                roomInfo.UserCategory |= UserCategory.Turbo;
            }
        }
    }
}
