using System;
using System.Collections.Generic;
using System.Linq;
using Curse.Extensions;
using Curse.Friends.Data;
using Curse.Friends.Data.Messaging;
using Curse.Friends.Enums;
using Curse.Logging;

namespace Curse.Friends.TwitchService.Chat.Parsing
{
    public enum TwitchChatTag
    {
        Badges,
        Color,
        DisplayName,
        Emotes,
        EmoteSets,
        Id,
        Mod,
        MessageId,
        RoomId,
        Slow,
        SubsOnly,
        Subscriber,
        Timestamp,
        Turbo,
        UserId,
        UserType,
        BanDuration,
        BanReason,
        R9k,
        EmoteOnly,
        BroadcasterLanguage,
        SentTimestamp,
        Bits,
        SystemMessage,
        MonthsParam,
        Login,
        FollowersOnly,
        FollowersAvailable,
        TargetUserID,
        SubPlan,
    }

    public enum IrcMessageType
    {
        Unknown,
        PrivMsg,
        Notice,
        Userstate,
        Roomstate,
        Mode,
        ClearChat,
        Join,
        Ack,
        Ping,
        Cap,
        Part,
        UserNotice,

        IntroWelcomeMessage,
        IntroHostMessage,
        IntroServerNewMessage,
        IntroDash,
        JoinDisplayName,
        JoinEndOfNamesList,
        IntroTwistyPassages,
        IntroDash2,
        IntroGreaterThan,
        HostTarget
    }

    public class TwitchMessageParser
    {
        public static string GetTagStringValue(string input)
        {
            if (input == null)
            {
                return null;
            }

            return input.Replace(@"\s", " ");
        }

        public static ConversationMessageEmoteSubstitution[] ResolveEmoteSubstitutions(TwitchEmoteSubstitution[] substitutions, string messageBody)
        {
            if (substitutions == null || substitutions.Length == 0 || string.IsNullOrEmpty(messageBody))
            {
                return new ConversationMessageEmoteSubstitution[0];
            }

            var emotes = TwitchEmote.MultiGetByIDs(substitutions.Select(s => s.EmoteID)).ToDictionary(e => e.EmoteID);
            var resolvedEmotes = new List<ConversationMessageEmoteSubstitution>();
            foreach (var substitution in substitutions.Where(s => s.StartIndex >= 0 && s.EndIndex < messageBody.Length))
            {
                var resolved = new ConversationMessageEmoteSubstitution
                {
                    EmoteID = substitution.EmoteID,
                    StartIndex = substitution.StartIndex,
                    EndIndex = substitution.EndIndex,
                    MessageText = messageBody.Substring(substitution.StartIndex, substitution.EndIndex - substitution.StartIndex + 1)
                };

                TwitchEmote emote;
                if (emotes.TryGetValue(substitution.EmoteID, out emote))
                {
                    resolved.EmoteHeight = emote.Height;
                    resolved.EmoteWidth = emote.Width;
                    resolved.EmoteSet = emote.EmoticonSet;
                }
                resolvedEmotes.Add(resolved);
            }
            return resolvedEmotes.ToArray();
        }

        public static ConversationMessageBadge[] GetBadges(TwitchBadge[] badges, long channelID)
        {
            if (badges == null || badges.Length == 0)
            {
                return new ConversationMessageBadge[0];
            }

            var chatBadges = TwitchChatBadge.GetAllGlobals().ToDictionary(gb => Tuple.Create(gb.BadgeSet, gb.Version));
            if (channelID > 0)
            {
                foreach (var channelBadge in TwitchChatBadge.GetAllLocal(c => c.ExternalID, channelID.ToString()))
                {
                    chatBadges[Tuple.Create(channelBadge.BadgeSet, channelBadge.Version)] = channelBadge;
                }
            }

            var resolvedBadges = new List<ConversationMessageBadge>();
            foreach (var badge in badges)
            {
                var resolvedBadge = new ConversationMessageBadge
                {
                    BadgeSet = badge.BadgeSet,
                    Version = badge.Version,
                };

                TwitchChatBadge globalBadge;
                if (chatBadges.TryGetValue(Tuple.Create(badge.BadgeSet, badge.Version), out globalBadge))
                {
                    resolvedBadge.Description = globalBadge.Description;
                    resolvedBadge.Title = globalBadge.Title;
                    resolvedBadge.ClickAction = globalBadge.ClickAction;
                    resolvedBadge.ClickUrl = globalBadge.ClickUrl;
                }
                resolvedBadges.Add(resolvedBadge);
            }

            return resolvedBadges.ToArray();
        }

        public static Guid GetMessageID(string id)
        {
            Guid guid;
            Guid.TryParse(id, out guid);
            return guid;
        }

        private static readonly Dictionary<string, TwitchChatNoticeType> _noticeTypes = new Dictionary<string, TwitchChatNoticeType>
        {
            {"whisper_banned", TwitchChatNoticeType.WhisperBanned},
            {"whisper_banned_recipient", TwitchChatNoticeType.WhisperBannedRecipient},
            {"whisper_invalid_args", TwitchChatNoticeType.WhisperInvalidArgs},
            {"whisper_invalid_login", TwitchChatNoticeType.WhisperInvalidLogin},
            {"whisper_invalid_self", TwitchChatNoticeType.WhisperInvalidSelf},
            {"whisper_limit_per_min", TwitchChatNoticeType.WhisperLimitPerMin},
            {"whisper_limit_per_sec", TwitchChatNoticeType.WhisperLimitPerSec},
            {"whisper_restricted", TwitchChatNoticeType.WhisperRestricted},
            {"whisper_restricted_recipient", TwitchChatNoticeType.WhisperRestrictedRecipient},
            {"subs_on", TwitchChatNoticeType.SubsOnlyOn},
            {"subs_off", TwitchChatNoticeType.SubsOnlyOff},
            {"followers_on_zero", TwitchChatNoticeType.FollowersOnlyOnZero},
            {"followers_on", TwitchChatNoticeType.FollowersOnlyOn},
            {"followers_off", TwitchChatNoticeType.FollowersOnlyOff},
            {"emote_only_on", TwitchChatNoticeType.EmoteOnlyOn},
            {"emote_only_off", TwitchChatNoticeType.EmoteOnlyOff},
            {"slow_on", TwitchChatNoticeType.SlowModeOn},
            {"slow_off", TwitchChatNoticeType.SlowModeOff},
            {"r9k_on", TwitchChatNoticeType.R9KOn},
            {"r9k_off", TwitchChatNoticeType.R9KOff},
            {"host_on", TwitchChatNoticeType.HostOn},
            {"host_off", TwitchChatNoticeType.HostOff},
            {"host_target_went_offline", TwitchChatNoticeType.HostTargetWentOffline},
            {"unrecognized_cmd", TwitchChatNoticeType.UnrecognizedCommand},
            {"no_help", TwitchChatNoticeType.NoHelp},
            {"usage_me", TwitchChatNoticeType.UsageMe},
            {"usage_help", TwitchChatNoticeType.UsageHelp},
            {"usage_disconnect", TwitchChatNoticeType.UsageDisconnect},
            {"usage_mods", TwitchChatNoticeType.UsageMods},
            {"usage_subs_on", TwitchChatNoticeType.UsageSubsOn},
            {"usage_subs_off", TwitchChatNoticeType.UsageSubsOff},
            {"usage_emote_only_on", TwitchChatNoticeType.UsageEmoteOnlyOn},
            {"usage_emote_only_off", TwitchChatNoticeType.UsageEmoteOnlyOff},
            {"usage_slow_on", TwitchChatNoticeType.UsageSlowOn},
            {"usage_slow_off", TwitchChatNoticeType.UsageSlowOff},
            {"usage_r9k_on", TwitchChatNoticeType.UsageR9kOn},
            {"usage_r9k_off", TwitchChatNoticeType.UsageR9kOff},
            {"usage_clear", TwitchChatNoticeType.UsageClear},
            {"usage_color", TwitchChatNoticeType.UsageColor},
            {"usage_cheerbadge", TwitchChatNoticeType.UsageCheerBadge},
            {"usage_timeout", TwitchChatNoticeType.UsageTimeout},
            {"usage_mod", TwitchChatNoticeType.UsageMod},
            {"usage_unmod", TwitchChatNoticeType.UsageUnmod},
            {"usage_ban", TwitchChatNoticeType.UsageBan},
            {"usage_unban", TwitchChatNoticeType.UsageUnban},
            {"usage_host", TwitchChatNoticeType.UsageHost},
            {"usage_unhost", TwitchChatNoticeType.UsageUnhost},
            {"usage_commercial", TwitchChatNoticeType.UsageCommercial},
            {"cmds_available", TwitchChatNoticeType.CmdsAvailable},
            {"no_mods", TwitchChatNoticeType.NoMods},
            {"room_mods", TwitchChatNoticeType.RoomMods},
            {"no_permission", TwitchChatNoticeType.NoPermission},
            {"already_subs_on", TwitchChatNoticeType.AlreadySubsOn},
            {"already_subs_off", TwitchChatNoticeType.AlreadySubsOff},
            {"already_emote_only_on", TwitchChatNoticeType.AlreadyEmoteOnlyOn},
            {"already_emote_only_off", TwitchChatNoticeType.AlreadyEmoteOnlyOff},
            {"already_r9k_on", TwitchChatNoticeType.AlreadyR9kOn},
            {"already_r9k_off", TwitchChatNoticeType.AlreadyR9kOff},
            {"turbo_only_color", TwitchChatNoticeType.TurboOnlyColor},
            {"color_changed", TwitchChatNoticeType.ColorChanged},
            {"cheer_badge_deselected", TwitchChatNoticeType.CheerBadgeDeselected},
            {"cheer_badge_selected", TwitchChatNoticeType.CheerBadgeSelected},
            {"no_cheer_badge", TwitchChatNoticeType.NoCheerBadge},
            {"invalid_user", TwitchChatNoticeType.InvalidUser},
            {"bad_timeout_anon", TwitchChatNoticeType.BadTimeoutAnon},
            {"bad_timeout_broadcaster", TwitchChatNoticeType.BadTimeoutBroadcaster},
            {"bad_timeout_staff", TwitchChatNoticeType.BadTimeoutStaff},
            {"bad_timeout_admin", TwitchChatNoticeType.BadTimeoutAdmin},
            {"bad_timeout_global_mod", TwitchChatNoticeType.BadTimeoutGlobalMod},
            {"bad_timeout_self", TwitchChatNoticeType.BadTimeoutSelf},
            {"bad_timeout_mod", TwitchChatNoticeType.BadTimeoutMod},
            {"bad_timeout_duration", TwitchChatNoticeType.BadTimeoutDuration},
            {"timeout_success", TwitchChatNoticeType.TimeoutSuccess},
            {"bad_ban_anon", TwitchChatNoticeType.BadBanAnon},
            {"bad_ban_broadcaster", TwitchChatNoticeType.BadBanBroadcaster},
            {"bad_ban_staff", TwitchChatNoticeType.BadBanStaff},
            {"bad_ban_admin", TwitchChatNoticeType.BadBanAdmin},
            {"bad_ban_global_mod", TwitchChatNoticeType.BadBanGlobalMod},
            {"bad_ban_self", TwitchChatNoticeType.BadBanSelf},
            {"bad_ban_mod", TwitchChatNoticeType.BadBanMod},
            {"bad_unban_no_ban", TwitchChatNoticeType.BadUnbanNoBan},
            {"already_banned", TwitchChatNoticeType.AlreadyBanned},
            {"ban_success", TwitchChatNoticeType.BanSuccess},
            {"unban_success", TwitchChatNoticeType.UnbanSuccess},
            {"tos_ban", TwitchChatNoticeType.TosBanned},
            {"bad_host_self", TwitchChatNoticeType.BadHostSelf},
            {"bad_host_hosting", TwitchChatNoticeType.BadHostHosting},
            {"bad_host_rejected", TwitchChatNoticeType.BadHostRejected},
            {"bad_host_rate_exceeded", TwitchChatNoticeType.BadHostRateExceeded},
            {"bad_host_error", TwitchChatNoticeType.BadHostError},
            {"hosts_remaining", TwitchChatNoticeType.HostsRemaining},
            {"host_success", TwitchChatNoticeType.HostSuccess},
            {"host_success_viewers", TwitchChatNoticeType.HostSuccessViewers},
            {"not_hosting", TwitchChatNoticeType.NotHosting},
            {"bad_unhost_error", TwitchChatNoticeType.BadUnhostError},
            {"host_tagline_length_error", TwitchChatNoticeType.HostTaglineLengthError},
            {"bad_commercial_error", TwitchChatNoticeType.BadCommercialError},
            {"commercial_success", TwitchChatNoticeType.CommercialSuccess},
            {"bad_mod_banned", TwitchChatNoticeType.BadModBanned},
            {"bad_mod_mod", TwitchChatNoticeType.BadModMod},
            {"mod_success", TwitchChatNoticeType.ModSuccess},
            {"bad_unmod_mod", TwitchChatNoticeType.BadUnmodMod},
            {"unmod_success", TwitchChatNoticeType.UnmodSuccess},
            {"bad_slow_duration", TwitchChatNoticeType.BadSlowDuration},
            {"msg_banned", TwitchChatNoticeType.MessageBanned},
            {"msg_timedout", TwitchChatNoticeType.MessageTimedOut},
            {"msg_ratelimit", TwitchChatNoticeType.MessageRateLimit},
            {"msg_duplicate", TwitchChatNoticeType.MessageDuplicate},
            {"msg_subsonly", TwitchChatNoticeType.MessageSubsOnly},
            {"msg_emoteonly", TwitchChatNoticeType.MessageEmotesOnly},
            {"msg_verified_email", TwitchChatNoticeType.MessageVerifiedEmail},
            {"msg_slowmode", TwitchChatNoticeType.MessageSlowMode},
            {"msg_facebook", TwitchChatNoticeType.MessageFacebook},
            {"msg_r9k", TwitchChatNoticeType.MessageR9k},
            {"msg_conversation_mode", TwitchChatNoticeType.MessageConversationMode},
            {"msg_channel_suspended", TwitchChatNoticeType.MessageChannelSuspended},
            {"msg_suspended", TwitchChatNoticeType.MessageSuspended},
            {"msg_rejected", TwitchChatNoticeType.MessageRejected},
            {"resub", TwitchChatNoticeType.Resub},
            {"untimeout_success", TwitchChatNoticeType.UntimeoutSuccess},
            {"untimeout_banned", TwitchChatNoticeType.UntimeoutIsBanned},
            {"usage_untimeout", TwitchChatNoticeType.UsageUntimeout},
        }; 

        public static TwitchChatNoticeType GetNoticeType(string messageId, string channel, string data)
        {
            if (string.IsNullOrEmpty(messageId))
            {
                return !string.IsNullOrEmpty(data) && channel == "*" && data == "Login authentication failed" ? TwitchChatNoticeType.AuthenticationFailed : TwitchChatNoticeType.Unknown;
            }

            return _noticeTypes.GetValueOrDefault(messageId);
        }

        public static IrcMessageType GetType(string commandText)
        {

            switch (commandText.ToLowerInvariant())
            {
                case "privmsg":
                    return IrcMessageType.PrivMsg;
                case "notice":
                    return IrcMessageType.Notice;
                case "userstate":
                    return IrcMessageType.Userstate;
                case "roomstate":
                    return IrcMessageType.Roomstate;
                case "mode":
                    return IrcMessageType.Mode;
                case "clearchat":
                    return IrcMessageType.ClearChat;
                case "join":
                    return IrcMessageType.Join;
                case "ack":
                    return IrcMessageType.Ack;
                case "ping":
                    return IrcMessageType.Ping;
                case "cap":
                    return IrcMessageType.Cap;
                case "part":
                    return IrcMessageType.Part;
                case "hosttarget":
                    return IrcMessageType.HostTarget;
                case "usernotice":
                    return IrcMessageType.UserNotice;

                case "001":
                    return IrcMessageType.IntroWelcomeMessage;
                case "002":
                    return IrcMessageType.IntroHostMessage;
                case "003":
                    return IrcMessageType.IntroServerNewMessage;
                case "004":
                    return IrcMessageType.IntroDash;
                case "353":
                    return IrcMessageType.JoinDisplayName;
                case "366":
                    return IrcMessageType.JoinEndOfNamesList;
                case "372":
                    return IrcMessageType.IntroTwistyPassages;
                case "375":
                    return IrcMessageType.IntroDash2;
                case "376":
                    return IrcMessageType.IntroGreaterThan;
                default:
                    Logger.Debug("Unsupported IRC Message Type: " + commandText);
                    return IrcMessageType.Unknown;
            }
        }

        public static Dictionary<TwitchChatTag, object> ParseTags(string tagsText)
        {
            var tagsDict = new Dictionary<TwitchChatTag, object>();
            if (string.IsNullOrEmpty(tagsText))
            {
                return tagsDict;
            }

            foreach (var tag in tagsText.Split(';'))
            {
                var kvp = tag.Split('=');
                if (kvp.Length != 2 || string.IsNullOrEmpty(kvp[1]))
                {
                    continue;
                }
                switch (kvp[0])
                {
                    case "badges":
                        tagsDict[TwitchChatTag.Badges] = kvp[1].Split(',')
                            .Select(str => str.Split('/'))
                            .Where(split => split.Length == 2)
                            .Select(split => new TwitchBadge {BadgeSet = split[0], Version = split[1]}).ToArray();
                        break;
                    case "color":
                        tagsDict[TwitchChatTag.Color] = kvp[1];
                        break;
                    case "display-name":
                        tagsDict[TwitchChatTag.DisplayName] = GetTagStringValue(kvp[1]);
                        break;
                    case "emotes":
                        tagsDict[TwitchChatTag.Emotes] = kvp[1].Split('/').SelectMany(v =>
                        {
                            var indexOfColon = v.IndexOf(':');
                            var emoteID = int.Parse(v.Substring(0, indexOfColon));
                            return v.Substring(indexOfColon + 1).Split(',').Select(s =>
                            {
                                var indexOfDash = s.IndexOf('-');
                                return new TwitchEmoteSubstitution
                                {
                                    EmoteID = emoteID,
                                    StartIndex = int.Parse(s.Substring(0, indexOfDash)),
                                    EndIndex = int.Parse(s.Substring(indexOfDash + 1))
                                };
                            });
                        }).ToArray();
                        break;
                    case "emote-sets":
                        tagsDict[TwitchChatTag.EmoteSets] = kvp[1];
                        break;

                    case "id":
                        tagsDict[TwitchChatTag.Id] = kvp[1];
                        break;
                    case "mod":
                        tagsDict[TwitchChatTag.Mod] = kvp[1] == "1";
                        break;
                    case "msg-id":
                        tagsDict[TwitchChatTag.MessageId] = kvp[1];
                        break;
                    case "room-id":
                        tagsDict[TwitchChatTag.RoomId] = long.Parse(kvp[1]);
                        break;
                    case "slow":
                        tagsDict[TwitchChatTag.Slow] = int.Parse(kvp[1]);
                        break;
                    case "subs-only":
                        tagsDict[TwitchChatTag.SubsOnly] = kvp[1] == "1";
                        break;
                    case "subscriber":
                        tagsDict[TwitchChatTag.Subscriber] = kvp[1] == "1";
                        break;
                    case "tmi-sent-ts":
                        tagsDict[TwitchChatTag.Timestamp] = long.Parse(kvp[1]);
                        break;
                    case "turbo":
                        tagsDict[TwitchChatTag.Turbo] = kvp[1] == "1";
                        break;
                    case "user-id":
                        tagsDict[TwitchChatTag.UserId] = long.Parse(kvp[1]);
                        break;
                    case "user-type":
                        tagsDict[TwitchChatTag.UserType] = kvp[1];
                        break;
                    case "ban-duration":
                        tagsDict[TwitchChatTag.BanDuration] = long.Parse(kvp[1]);
                        break;
                    case "ban-reason":
                        tagsDict[TwitchChatTag.BanReason] = GetTagStringValue(kvp[1]);
                        break;

                    case "r9k":
                        tagsDict[TwitchChatTag.R9k] = kvp[1] == "1";
                        break;
                    case "emote-only":
                        tagsDict[TwitchChatTag.EmoteOnly] = kvp[1] == "1";
                        break;
                    case "sent-ts":
                        tagsDict[TwitchChatTag.SentTimestamp] = kvp[1];
                        break;
                    case "broadcaster-lang":
                        tagsDict[TwitchChatTag.BroadcasterLanguage] = kvp[1];
                        break;
                    case "bits":
                        tagsDict[TwitchChatTag.Bits] = int.Parse(kvp[1]);
                        break;
                    case "system-msg":
                        tagsDict[TwitchChatTag.SystemMessage] = GetTagStringValue(kvp[1]);
                        break;
                    case "msg-param-months":
                        tagsDict[TwitchChatTag.MonthsParam] = int.Parse(kvp[1]);
                        break;
                    case "login":
                        tagsDict[TwitchChatTag.Login] = GetTagStringValue(kvp[1]);
                        break;

                    case "followers-only":
                        tagsDict[TwitchChatTag.FollowersOnly] = kvp[1] == "1";
                        break;
                    case "followers-available":
                        tagsDict[TwitchChatTag.FollowersAvailable] = int.Parse(kvp[1]);
                        break;
                    case "target-user-id":
                        tagsDict[TwitchChatTag.TargetUserID] = long.Parse(kvp[1]);
                        break;
                    case "sub_plan":
                        tagsDict[TwitchChatTag.SubPlan] = kvp[1];
                        break;

                    default:
                        Logger.Debug("Unknown tag: " + tag);
                        break;
                }
            }
            return tagsDict;
        }
    }
}