﻿using System;
using System.IO;
using System.Net.Sockets;
using System.Text;
using System.Threading;
using Curse.Friends.TwitchService.Chat.Irc;

namespace Curse.Friends.TwitchService.Chat
{
    public class TwitchClient
    {
        private readonly StringBuilder _builder = new StringBuilder(512);

        private TrackingTcpClient _client;
        private readonly TwitchLoginInfo _info;

        public TwitchClient(TwitchLoginInfo info)
        {
            _info = info;
        }

        public bool IsConnected { get { return _client?.Client?.Connected ?? false; } }

        public bool IsShuttingDown { get; private set; }


        private readonly byte[] _readBuffer = new byte[1024];

        public bool Connect()
        {
            if (_isDisposed)
            {
                return false;
            }

            var oldClient = _client;

            try
            {
                var newClient = new TrackingTcpClient(AddressFamily.InterNetwork);
                _client = newClient;
                _builder.Clear();

                try
                {
                    newClient.Connect("irc.twitch.tv", 6667);
                    if (newClient.Connected)
                    {
                        newClient.GetStream().BeginRead(_readBuffer, 0, _readBuffer.Length, ReadCallback, newClient);
                        SendRawMessageFormatted("PASS {0}", _info.Password);
                        SendRawMessageFormatted("NICK {0}", _info.Nickname);
                        return true;
                    }
                    return false;
                }
                catch (ObjectDisposedException)
                {
                    return false;
                }
                catch (Exception ex)
                {
                    // log
                    OnEvent(Error, new IrcErrorEventArgs(ex));
                    return false;
                }
            }
            finally
            {
                try
                {
                    oldClient?.Close();
                }
                catch (Exception ex)
                {
                    OnEvent(Error, new IrcErrorEventArgs(ex));
                }
            }
        }

        private void ReadCallback(IAsyncResult result)
        {
            if (_isDisposed)
            {
                return;
            }

            try
            {
                var client = result.AsyncState as TcpClient;
                if (client == null)
                {
                    return;
                }

                if (client != _client)
                {
                    // Client reconnected, close old client and clean async state if needed
                    if (client.Connected)
                    {
                        client.GetStream().EndRead(result);
                        client.Close();
                    }
                    return;
                }

                if (!client.Connected)
                {
                    OnEvent(Disconnected);
                    return;
                }

                var bytesRead = client.GetStream().EndRead(result);
                if (bytesRead == 0)
                {
                    client.Close();
                    OnEvent(Disconnected);
                    return;
                }

                var str = Encoding.UTF8.GetString(_readBuffer, 0, bytesRead);

                foreach (var individualReply in str.Split('\n'))
                {
                    _builder.Append(individualReply);
                    if (individualReply.EndsWith("\r"))
                    {
                        OnMessageReceived(_builder.ToString().TrimEnd('\r'));
                        _builder.Clear();
                    }
                }
                client.GetStream().BeginRead(_readBuffer, 0, _readBuffer.Length, ReadCallback, client);
            }
            catch (ObjectDisposedException)
            {
                OnEvent(Disconnected);
            }
            catch (Exception ex)
            {
                OnEvent(Error, new IrcErrorEventArgs(ex));
                OnEvent(Disconnected);
            }
        }

        public void Stop()
        {
            if (_isDisposed)
            {
                return;
            }

            IsShuttingDown = true;
            var client = _client;
            if (client != null)
            {
                client.Close();
                _client = null;
            }
        }

        #region Commands

        public bool SendMessage(string channel, string message)
        {
            return SendRawMessageFormatted("PRIVMSG #{0} :{1}", channel, message);
        }

        public bool RequestCapabilities(params string[] capabilities)
        {
            return SendRawMessageFormatted("CAP REQ :{0}", string.Join(" ", capabilities));
        }

        public bool JoinChannel(string channel)
        {
            return SendRawMessageFormatted("JOIN #{0}", channel);
        }

        public bool LeaveChannel(string channel)
        {
            return SendRawMessageFormatted("PART #{0}", channel);
        }

        public bool SendRawMessageFormatted(string format, params object[] args)
        {
            return SendRawMessage(string.Format(format, args));
        }

        public bool SendRawMessage(string message)
        {
            if (_isDisposed)
            {
                return false;
            }

            try
            {
                var client = _client;
                if (client != null)
                {
                    var writer = new StreamWriter(client.GetStream());
                    writer.WriteLine(message);
                    writer.Flush();
                    return true;
                }
                return false;
            }
            catch (ObjectDisposedException)
            {
                return false;
            }
            catch (Exception ex)
            {
                OnEvent(Error, new IrcErrorEventArgs(ex));
                return false;
            }
        }

        #endregion

        #region Events

        public event EventHandler Disconnected;

        public event EventHandler<IrcErrorEventArgs> Error; 

        private void OnEvent(EventHandler handler)
        {
            handler?.Invoke(this, EventArgs.Empty);
        }

        private void OnEvent<TArgs>(EventHandler<TArgs> handler, TArgs args)
        {
            handler?.Invoke(this, args);
        }

        public event EventHandler<RawMessageReceivedEventArgs> RawMessageReceived;

        private void OnMessageReceived(string message)
        {
            if (message.StartsWith("PING"))
            {
                SendRawMessage(message.Replace("PING", "PONG"));
            }

            try
            {
                if (RawMessageReceived != null)
                {
                    RawMessageReceived(this, new RawMessageReceivedEventArgs(message));
                }
            }
            catch(Exception ex)
            {
                OnEvent(Error, new IrcErrorEventArgs(ex));
            }
        }

        #endregion

        #region Dispose

        private readonly object _disposeLock = new object();
        private bool _isDisposed;

        public void Dispose()
        {
            var acquired = Monitor.TryEnter(_disposeLock, TimeSpan.FromSeconds(1));

            if (!acquired)
            {
                return;
            }

            if (_isDisposed)
            {
                return;
            }

            try
            {
                _isDisposed = true;

                var client = _client;
                if (client != null)
                {
                    _client = null;
                    client.Close();
                }
            }
            catch (Exception ex)
            {
                OnEvent(Error, new IrcErrorEventArgs(ex));
            }
            finally
            {
                Monitor.Exit(_disposeLock);
            }
        }

        #endregion
    }
}
