﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Linq.Expressions;
using Curse.Aerospike;
using Curse.CloudServices.Jobs;
using Curse.Friends.Data;
using Curse.Friends.ServerHosting;
using Curse.Logging;

namespace Curse.Friends.TwitchService.Jobs
{
    class CommunityHostableRegionJob : BaseJob
    {
        public override int RunFrequencyMinutes { get { return 1; } }

        public override JobScheduleMode ScheduleMode { get {return JobScheduleMode.Interval;} }

        public override bool OnlyDefaultRegion { get { return false; } }

        public override void Run()
        {
            var localRegion = ExternalCommunity.LocalConfigID;

            Logger.Info($"Updating hostable status of ExternalCommunities in region {localRegion}");

            var allHosts = TwitchHost.GetAllLocal(p => p.IndexMode, IndexMode.Default);
            var fixUnknowns = allHosts.Length > 0;
            var hostnames = new HashSet<string>(allHosts.Select(h => h.MachineName));

            var total = 0;
            var unhosted = 0;
            var hosted = 0;
            var unknownHosts = new Dictionary<string,int>();
            ExternalCommunity.BatchOperateOnIndexLocal(c=>c.RegionID, localRegion, 1000, communities =>
            {
                foreach (var community in communities)
                {
                    total++;

                    var updates = new List<Expression<Func<ExternalCommunity, object>>>();

                    if (community.HasMappedGroups && community.HostableRegion == 0)
                    {
                        hosted++;
                        community.HostableRegion = localRegion;
                        updates.Add(c => c.HostableRegion);
                    }
                    else if (!community.HasMappedGroups && community.HostableRegion > 0)
                    {
                        unhosted++;
                        community.HostableRegion = 0;
                        updates.Add(c => c.HostableRegion);
                    }

                    if (fixUnknowns && !string.IsNullOrWhiteSpace(community.MachineName) && !hostnames.Contains(community.MachineName))
                    {
                        if (unknownHosts.ContainsKey(community.MachineName))
                        {
                            unknownHosts[community.MachineName] += 1;
                        }
                        else
                        {
                            unknownHosts[community.MachineName] = 1;
                        }

                        community.MachineName = string.Empty;
                        updates.Add(c => c.MachineName);
                    }

                    if (updates.Count > 0)
                    {
                        community.Update(updates.ToArray());
                    }
                }
            });

            Logger.Info($"Finished updating hostable status of ExternalCommunities in region {localRegion}", new
            {
                TotalCommunities = total,
                CommunitiesSetHostable = hosted,
                CommunitiesSetUnhostable = unhosted,
                UnknownHostsFixed = unknownHosts
            });
        }
    }
}
