﻿using System.Linq;
using Curse.Aerospike;
using Curse.CloudServices.Jobs;
using Curse.Extensions;
using Curse.Friends.Data;
using Curse.Friends.Enums;
using Curse.Friends.TwitchApi;
using Curse.Friends.TwitchService.Configuration;
using Curse.Logging;

namespace Curse.Friends.TwitchService.Jobs
{
    public class TwitchEmoteSyncJob : BaseJob
    {
        public override JobScheduleMode ScheduleMode
        {
            get { return JobScheduleMode.Interval; }
        }

        public override int RunFrequencyMinutes
        {
            get { return TwitchServiceConfiguration.Instance.FullSyncIntervalMinutes; }
        }

        public override bool OnlyDefaultRegion
        {
            get { return true; }
        }

        public override void Run()
        {
            var emoteResponse = TwitchApiHelper.Default.GetAllEmoticons();
            if (emoteResponse.Status != TwitchResponseStatus.Success || emoteResponse.Value.Emoticons == null)
            {
                Logger.Debug("Failed to get full emoticons list from Twitch", emoteResponse);
                return;
            }

            var emoteImagesResponse = TwitchApiHelper.Default.GetAllEmoticonImages();
            if (emoteImagesResponse.Status != TwitchResponseStatus.Success || emoteImagesResponse.Value.Emoticons == null)
            {
                Logger.Debug("Failed to get full emoticon images list from Twitch", emoteImagesResponse);
                return;
            }

            // put all Twitch Emotes into consumable format
            var emoteList = TwitchEmoteDescriptor.GetDescriptors(emoteResponse.Value, emoteImagesResponse.Value).GroupBy(e => e.ID).ToDictionary(g => g.Key, g => g.First());

            // Add or update current emotes
            var existing = TwitchEmote.GetAllLocal(e => e.IndexMode, IndexMode.Default).ToDictionary(e => e.EmoteID);
            var avatars = Avatar.MultiGetLocal(existing.Select(e => new KeyInfo((int)AvatarType.TwitchEmote, e.Key.ToString()))).ToDictionary(a => a.EntityID);
            foreach (var kvp in emoteList)
            {
                kvp.Value.UpdateModel(existing.GetValueOrDefault(kvp.Key), avatars.GetValueOrDefault(kvp.Key.ToString()));
            }

            // Remove old emotes
            foreach (var kvp in existing)
            {
                TwitchEmoteDescriptor twitchEmote;
                if (emoteList.TryGetValue(kvp.Key, out twitchEmote) || kvp.Value.IsDeleted)
                {
                    continue;
                }

                kvp.Value.IsDeleted = true;
                kvp.Value.Update(e => e.IsDeleted);

                var entityID = kvp.Key.ToString();

                var avatar = Avatar.GetLocal((int)AvatarType.TwitchEmote, entityID);
                if (avatar == null || string.IsNullOrEmpty(avatar.Url))
                {
                    // Already deleted or missing, no avatar update needed
                    continue;
                }

                AvatarUpdatePump.UpdateAvatar(AvatarType.TwitchEmote, entityID, string.Empty);
            }
        }
    }
}
