﻿using System;
using System.Collections.Generic;
using System.Linq;
using Curse.Aerospike;
using Curse.Friends.Data;
using Curse.Friends.Enums;
using Curse.Friends.NotificationContracts;
using Curse.Friends.Tracing;
using Curse.Friends.Data.Queues;
using Curse.Friends.TwitchApi;
using Curse.Logging;

namespace Curse.Friends.TwitchService.QueueProcessors
{
    public class TwitchAccountSyncProcessor
    {
        private static readonly FilteredUserLogger Logger = new FilteredUserLogger("TwitchAccountSync");
        private static readonly LogCategory BanLogger = new LogCategory("TwitchAccountSyncBans") { Throttle = TimeSpan.FromMinutes(5), ReleaseLevel = LogLevel.Debug };

        public static void Process(TwitchAccountSyncWorker worker)
        {
            try
            {
                
                var account = ExternalAccount.GetByTwitchUserID(worker.TwitchID);
                if (account == null)
                {
                    Logger.Warn(worker.TwitchID, "Unable to resync account: Account could not be retrieved from database.", worker);
                    return;
                }

                var community = ExternalCommunity.GetByTwitchID(worker.TwitchID);

                if (string.IsNullOrEmpty(account.AuthToken))
                {
                    Logger.Log(worker.TwitchID, "Unable to resync account: No auth token.", new {account.ExternalID, account.ExternalUsername, account.ExternalDisplayName});
                    return;
                }

                if (!worker.BypassThrottle && DateTime.UtcNow.Subtract(account.LastSyncTime) < TimeSpan.FromMinutes(15))
                {
                    Logger.Log(worker.TwitchID, "Skipping account sync, since it has happened recently.", new {account.ExternalID, account.ExternalUsername, account.ExternalDisplayName});
                    // Try not to spam Twitch
                    return;
                }

                account.LastSyncTime = DateTime.UtcNow;
                account.Update(a => a.LastSyncTime);

                var falselyUnauthorized = false;
                if (account.NeedsReauthentication)
                {
                    falselyUnauthorized = TwitchModelHelper.TestUnauthorizedTokenValidity(account.ClientID, account.AuthToken);
                    if (!falselyUnauthorized)
                    {
                        // Truly invalid token, wait for a reauth to sync
                        Logger.Log(worker.TwitchID, "Unable to resync account: Auth token is truly unauthorized.",
                            new {account.ExternalID, account.ExternalUsername, account.ExternalDisplayName});
                        return;
                    }
                   
                    Logger.Log(worker.TwitchID, "Detected an account falsely marked as need reauthentication.", new {account.ExternalID, account.ExternalUsername, account.ExternalDisplayName});                    
                }

                // Emotes
                var allStats = UserStatistics.MultiGetLocal(account.MappedUsers.Select(id => new KeyInfo(id)));
                foreach (var stats in allStats)
                {
                    try
                    {
                        // Use a throttle to avoid double requests on link account changes
                        stats.EnsureEmotes(TimeSpan.FromMinutes(5));
                    }
                    catch (Exception ex)
                    {
                        Logger.Warn(ex, null, account, "Failed to sync emotes for a user", new {stats.UserID, stats.TwitchID});
                    }
                }

                // Fix Broken Accounts
                FixBrokenAccount(account);

                // Check Ban Status
                FixBanStatus(account);

                var wasPartnered = account.IsPartnered || (community?.CanHaveSubs ?? false);
                try
                {
                    var oldAvatar = account.AvatarUrl;
                    TwitchModelHelper.UpdateAccount(account);
                    if(oldAvatar != account.AvatarUrl)
                    {
                        AvatarUpdatePump.UpdateAvatar(AvatarType.SyncedAccount, account.GetAvatarEntityID(), account.AvatarUrl);
                    }
                }
                catch (InvalidOperationException ex)
                {
                    Logger.Warn(ex, null, account, "Failed to update account from Twitch, skipping UpdateAccount step",
                        new {Account = account.GetLogData(), HasAuthToken = !string.IsNullOrEmpty(account.AuthToken)});
                }

                try
                {
                    community = TwitchModelHelper.CreateOrUpdateCommunity(account.ExternalID, account);
                }
                catch (DataNotFoundException ex)
                {                    
                    Logger.Warn(ex, null, account, "Unable to create or update community due to an API error", new { account = account.GetLogData() });
                    return;
                }              

                if (!wasPartnered && (account.IsPartnered || community.CanHaveSubs))
                {
                    ProcessNewPartnership(community, account);
                }

                if (falselyUnauthorized)
                {
                    // Shouldn't actually be marked for reauth - missing a scope that needs to be added twitch-side
                    Logger.Log(worker.TwitchID, "Unmarking Account for Reauth because the token is valid, but was missing scopes", new {account.ExternalUsername, account.ExternalID});
                    account.NeedsReauthentication = false;
                    account.Update(a => a.NeedsReauthentication);

                    if (community.HasMappedGroups)
                    {
                        ExternalCommunityCoordinator.OwnerReauthenticated(community, account);
                    }
                }

                // Sync subs
                if (community.CanHaveSubs && community.HasMappedGroups)
                {
                    TwitchCommunitySubscriptionsWorker.Create(community);
                    // TODO: TwitchUserSubscriptionsWorker.Create(account);
                }

                // Sync follows
                TwitchUserFollowsWorker.Create(account);

                // Sync mods.
                if (community.HasMappedGroups)
                {
                    TwitchCommunityModsWorker.Create(community);
                }

                // Sync owner role
                community.AddMemberRole(account.ExternalID, GroupRoleTag.SyncedOwner, account.ExternalUsername, account.ExternalDisplayName, community.ExternalDateCreated);
                foreach (var userID in account.MappedUsers)
                {
                    ExternalUserSyncWorker.Create(userID, worker.TwitchID, AccountType.Twitch, GroupRoleTag.SyncedOwner);
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Error processing Twitch Account Sync Worker", worker);
            }
        }

        private static void FixBanStatus(ExternalAccount account)
        {
            if (!account.IsMerged)
            {
                return;
            }

            try
            {
                var resp = TwitchApiHelper.Default.GetPublicUser(account.ExternalID);
                if(resp.Status == TwitchResponseStatus.Success)
                {
                    if (UserBan.IsBannedFrom(account.MergedUserID, UserBanType.Social))
                    {
                        UserBan.UnbanUser(account.MergedUserID, UserBanType.Social);
                        BanLogger.Debug("Social unbanned a user", new { Account = account.GetLogData() });
                    }
                }
                else if(resp.Status == TwitchResponseStatus.Unprocessable || resp.StatusCode == 422)
                {
                    if (!UserBan.IsBannedFrom(account.MergedUserID, UserBanType.Social))
                    {
                        UserBan.BanUser(account.MergedUserID, UserBanType.Social);
                        BanLogger.Debug("Social banned a user", new { Account = account.GetLogData() });
                    }
                }
            }
            catch(Exception ex)
            {
                Logger.Warn(ex, null, account, "Failed to fix ban status", new { Account = account.GetLogData() });
            }
        }

        private static void FixBrokenAccount(ExternalAccount account)
        {
            try
            {
                if (account.MappedUsers == null)
                {
                    account.MappedUsers = new HashSet<int>();
                    account.Update(a => a.MappedUsers);
                }

                var links = ExternalAccountMapping.GetAllLocal(a => a.ExternalID, account.ExternalID);
                var added = false;
                foreach (var link in links.Where(l => !l.IsDeleted))
                {
                    added |= account.MappedUsers.Add(link.UserID);
                    if (added)
                    {
                        Logger.Log(null, account, "Fixing Twitch account", new {account.ExternalUsername, link.UserID});
                        TwitchAccountResolver.UserLinkChanged(link.UserID, account);
                    }
                }

                if (added)
                {
                    account.Update(a => a.MappedUsers);
                }
            }
            catch (Exception ex)
            {
                Logger.Warn(ex, null, account, "Failed to fix broken Twitch account", account.GetLogData());
            }
        }

        private static void ProcessNewPartnership(ExternalCommunity community, ExternalAccount account)
        {
            try
            {
                var groups = Group.MultiGetLocal(community.MappedGroups.Select(id => new KeyInfo(id)));
                var roles = community.GetRoles();
                foreach (var role in roles.Where(r => r.CheckPremium()).OrderBy(r=>r.RoleRank))
                {
                    foreach (var group in groups)
                    {
                        Group.CreateSyncedRole(group.EnsureWritable(), role);
                    }
                }

                if (community.IsPartnered)
                {
                    foreach (var link in ExternalAccountMapping.MultiGetLocal(account.MappedUsers.Select(id => new KeyInfo(id, account.Type, account.ExternalID))))
                    {
                        var notification = new ExternalAccountChangedNotification
                        {
                            ChangeType = ExternalAccountChangeType.Partnered,
                            Account = account.ToContract(false, link)
                        };
                        ClientEndpoint.DispatchNotification(link.UserID, ep => ExternalAccountChangedNotifier.Create(notification, ep));
                    }
                }
            }
            catch (Exception ex)
            {
                Logger.Warn(ex, null, account, "Failed to add premium roles", account.GetLogData());
            }
        }

    }
}
