﻿using System;
using System.Linq;
using Curse.Extensions;
using Curse.Friends.Data;
using Curse.Friends.Data.DerivedModels;
using Curse.Friends.Enums;
using Curse.Friends.Tracing;
using Curse.Friends.TwitchApi;
using Curse.Logging;

namespace Curse.Friends.TwitchService.QueueProcessors
{
    public class TwitchCommunitySubscriptionsProcessor
    {
        private static readonly FilteredUserLogger Logger = new FilteredUserLogger("CommunitySubsProcessor");

        public static void Process(TwitchCommunitySubscriptionsWorker worker)
        {
            var owner = ExternalAccount.GetLocal(worker.TwitchID, AccountType.Twitch);
            var community = ExternalCommunity.GetLocal(worker.TwitchID, AccountType.Twitch);
            if (owner == null || community == null)
            {
                Logger.Log(null, owner, "Unable to sync subscribers, account or community was null", worker);
                return;
            }

            if (!community.CanHaveSubs)
            {
                Logger.Log(null, owner, "Skipping subscriber sync, community is not partnered");
                return;
            }

            // If the account needs reauth, we assume it is not possible to access the subscriber sync APIs
            if (owner.NeedsReauthentication)
            {
                Logger.Log(null, owner, "Skipping subscriber sync, owner account needs reauthentication", owner.GetLogData());
                return;
            }

            // Always check recents
            SyncRecentSubscribers(owner, community);

            if (worker.ForceFullSync || community.ShouldSyncSubs())
            {
                SyncSubscribers(owner, community);
            }
        }

        public static void SyncRecentSubscribers(ExternalAccount owner, ExternalCommunity community)
        {
            try
            {
                Logger.Log(null, owner, "[Recent] Syncing recent subscribers", community.GetLogData());

                var subscriptionsResponse = TwitchApiHelper.GetClient(owner.ClientID).GetSubscriptions(community.ExternalID, owner.AuthToken, 0, 100, "desc");
                if (subscriptionsResponse.Status != TwitchResponseStatus.Success)
                {
                    if (subscriptionsResponse.Status == TwitchResponseStatus.Unauthorized)
                    {
                        TwitchModelHelper.MarkForReauthorization(owner, "GetSubscriptions");
                    }

                    switch (subscriptionsResponse.Status)
                    {
                        case TwitchResponseStatus.NotFound:
                        case TwitchResponseStatus.TwitchServerError:
                        case TwitchResponseStatus.Unprocessable:
                            Logger.Log(null, owner, "[Recent] Failed to sync subscribers", new {subscriptionsResponse, externalAccount = owner.GetLogData()});
                            break;
                        default:
                            Logger.Warn(null, owner, "[Recent] Failed to sync subscribers", new {subscriptionsResponse, externalAccount = owner.GetLogData()});
                            break;
                    }
                    return;
                }

                if (subscriptionsResponse.Value == null || subscriptionsResponse.Value.Subscriptions == null)
                {
                    Logger.Log(null, owner, "[Recent] Failed to retrieve subscriptions, response has invalid format", new {Owner = owner.GetLogData(), subscriptionResponse = subscriptionsResponse});
                    return;
                }

                if (subscriptionsResponse.Value.Subscriptions.Length == 0)
                {
                    Logger.Log(null, owner, "[Recent] Skipping subscriber sync, community has no subscribers", new {Community = community.GetLogData()});
                    return;
                }

                var mostRecentTime = subscriptionsResponse.Value.Subscriptions.Max(s => s.CreatedAt);
                if (mostRecentTime <= community.DateRecentSub)
                {
                    Logger.Log(null, owner, "[Recent] Skipping subscriber sync, no more recent subs were found", owner.GetLogData());
                    return;
                }

                ProcessSubscriptions(community, subscriptionsResponse.Value.Subscriptions, GroupRoleTag.SyncedSubscriber);
                ProcessSubscriptions(community, subscriptionsResponse.Value.Subscriptions, GroupRoleTag.SyncedSubscriberTier2);
                ProcessSubscriptions(community, subscriptionsResponse.Value.Subscriptions, GroupRoleTag.SyncedSubscriberTier3);

                community.DateRecentSub = mostRecentTime;
                community.Update(c => c.DateRecentSub);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "[Recent] Failed to sync subscribers", new {Community = community.GetLogData()});
            }
        }

        private static void ProcessSubscriptions(ExternalCommunity community, Subscription[] subscriptions, GroupRoleTag role)
        {
            var existingMemberships = ExternalCommunityMembership.GetAllLocal(m => m.CommunityTypeAndRoleIndex, community.ExternalID + AccountType.Twitch + role)
                .ToDictionary(m => m.ExternalUserID);

            var subs = subscriptions.GroupBy(s => s.User.ID).Select(g => g.First());

            if(role == GroupRoleTag.SyncedSubscriberTier2)
            {
                subs = subs.Where(sub => sub.Plan == "2000" || sub.Plan == "3000");
            }
            else if(role == GroupRoleTag.SyncedSubscriberTier3)
            {
                subs = subs.Where(sub => sub.Plan == "3000");
            }

            var subDictionary = subs
                .ToDictionary(
                    s => s.User.ID,
                    s => new NewCommunityMember
                    {
                        ExternalUserID = s.User.ID,
                        ExternalUsername = s.User.Name,
                        ExternalUserDisplayName = s.User.DisplayName,
                        RoleDate = s.CreatedAt
                    });

            if(subDictionary.Count == 0 && existingMemberships.Count(m=>m.Value.Status == ExternalCommunityMembershipStatus.Active) == 0)
            {
                // Skip this - nobody has these roles, don't resolve anything
                return;
            }

            community.ResolveMemberships(existingMemberships, subDictionary, role, true);
            ExternalCommunityMemberSyncWorker.Create(community.RegionID, community.ExternalID, AccountType.Twitch, role);
        }

        public static void SyncSubscribers(ExternalAccount owner, ExternalCommunity community)
        {
            try
            {
                Logger.Log(null, owner, "[Full] Syncing all subscribers", community.GetLogData());

                var subscriptionsResponse = TwitchApiHelper.GetClient(owner.ClientID).GetAllSubscriptions(owner.ExternalID, owner.AuthToken);
                if (subscriptionsResponse.Status != TwitchResponseStatus.Success)
                {
                    if (subscriptionsResponse.Status == TwitchResponseStatus.Unauthorized)
                    {
                        TwitchModelHelper.MarkForReauthorization(owner, "GetAllSubscriptions");
                    }

                    switch (subscriptionsResponse.Status)
                    {
                        case TwitchResponseStatus.NotFound:
                        case TwitchResponseStatus.TwitchServerError:
                        case TwitchResponseStatus.Unprocessable:
                            Logger.Log(null, owner, "[Full] Failed to sync subscribers", new { subscriptionsResponse, externalAccount = owner.GetLogData() });
                            break;
                        default:
                            Logger.Warn(null, owner, "[Full] Failed to sync subscribers", new { subscriptionsResponse, externalAccount = owner.GetLogData() });
                            break;
                    }
                    return;
                }
                var subscriptions = subscriptionsResponse.Value.ToArray();

                ProcessSubscriptions(community, subscriptions, GroupRoleTag.SyncedSubscriber);
                ProcessSubscriptions(community, subscriptions, GroupRoleTag.SyncedSubscriberTier2);
                ProcessSubscriptions(community, subscriptions, GroupRoleTag.SyncedSubscriberTier3);

                if (subscriptions.Length > 0)
                {
                    community.DateRecentSub = subscriptions.Max(s => s.CreatedAt);
                }
                community.SubSyncTimestamp = DateTime.UtcNow.ToEpochMilliseconds();
                community.Update(s => s.SubSyncTimestamp, s => s.DateRecentSub);

                Logger.Log(null, owner, "[Full] Subscribers synced", new
                {
                    externalAccount = owner.GetLogData(),
                    subCountFromApi = subscriptions.Length,
                });
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "[Full] Failed to sync subscribers", new { Community = community.GetLogData() });
            }
        }

    }
}
