﻿using System.Linq;
using Curse.Aerospike;
using Curse.Friends.Data;
using Curse.Friends.Enums;
using Curse.Friends.Tracing;
using Curse.Friends.TwitchApi;
using Curse.Logging;

namespace Curse.Friends.TwitchService.QueueProcessors
{
    class TwitchServerRoleProcessor
    {
        private static readonly FilteredUserLogger Logger = new FilteredUserLogger("TwitchServerRoles");

        public static void Process(TwitchServerRoleWorker worker)
        {
            Logger.Log(worker.UserID, "Checking server roles for new group member", worker);

            var linkedCommunities = ExternalCommunityMapping.GetAllLocal(c => c.GroupID, worker.ServerID).Where(c => c.Type == AccountType.Twitch && !c.IsDeleted).ToArray();
            if (!linkedCommunities.Any())
            {
                Logger.Log(worker.UserID, "Skipping Twitch role checks, server has no linked Twitch communities!", worker);
                return;
            }

            // Get linked accounts or just the merged account?
            var linkedAccounts = ExternalAccountMapping.GetAllLocal(a => a.UserID, worker.UserID).Where(a => a.Type == AccountType.Twitch && !a.IsDeleted).ToArray();
            if (!linkedAccounts.Any())
            {
                Logger.Log(worker.UserID, "Skipping Twitch role checks, user has no linked Twitch accounts!", worker);
                return;
            }

            var accounts = ExternalAccount.MultiGetLocal(linkedAccounts.Select(a => new KeyInfo(a.ExternalID, a.Type))).ToDictionary(a => a.ExternalID);
            var communities = ExternalCommunity.MultiGetLocal(linkedCommunities.Select(c => new KeyInfo(c.ExternalID, c.Type))).ToDictionary(c => c.ExternalID);
            var communityOwners = ExternalAccount.MultiGetLocal(linkedCommunities.Select(c => new KeyInfo(c.ExternalID, c.Type))).ToDictionary(c => c.ExternalID);

            foreach (var linkedAccount in linkedAccounts)
            {
                ExternalAccount account;
                if (!accounts.TryGetValue(linkedAccount.ExternalID, out account))
                {
                    Logger.Warn(worker.UserID, "Couldn't find the account for a linked account", new { linkedAccount.ExternalID, linkedAccount.ExternalName });
                    continue;
                }

                foreach (var linkedCommunity in linkedCommunities)
                {
                    ExternalCommunity community;
                    if (!communities.TryGetValue(linkedCommunity.ExternalID, out community))
                    {
                        Logger.Warn(worker.UserID, "Couldn't find the community for a linked community", new {linkedCommunity.ExternalID, linkedCommunity.ExternalName});
                        continue;
                    }

                    //var follow = TwitchApiHelper.Default.GetUserChannelFollow(account.ExternalID, community.ExternalID);
                    var follow = TwitchApiHelper.Default.GetUserChannelFollowV3(account.ExternalUsername, community.ExternalName);
                    switch (follow.Status)
                    {
                        case TwitchResponseStatus.Success:
                            Logger.Log(worker.UserID, "Adding follower role");
                            community.AddMemberRole(account.ExternalID, GroupRoleTag.SyncedFollower, account.ExternalUsername, account.ExternalDisplayName, follow.Value.CreatedAt);
                            break;
                        case TwitchResponseStatus.NotFound:
                            break;
                        default:
                            Logger.Log(worker.UserID, "Failed to fetch follower role from Twitch", follow);
                            break;
                    }

                    ExternalAccount owner;
                    if (communityOwners.TryGetValue(linkedCommunity.ExternalID, out owner) && owner.IsPartnered)
                    {
                        var subscription = TwitchApiHelper.Default.GetUserSubscription(linkedAccount.ExternalID, linkedCommunity.ExternalID, owner.AuthToken);
                        switch (subscription.Status)
                        {
                            case TwitchResponseStatus.Success:
                                Logger.Log(worker.UserID, "Adding subscriber role");
                                community.AddMemberRole(account.ExternalID, GroupRoleTag.SyncedSubscriber, account.ExternalUsername, account.ExternalDisplayName, subscription.Value.CreatedAt);
                                break;
                            case TwitchResponseStatus.NotFound:
                                // Not found just means they aren't subbed
                                break;
                            default:
                                Logger.Log(worker.UserID, "Failed to fetch subscriber role from Twitch", subscription);
                                break;
                        }
                    }
                }

                ExternalUserSyncWorker.Create(worker.UserID, linkedAccount.ExternalID, AccountType.Twitch);
            }
        }
    }
}
