﻿using System;
using System.Linq;
using Curse.Aerospike;
using Curse.Friends.Data;
using Curse.Friends.Enums;
using Curse.Friends.Tracing;
using Curse.Friends.TwitchApi;
using Curse.Logging;

namespace Curse.Friends.TwitchService.QueueProcessors
{
    class TwitchUserFollowsProcessor
    {
        private static readonly FilteredUserLogger Logger = new FilteredUserLogger("UserFollows");

        public static void Process(TwitchUserFollowsWorker worker)
        {
            var account = ExternalAccount.GetByTwitchUserID(worker.TwitchID);
            if (account == null)
            {
                Logger.Warn(null, null, null, "Unable to sync follows, account not found", worker);
                return;
            }

            SyncRecentFollows(account);

            if (worker.ForceFullSync || account.ShouldSyncFollows())
            {
                SyncFollows(account);
            }
        }

        private static void SyncRecentFollows(ExternalAccount account)
        {
            var followsResponse = TwitchApiHelper.Default.GetUserFollows(account.ExternalID);
            if (followsResponse.Status != TwitchResponseStatus.Success)
            {
                switch (followsResponse.Status)
                {
                    case TwitchResponseStatus.NotFound:
                    case TwitchResponseStatus.Unprocessable:
                    case TwitchResponseStatus.TwitchServerError:
                        Logger.Log(null, account, "[Recent] Unable to sync follows", new { account.ExternalID, account.ExternalUsername, followsResponse });
                        break;
                    default:
                        Logger.Warn(null, account, "[Recent] Unable to sync follows, due to unknown response status: " + followsResponse.Status, new { account.ExternalID, account.ExternalUsername, followsResponse });
                        break;
                }
                return;
            }

            if (followsResponse.Value == null || followsResponse.Value.Follows == null)
            {
                Logger.Warn(null, account, "[Recent] Unable to sync follows due to invalid response info", new {Account = account.GetLogData(), followsResponse});
                return;
            }

            if (followsResponse.Value.Follows.Length == 0)
            {
                Logger.Log(null, account, "[Recent] Skipping follow sync, no follows");
                return;
            }

            var mostRecentFollow = followsResponse.Value.Follows.Max(f => f.CreatedAt);
            if (mostRecentFollow <= account.DateRecentFollow)
            {
                Logger.Log(null, account, "[Recent] Skipping follow sync, no more recent follows were found", account.GetLogData());
                return;
            }

            var follows = followsResponse.Value.Follows.GroupBy(f => f.Channel.Name).Select(g => g.First()).ToArray();

            // Load current follows
            var existingMemberships = ExternalCommunityMembership.GetAllLocal(m => m.UserTypeAndRoleIndex, account.ExternalID + account.Type + GroupRoleTag.SyncedFollower).ToArray();
            var membershipsDictionary = existingMemberships.ToDictionary(m => m.ExternalCommunityID);
            var communities = ExternalCommunity.MultiGetLocal(membershipsDictionary.Keys.Select(k => new KeyInfo(k, AccountType.Twitch))).ToDictionary(c => c.ExternalID);

            foreach (var follow in follows)
            {
                try
                {
                    ExternalCommunity community;
                    if (!communities.TryGetValue(follow.Channel.ID, out community))
                    {
                        community = TwitchModelHelper.GetCommunityFromFollow(follow);
                    }

                    if (community == null)
                    {
                        continue;
                    }

                    if (community.ShouldSyncInfo())
                    {
                        TwitchModelHelper.UpdateCommunity(community, follow.Channel);
                    }
                    community.AddMemberRole(account.ExternalID, GroupRoleTag.SyncedFollower, account.ExternalUsername, account.ExternalDisplayName, follow.CreatedAt);
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "[Recent] Failed to sync a channel follower", follow);
                }
            }

            foreach (var userID in account.MappedUsers)
            {
                ExternalUserSyncWorker.Create(userID, account.ExternalID, AccountType.Twitch, GroupRoleTag.SyncedFollower);
            }

            account.DateRecentFollow = mostRecentFollow;
            account.Update(a => a.DateRecentFollow);
        }

        private static void SyncFollows(ExternalAccount account)
        {
            var followsResponse = TwitchApiHelper.Default.GetAllUserFollows(account.ExternalID);
            if (followsResponse.Status != TwitchResponseStatus.Success)
            {
                switch (followsResponse.Status)
                {
                    case TwitchResponseStatus.NotFound:
                    case TwitchResponseStatus.Unprocessable:
                    case TwitchResponseStatus.TwitchServerError:
                        Logger.Log(null, account, "[Full] Unable to sync follows", new { account.ExternalID, account.ExternalUsername, followsResponse });
                        break;
                    default:
                        Logger.Warn(null, account, "[Full] Unable to sync follows, due to unknown response status: " + followsResponse.Status, new { account.ExternalID, account.ExternalUsername, followsResponse });
                        break;
                }
                return;
            }

            var follows = followsResponse.Value.GroupBy(f => f.Channel.Name).Select(g => g.First()).ToArray();
            var followsDictionary = follows.ToDictionary(f => f.Channel.ID);

            // Load current follows
            var existingMemberships = ExternalCommunityMembership.GetAllLocal(m => m.UserTypeAndRoleIndex, account.ExternalID + account.Type + GroupRoleTag.SyncedFollower).ToArray();
            var membershipsDictionary = existingMemberships.ToDictionary(m => m.ExternalCommunityID);
            var communities = ExternalCommunity.MultiGetLocal(membershipsDictionary.Keys.Select(k => new KeyInfo(k, AccountType.Twitch))).ToDictionary(c => c.ExternalID);

            foreach (var follow in follows)
            {
                try
                {
                    ExternalCommunity community;
                    if (!communities.TryGetValue(follow.Channel.ID, out community))
                    {
                        community = TwitchModelHelper.GetCommunityFromFollow(follow);
                    }

                    if (community == null)
                    {
                        continue;
                    }

                    community.AddMemberRole(account.ExternalID, GroupRoleTag.SyncedFollower, account.ExternalUsername, account.ExternalDisplayName, follow.CreatedAt);
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "[Full] Failed to sync a channel follower", follow);
                }
            }

            foreach (var membership in existingMemberships)
            {
                try
                {
                    Follow follow;
                    if (!followsDictionary.TryGetValue(membership.ExternalCommunityID, out follow))
                    {
                        ExternalCommunity community;
                        if (!communities.TryGetValue(membership.ExternalCommunityID, out community))
                        {
                            continue;
                        }
                        community.RemoveMemberRole(membership.ExternalUserID, GroupRoleTag.SyncedFollower);
                    }
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "[Full] Failed to sync a channel membership.", membership);
                }
            }

            foreach (var userID in account.MappedUsers)
            {
                ExternalUserSyncWorker.Create(userID, account.ExternalID, AccountType.Twitch, GroupRoleTag.SyncedFollower);
            }

            if (follows.Length > 0)
            {
                account.DateRecentFollow = follows.Max(f => f.CreatedAt);
            }
            account.DateFullFollowSync = DateTime.UtcNow;
            account.Update(a => a.DateFullFollowSync, a => a.DateRecentFollow);

        }

    }
}
