﻿
using System;
using System.Collections.Generic;
using System.Linq;
using System.Linq.Expressions;
using System.Text.RegularExpressions;
using Curse.Friends.Data;
using Curse.Friends.Enums;
using Curse.Friends.TwitchApi;
using Curse.Logging;

namespace Curse.Friends.TwitchService
{
    public class TwitchEmoteDescriptor
    {
        public long ID { get; set; }

        public string Regex { get; set; }

        public int EmoteSet { get; set; }

        public string Url { get; set; }

        public int Height { get; set; }

        public int Width { get; set; }

        public bool IsAlphanumeric { get; set; }

        private static readonly Regex _alphaNumericRegex = new Regex(@"^[\p{L}\p{N}]+$", RegexOptions.Compiled);

        public static TwitchEmoteDescriptor[] GetDescriptors(AllEmoticonsResponse emoteResponse, EmoticonImageResponse emoteImagesResponse)
        {
            // put all Twitch Emotes into consumable format
            var emotes = new List<TwitchEmoteDescriptor>();
            var imageLookup = emoteImagesResponse.Emoticons.GroupBy(e => Tuple.Create(e.Code, e.EmoticonSet ?? 0)).ToDictionary(g => g.Key, g => g.First());
            foreach (var emoticon in emoteResponse.Emoticons)
            {
                if (emoticon.Images == null)
                {
                    continue;
                }

                foreach (var image in emoticon.Images)
                {
                    if (image == null)
                    {
                        continue;
                    }

                    EmoticonImageDetails imageDetails;
                    if (imageLookup.TryGetValue(Tuple.Create(emoticon.Regex, image.EmoticonSet ?? 0), out imageDetails))
                    {
                        emotes.Add(new TwitchEmoteDescriptor
                        {
                            ID = imageDetails.ID,
                            Regex = imageDetails.Code,
                            EmoteSet = imageDetails.EmoticonSet ?? 0,
                            Url = image.Url,
                            Height = image.Height,
                            Width = image.Width,
                            IsAlphanumeric = _alphaNumericRegex.IsMatch(imageDetails.Code)
                        });
                    }
                }
            }
            return emotes.ToArray();
        }

        public void UpdateModel(TwitchEmote emote, Avatar avatar)
        {
            if (emote==null)
            {
                emote = new TwitchEmote
                {
                    EmoteID = ID,
                    Regex = Regex,
                    Url = Url,
                    EmoticonSet = EmoteSet,
                    Height = Height,
                    Width = Width,
                    IsAlphanumeric = IsAlphanumeric,
                };
                emote.InsertLocal();
            }
            else
            {
                var changes = new List<Expression<Func<TwitchEmote, object>>>();
                if (emote.IsDeleted)
                {
                    emote.IsDeleted = false;
                    changes.Add(e => e.IsDeleted);
                }
                if (emote.EmoticonSet != EmoteSet)
                {
                    emote.EmoticonSet = EmoteSet;
                    changes.Add(e => e.EmoticonSet);
                }
                if (emote.Width != Width)
                {
                    emote.Width = Width;
                    changes.Add(e => e.Width);
                }
                if (emote.Height != Height)
                {
                    emote.Height = Height;
                    changes.Add(e => e.Height);
                }
                if (emote.Regex != Regex)
                {
                    emote.Regex = Regex;
                    changes.Add(e => e.Regex);
                }
                if (emote.Url != Url)
                {
                    emote.Url = Url;
                    changes.Add(e => e.Url);
                }

                if (emote.IsAlphanumeric != IsAlphanumeric)
                {
                    emote.IsAlphanumeric = IsAlphanumeric;
                    changes.Add(e => e.IsAlphanumeric);
                }

                if (changes.Count > 0)
                {
                    emote.Update(changes.ToArray());
                }
            }

            var entityID = ID.ToString();
            var url = Url;
            var existing = avatar ?? Avatar.GetByTypeAndID(AvatarType.TwitchEmote, entityID);
            if (existing == null || existing.Url != url)
            {
                AvatarUpdatePump.UpdateAvatar(AvatarType.TwitchEmote, entityID, url);
            }
        }
    }
}
