﻿using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using Aerospike.Client;
using Curse.Aerospike;
using Curse.Friends.Data;
using Curse.Friends.Enums;
using Curse.Friends.ServerHosting;
using Curse.Friends.TwitchApi;
using Curse.Friends.TwitchService.Configuration;
using Curse.Friends.TwitchService.QueueProcessors;
using Curse.Logging;
using System.Threading.Tasks;
using System.Threading;
using Curse.CloudServices.Jobs;
using Curse.Extensions;
using Curse.Friends.Configuration;
using Curse.Friends.Data.Queues;
using Curse.Friends.NotificationContracts;
using Curse.Friends.TwitchService.Chat.Firehose;
using Curse.Friends.TwitchService.Chat.Irc;

namespace Curse.Friends.TwitchService
{
    public class TwitchServer : ServerHost<TwitchHost, ExternalCommunity>
    {
        private static readonly TwitchServer _instance = new TwitchServer(TwitchServiceConfiguration.Instance.BucketCount);

        public static void StartServer()
        {
            _instance.Start();
        }

        public static void StopServer()
        {
            _instance.Stop();
        }

        private readonly int BucketCount;
        private readonly ConcurrentDictionary<string, TwitchStreamSession> _sessions = new ConcurrentDictionary<string, TwitchStreamSession>();

        private readonly SimpleMovingAverage _checkGiveawaysFollowersDuration = new SimpleMovingAverage(25);
        private readonly SimpleMovingAverage _checkGiveawaysFollowersPerStream = new SimpleMovingAverage(25);
        private readonly SimpleMovingAverage _generalBucketedTasksDuration = new SimpleMovingAverage(5);
        private readonly SimpleMovingAverage _generalBucketedTasksPerStream = new SimpleMovingAverage(5);
        private readonly SimpleMovingAverage _checkStreamingDuration;
        private readonly SimpleMovingAverage _checkStreamingPerStream;

        private TwitchServer(int bucketCount)
        {
            BucketCount = bucketCount <= 0 ? 30 : bucketCount;

            _checkStreamingDuration = new SimpleMovingAverage(BucketCount * 2);
            _checkStreamingPerStream = new SimpleMovingAverage(BucketCount * 2);
        }

        protected override void CustomStartup()
        {
            _isStarted = true;

            // Not Session Specific
            TwitchAccountSyncWorker.StartProcessor(TwitchAccountSyncProcessor.Process);
            TwitchCommunitySubscriptionsWorker.StartProcessor(TwitchCommunitySubscriptionsProcessor.Process);
            TwitchServerRoleWorker.StartProcessor(TwitchServerRoleProcessor.Process);
            TwitchUserFollowsWorker.StartProcessor(TwitchUserFollowsProcessor.Process);

            // Session Specific
            ExternalCommunityCoordinator.StartProcessor(ProcessExternalCommunityCoordinator);
            GroupGiveawayCommunityCoordinator.StartProcessor(ProcessGroupGiveawayCommunityNotifier);
            GroupPollChangedCommunityCoordinator.StartProcessor(ProcessGroupPollChangedCommunityNotifier);
            RoleSyncCoordinator.StartProcessor(ProcessRoleSyncCheck);
            ExternalMessageResolver.StartProcessor(ProcessExternalMessageResolver);

            AddTimer("Claim Orphaned Stream", TimeSpan.FromSeconds(31), ClaimOrphanedStreams);
            AddTimer("ReportState", TimeSpan.FromMinutes(1), ReportState);
            AddTimer("RebalanceStreams", TimeSpan.FromSeconds(61), RebalanceStreams);
            AddTimer("General Bucketed Tasks", TimeSpan.FromSeconds(121), RunBucketedTasks);

            var periodicBucketSeconds = 59.0 / BucketCount;
            var periodicBucketTimespan = TimeSpan.FromSeconds(periodicBucketSeconds);
            AddTimer("Check Giveaways/Followers", periodicBucketTimespan, CheckGiveawaysFollowers, false);

            AddTimer("Cleanup Idle User Sessions", TimeSpan.FromSeconds(5), CleanIdleUsers);

            var streamingStatusTimespan = TimeSpan.FromSeconds(30.0 / BucketCount);
            AddTimer("Check Streaming Status", streamingStatusTimespan, CheckStreaming, false);

            AddTimer("Report Tracking Stats", TimeSpan.FromMinutes(1), ReportTrackingStats);

            new Thread(RunPeriodicTasks)
            {
                IsBackground = true
            }.Start();

            AvatarUpdatePump.Start();
            JobScheduler.Initialize(StorageConfiguration.CurrentRegion.IsDefault);

            FirehoseManager.Instance.MessageReceived += DispatchMessage;
            FirehoseManager.Instance.Open();
        }

        private bool _isStarted;

        private void RunPeriodicTasks()
        {
            while (_isStarted)
            {
                var sessions = _sessions.Values.ToArray();
                Parallel.ForEach(sessions, new ParallelOptions { MaxDegreeOfParallelism = TwitchServiceConfiguration.Instance.TaskParallelism }, session =>
                {
                    if (!_isStarted)
                    {
                        return;
                    }

                    try
                    {
                        session.RunPeriodicTasks();
                    }
                    catch (Exception ex)
                    {

                        Logger.Error(ex, "Failed to run periodic tasks!");
                    }
                });

                Thread.Sleep(1000);
            }
        }

        private void ReportTrackingStats()
        {
            Logger.Info("Reporting Tracking Stats", new
            {
                TcpClientStats = TrackingTcpClient.GetTrackingReport()
            });
        }

        private TwitchStreamSession[] GetBucketedStreams(ref int bucketIncrementer, out int bucket)
        {
            var bucketIncrement = Interlocked.Increment(ref bucketIncrementer) % BucketCount;
            var currentBucket = (bucketIncrement < 0 ? bucketIncrement + BucketCount : bucketIncrement) + 1;
            bucket = currentBucket;
            return _sessions.Values.Where(v => v.Bucket == currentBucket).ToArray();
        }

        #region Giveaways / Followers

        private static int _giveawayFollowerIncrementer = -1;

        private void CheckGiveawaysFollowers()
        {
            int bucket;
            var bucketedStreams = GetBucketedStreams(ref _giveawayFollowerIncrementer, out bucket);
            if (bucketedStreams.Length == 0)
            {
                return;
            }

            Logger.Trace(string.Format("[Bucket {0}]: Checking giveaways and followers for {1} streams", bucket, bucketedStreams.Length));
            var sw = new Stopwatch();
            sw.Start();
            Parallel.ForEach(bucketedStreams, new ParallelOptions { MaxDegreeOfParallelism = TwitchServiceConfiguration.Instance.TaskParallelism }, session =>
              {
                  if (!_isStarted)
                  {
                      return;
                  }

                  try
                  {
                      session.CheckGiveawaysFollowers();
                  }
                  catch (Exception ex)
                  {
                      Logger.Warn(ex, "Failed to check giveaways/followers!", new { session.Community });
                  }
              });
            sw.Stop();
            _checkGiveawaysFollowersDuration.AddSample(sw.ElapsedMilliseconds);
            _checkGiveawaysFollowersPerStream.AddSample((double)sw.ElapsedMilliseconds / bucketedStreams.Length);
            Logger.Trace(string.Format("[Bucket {0}]: Finished checking giveaways and followers for {1} streams", bucket, bucketedStreams.Length));
        }

        #endregion

        #region CheckStreaming

        private static int _checkStreamingIncrementer = -1;

        private void CheckStreaming()
        {
            int bucket;
            var bucketedStreams = GetBucketedStreams(ref _checkStreamingIncrementer, out bucket);
            if (bucketedStreams.Length == 0)
            {
                return;
            }

            Logger.Trace(string.Format("[Bucket {0}]: Checking stream status for {1} streams", bucket, bucketedStreams.Length));

            var sw = new Stopwatch();
            sw.Start();

            var response = TwitchApiHelper.Default.GetStreams(bucketedStreams.Select(s => s.Community.ExternalID).ToArray());
            if (response.Status != TwitchResponseStatus.Success || response.Value.Streams == null)
            {
                if (response.Status != TwitchResponseStatus.Unprocessable && response.Status != TwitchResponseStatus.NotFound && response.Status != TwitchResponseStatus.TwitchServerError)
                {
                    Logger.Warn("Failed to get a list of streams", response);
                }
                else
                {
                    Logger.Debug("Failed to get a list of streams", response);
                }
                return;
            }

            var streamDictionary = response.Value.Streams.GroupBy(s => s.Channel?.ID).ToDictionary(s => s.Key, s => s.First());
            Parallel.ForEach(bucketedStreams, new ParallelOptions { MaxDegreeOfParallelism = TwitchServiceConfiguration.Instance.TaskParallelism }, stream =>
               {
                   try
                   {
                       stream.UpdateStream(streamDictionary.GetValueOrDefault(stream.Community.ExternalID));
                   }
                   catch (Exception ex)
                   {
                       Logger.Warn(ex, "Failed to update stream status!", new { stream });
                   }
               });

            sw.Stop();
            _checkStreamingDuration.AddSample(sw.ElapsedMilliseconds);
            _checkStreamingPerStream.AddSample((double)sw.ElapsedMilliseconds / bucketedStreams.Length);

            Logger.Trace(string.Format("[Bucket {0}]: Finished updating {1} streams", bucket, bucketedStreams.Length));
        }

        #endregion

        #region General Bucketed Tasks

        private static int _bucketIncrementer = -1;

        private void RunBucketedTasks()
        {
            if (!_isStarted)
            {
                Logger.Info("Skipping bucketed tasks. Server is shutting down.");
                return;
            }

            var bucketIncrement = Interlocked.Increment(ref _bucketIncrementer) % BucketCount;
            var currentBucket = (bucketIncrement < 0 ? bucketIncrement + BucketCount : bucketIncrement) + 1;
            var bucketedStreams = _sessions.Values.Where(v => v.Bucket == currentBucket).ToArray();

            if (bucketedStreams.Length == 0)
            {
                Logger.Trace($"[Bucket {currentBucket}]: No streams in bucket");
                return;
            }

            Logger.Trace($"[Bucket {currentBucket}]: Running bucket tasks for {bucketedStreams.Length} streams");
            var sw = new Stopwatch();
            sw.Start();
            Parallel.ForEach(bucketedStreams, new ParallelOptions { MaxDegreeOfParallelism = TwitchServiceConfiguration.Instance.TaskParallelism }, session =>
            {
                if (!_isStarted)
                {
                    Logger.Info("Skipping bucketed tasks. Server is shutting down.");
                    return;
                }

                try
                {
                    session.RunBucketedTasks();
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to run bucketed tasks!", new { session.Community });
                }
            });
            sw.Stop();

            _generalBucketedTasksDuration.AddSample(sw.ElapsedMilliseconds);
            _generalBucketedTasksPerStream.AddSample((double)sw.ElapsedMilliseconds / bucketedStreams.Length);
            Logger.Trace($"[Bucket {currentBucket}]: Finished bucket tasks for {bucketedStreams.Length} streams");
        }

        #endregion

        private void CleanIdleUsers()
        {
            foreach (var userSession in _userSessions.Values)
            {
                if (userSession.ShouldShutdown)
                {
                    userSession.Shutdown();
                    TwitchUserSession throwaway;
                    _userSessions.TryRemove(userSession.ExternalID, out throwaway);
                }
            }
        }

        protected override void AfterHostRegistered()
        {
            ClaimMyOrphans();            
        }

        protected override void CustomStop()
        {
            _isStarted = false;

            try
            {
                Logger.Info("Closing Firehose connection");
                FirehoseManager.Instance.MessageReceived -= DispatchMessage;
                FirehoseManager.Instance.Close();
                Logger.Info("Successfully closed firehose");
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Unhandled exception while closing firehose");
            }

            try
            {
                AvatarUpdatePump.Stop();
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Unhandled exception while stopping avatar update pump");
            }

            try
            {
                var sessions = _userSessions.Values.ToArray();
                Logger.Info("Shutting down " + sessions.Length + " user sessions...");
                foreach (var session in sessions)
                {
                    session.Shutdown();
                }
                Logger.Info("All user sessions have been shut down!");
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Unhandled exception while shutting down user sessions");
            }

            try
            {
                var sessions = _sessions.Values.ToArray();
                Logger.Info("Shutting down " + sessions.Length + " sessions...");
                foreach (var session in sessions)
                {
                    session.Shutdown();
                }

                Logger.Info("All sessions have been shut down!");
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Unhandled exception while shutting down sessions.");

            }

        }


        private void ClaimMyOrphans()
        {
            if (_host == null || _host.IsCleaning)
            {
                // Someone is cleaning up my streams, don't try to reclaim them yet
                return;
            }

            // Rehost all of the streams I was hosting before that never got orphaned properly
            var orphans = ExternalCommunity.GetAllLocal(s => s.MachineName, Environment.MachineName);
            foreach (var orphan in orphans.Where(s => s.MappedGroups != null && s.MappedGroups.Any()))
            {
                bool retrievedFromCache;
                GetOrAddSession(orphan.ExternalID, out retrievedFromCache);
            }
        }

        private void ReportState()
        {
            Logger.Info("Currently hosting " + _sessions.Count + " streams.",
                new
                {
                    CheckGiveawaysTotal = _checkGiveawaysFollowersDuration.ToSerializeable(),
                    CheckGiveawaysPerStream = _checkGiveawaysFollowersPerStream.ToSerializeable(),

                    GeneralBucketedTasksTotal = _generalBucketedTasksDuration.ToSerializeable(),
                    GeneralBucketedTasksPerStream = _generalBucketedTasksPerStream.ToSerializeable(),

                    CheckStreamingTotal = _checkStreamingDuration.ToSerializeable(),
                    CheckStreamingPerStream = _checkStreamingPerStream.ToSerializeable(),
                });
        }

        public ExternalCommunity[] TryGetAllHostableCommunities(int maxAttempts, int attempt = 1)
        {
            try
            {
                return ExternalCommunity.GetAllLocal(s => s.HostableRegion, ExternalCommunity.LocalConfigID);
            }
            catch (Exception ex)
            {
                if (attempt == maxAttempts)
                {
                    Logger.Error(ex, "Failed to retrieve external communities after " + attempt + " attempts");
                    return null;
                }

                return TryGetAllHostableCommunities(maxAttempts, ++attempt);
            }
        }

        private void ClaimOrphanedStreams()
        {
            if (_host == null)
            {
                return;
            }

            var allHosts = TwitchHost.GetAllLocal(p => p.IndexMode, IndexMode.Default);
            var hosts = allHosts.Where(p => p.Status == ServiceHostStatus.Online).ToArray();
            var hostIndex = Array.FindIndex(hosts, p => p.MachineName == _host.MachineName);
            var unhealthyHostNames = new HashSet<string>(allHosts.Select(h => h.MachineName).Except(hosts.Select(h => h.MachineName)));

            // Retrieve all communities that are hosted locally, to find orphans
            var allLocalCommunities = TryGetAllHostableCommunities(3);

            if (allLocalCommunities == null)
            {
                Logger.Warn("Unable to claim orphaned streams. External communities could not be retrieved from the database.");
                return;
            }



            var allOrphanedCommunities = allLocalCommunities.Where(m => (string.IsNullOrWhiteSpace(m.MachineName) || unhealthyHostNames.Contains(m.MachineName)) &&
                                                                        m.MappedGroups != null && m.MappedGroups.Any()).ToArray();

            if (!allOrphanedCommunities.Any())
            {
                return;
            }


            var take = (int)Math.Ceiling(allOrphanedCommunities.Length / (double)hosts.Length);
            var skip = hostIndex * take;
            var orphans = allOrphanedCommunities.Skip(skip).Take(take).ToArray();

            Logger.Info("Found " + allOrphanedCommunities.Length + " orphaned streams. Attempting to rehost...", new { hostIndex, skip, take });
            var hostedStreams = new List<string>();

            foreach (var orphan in orphans)
            {
                if (orphan == null)
                {
                    return;
                }

                orphan.Refresh();

                if (!string.IsNullOrEmpty(orphan.MachineName) && !unhealthyHostNames.Contains(orphan.MachineName))
                {
                    Logger.Debug("Skipping stream. It's been rehosted by: " + orphan.MachineName);   
                    continue;
                }

                try
                {
                    orphan.MachineName = Environment.MachineName;
                    orphan.Update(UpdateMode.Concurrent, m => m.MachineName);
                }
                catch (AerospikeException ex)
                {
                    if (ex.Message.Contains("Generation error"))  // This means that another host has already claimed it
                    {
                        return;
                    }

                    Logger.Error(ex, "Error claiming an orphaned stream.");
                    continue;
                }


                if (_sessions.ContainsKey(orphan.ExternalID))
                {
                    Logger.Debug("Skipping stream '" + orphan.ExternalID + "'. It is already hosted here.");
                    continue;
                }

                try
                {
                    // Force create a session
                    bool retrievedFromCache;
                    GetOrAddSession(orphan.ExternalID, out retrievedFromCache);
                    hostedStreams.Add(string.Format("{0} - {1}", orphan.ExternalName, orphan.ExternalID));
                }
                catch (Exception ex)
                {

                    Logger.Error(ex, "Failed to create Twitch session for community", orphan);
                }

            }

            Logger.Info("Finished hosting orpaned streams! We are now hosting: " + _sessions.Count + " sessions", new
            {
                HostedOrphans = hostedStreams
            });

        }

        private void RebalanceStreams()
        {
            if(_host == null)
            {
                return; 
            }
            // Retrieve all communities that are hosted locally, to find orphans
            var allLocalCommunities = TryGetAllHostableCommunities(3);

            if (allLocalCommunities == null)
            {
                Logger.Warn("Unable to rebalance streams. External communities could not be retrieved from the database.");
                return;
            }

            var streams = allLocalCommunities.Where(c => c.Type == AccountType.Twitch && !string.IsNullOrEmpty(c.MachineName)).ToArray();

            var streamsMachineGroups = streams.GroupBy(c => c.MachineName);
            var streamsByID = streams.ToDictionary(s => s.ExternalID);

            var allHostnames = new HashSet<string>(TwitchHost.GetAllLocal(g => g.IndexMode, IndexMode.Default).Where(h => h.Status == ServiceHostStatus.Online).Select(h => h.MachineName));
            if (allHostnames.Count == 0)
            {
                Logger.Warn("DB Thinks there are no Twitch Hosts");
                return;
            }

            var streamsByMachine = streamsMachineGroups.Where(s => allHostnames.Contains(s.Key)).ToDictionary(g => g.Key, g => g.ToArray());
            foreach (var host in allHostnames.Where(h => !streamsByMachine.ContainsKey(h)))
            {
                streamsByMachine.Add(host, new ExternalCommunity[0]);
            }

            ExternalCommunity[] myCommunities;
            if (!streamsByMachine.TryGetValue(_host.MachineName, out myCommunities))
            {
                // DB thinks nothing is hosted, remove all hosted streams
                foreach (var session in _sessions.ToArray())
                {
                    TwitchStreamSession throwaway;
                    session.Value.Shutdown();
                    _sessions.TryRemove(session.Key, out throwaway);
                }
                return;
            }

            // Clear sessions that should not be hosted by this machine
            var keys = new HashSet<string>(myCommunities.Select(c => c.ExternalID));
            var removedSessions = _sessions.Where(s => !keys.Contains(s.Key)).ToArray();

            if (removedSessions.Any())
            {
                Logger.Info("Removing " + removedSessions.Length + " sessions that should be hosted by other machines");
            }

            foreach (var session in removedSessions)
            {
                string newHost = null;
                ExternalCommunity stream;
                if (streamsByID.TryGetValue(session.Key, out stream))
                {
                    newHost = stream.MachineName;
                }

                session.Value.Shutdown(newHost);
                TwitchStreamSession throwaway;
                _sessions.TryRemove(session.Key, out throwaway);
            }



            // Rehost sessions that should be hosted but are missing
            var addingSessions = myCommunities.Where(c => !_sessions.ContainsKey(c.ExternalID)).ToArray();

            if (addingSessions.Any())
            {
                Logger.Info("Adding " + addingSessions.Length + " sessions that should hosted by this machine");
            }

            foreach (var community in addingSessions)
            {
                bool retrievedFromCache;
                GetOrAddSession(community.ExternalID, out retrievedFromCache);
            }

            // Figure out if this machine is hosting the most streams
            var highestCount = streamsByMachine.Max(s => s.Value.Length);
            if (myCommunities.Length != highestCount)
            {
                return;
            }

            // Pass out streams, favoring hosts with the least streams hosted
            var total = streams.Length;
            var target = total / streamsByMachine.Count;
            var totalStreamsToGive = myCommunities.Length - target;
            var hostsToBalance = streamsByMachine.Where(s => s.Key != _host.MachineName && myCommunities.Length - s.Value.Length > myCommunities.Length * 0.2).ToArray();

            var streamsLeft = totalStreamsToGive;
            foreach (var hostKvp in hostsToBalance.OrderBy(h => h.Value.Length))
            {
                if (streamsLeft == 0)
                {
                    break;
                }

                var streamsToGive = new[] { totalStreamsToGive, target - hostKvp.Value.Length, streamsLeft }.Min();
                if (streamsToGive == 0)
                {
                    continue;
                }

                Logger.Info(string.Format("Giving {0} streams to {1}", streamsToGive, hostKvp.Key));
                var sessionsToGive = _sessions.Take(streamsToGive).ToArray();
                foreach (var session in sessionsToGive)
                {
                    session.Value.Shutdown(hostKvp.Key);
                    TwitchStreamSession throwaway;
                    _sessions.TryRemove(session.Key, out throwaway);
                }

                ExternalCommunityCoordinator.StreamsTransferred(ExternalCommunity.LocalConfigID, hostKvp.Key, sessionsToGive.Select(s => s.Key).ToArray());
                streamsLeft -= streamsToGive;
            }
        }

        private readonly ConcurrentDictionary<string, object> _createSessionLocks = new ConcurrentDictionary<string, object>();

        private TwitchStreamSession GetOrAddSession(string streamID, out bool retrievedFromCache)
        {
            TwitchStreamSession session;
            if (_sessions.TryGetValue(streamID, out session))
            {
                retrievedFromCache = true;
                return session;
            }

            var createSession = _createSessionLocks.GetOrAdd(streamID, new object());
            var lockAcquired = false;
            Monitor.TryEnter(createSession, TimeSpan.FromSeconds(20), ref lockAcquired);

            try
            {
                if (!lockAcquired)
                {
                    throw new InvalidOperationException("Failed to acquire create session lock for session: " + streamID);
                }

                lock (createSession)
                {
                    if (_sessions.TryGetValue(streamID, out session))
                    {
                        retrievedFromCache = true;
                        return session;
                    }


                    var bucket = GetBestBucket();

                    retrievedFromCache = false;
                    return _sessions.GetOrAdd(streamID, id => new TwitchStreamSession(id, bucket));
                }
            }
            finally
            {
                if (lockAcquired)
                {
                    Monitor.Exit(createSession);
                }
            }

        }

        private int GetBestBucket()
        {
            var weights = new Dictionary<int, double>();
            var groupings = _sessions.Values.GroupBy(s => s.Bucket).ToDictionary(s => s.Key, s => s.Select(g => g.Community.Subscribers).ToArray());
            for (var i = 1; i <= BucketCount; ++i)
            {
                int[] subCounts;
                if (!groupings.TryGetValue(i, out subCounts))
                {
                    subCounts = new int[0];
                }

                var subs = subCounts.Sum();
                var weight = subCounts.Length + (subs > 1 ? Math.Log(subCounts.Sum(), 100) : 0);
                weights.Add(i, weight);
            }

            return weights.OrderBy(w => w.Value).First().Key;
        }

        public void ProcessExternalCommunityCoordinator(ExternalCommunityCoordinator coordinator)
        {
            try
            {
                bool retrievedFromCache;
                TwitchStreamSession session;

                switch (coordinator.ChangeType)
                {
                    case ExternalCommunityChangeType.Commissioned:
                        {
                            GetOrAddSession(coordinator.ExternalID, out retrievedFromCache);
                            break;
                        }

                    case ExternalCommunityChangeType.Decommissioned:
                        {
                            if (_sessions.TryGetValue(coordinator.ExternalID, out session))
                            {
                                session.Shutdown();
                                _sessions.TryRemove(coordinator.ExternalID, out session);
                            }
                            break;
                        }

                    case ExternalCommunityChangeType.Transferred:
                        {
                            foreach (var id in coordinator.TransferredExternalIDs)
                            {
                                GetOrAddSession(id, out retrievedFromCache);
                            }
                            break;
                        }

                    case ExternalCommunityChangeType.OwnerReauthenticated:
                        {
                            session = GetOrAddSession(coordinator.ExternalID, out retrievedFromCache);
                            session.UpdateOwner();
                            break;
                        }

                    case ExternalCommunityChangeType.LinksChanged:
                        {
                            session = GetOrAddSession(coordinator.ExternalID, out retrievedFromCache);
                            session.UpdateLinks(coordinator.MappedGroups);
                            break;
                        }

                    default:
                        Logger.Error("Unknown type", coordinator);
                        break;
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Unhandled exception while processing ExternalCommunityCoordinator.", coordinator);

            }
        }

        public void ProcessGroupGiveawayCommunityNotifier(GroupGiveawayCommunityCoordinator coordinator)
        {
            try
            {
                bool retrievedFromCache;
                var session = GetOrAddSession(coordinator.ExternalCommunityID, out retrievedFromCache);

                session.UpdateGiveaway(coordinator);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Unhandled exception while processing GroupGiveawayCommunityNotifier.", coordinator);

            }
        }

        private void ProcessRoleSyncCheck(RoleSyncCoordinator coordinator)
        {
            try
            {
                bool retrievedFromCache;
                var session = GetOrAddSession(coordinator.ExternalCommunityID, out retrievedFromCache);
                bool eligible = false;
                var links = ExternalAccountMapping.GetAllLocal(m => m.UserID, coordinator.UserID).Where(l => !l.IsDeleted && l.Type == AccountType.Twitch);
                foreach (var link in links)
                {
                    switch (coordinator.RoleType)
                    {
                        case GroupRoleTag.SyncedFollower:
                            eligible |= session.IsFollower(link.ExternalID);
                            break;
                        case GroupRoleTag.SyncedSubscriber:
                            eligible |= session.IsSubscriber(link.ExternalID);
                            break;
                        case GroupRoleTag.SyncedModerator:
                            eligible |= session.IsModerator(link.ExternalID);
                            break;
                    }

                    if (eligible)
                    {
                        break;
                    }
                }
                RoleSyncCoordinator.CreateResponse(coordinator, eligible);
            }
            catch (Exception ex)
            {
                Logger.Warn(ex, "Unhandled exception while checking a user's role.", coordinator);
                RoleSyncCoordinator.CreateResponse(coordinator, false);
            }

        }

        private void ProcessGroupPollChangedCommunityNotifier(GroupPollChangedCommunityCoordinator coordinator)
        {
            try
            {
                bool retrievedFromCache;
                var session = GetOrAddSession(coordinator.ExternalID, out retrievedFromCache);
                session.UpdatePoll(coordinator.Notification);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Error processing Group Poll Changed Community Notifier", coordinator);
            }
        }

        private void ProcessExternalMessageResolver(ExternalMessageResolver resolver)
        {
            var response = new ConversationMessageResponse
            {
                ClientID = resolver.ClientMessageID,
                ConversationID = resolver.ConversationID,
                Status = DeliveryStatus.Error
            };
            try
            {
                if (resolver.RequestorAccount.NeedsReauthentication || string.IsNullOrEmpty(resolver.RequestorAccount.AuthToken))
                {
                    response.Status = DeliveryStatus.Forbidden;
                    response.ForbiddenReason = MessageForbiddenReason.LinkedAccountNeedsReauth;
                }
                else
                {
                    bool retrievedFromCache;
                    var session = GetOrAddUserSession(resolver.RequestorAccount, out retrievedFromCache);

                    long? retryAfter;
                    var status = session.TrySendMessage(resolver.Community.ExternalName, resolver.MessageBody, out retryAfter);
                    switch (status)
                    {
                        case MessageFailureReason.Success:
                            response.Status = DeliveryStatus.Successful;
                            break;
                        case MessageFailureReason.Throttled:
                            response.Status = DeliveryStatus.Throttled;
                            response.RetryAfter = retryAfter;
                            break;
                        default:
                            // TODO: LOG
                            response.Status = DeliveryStatus.Error;
                            break;
                    }
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Error processing External Message Resolver", resolver);
            }
            finally
            {
                if (resolver.RequestorEndpoint != null)
                {
                    ChatMessageResponseNotifier.Create(resolver.RequestorEndpoint, response);
                }
            }
        }

        private readonly ConcurrentDictionary<string, TwitchUserSession> _userSessions = new ConcurrentDictionary<string, TwitchUserSession>();

        private TwitchUserSession GetOrAddUserSession(ExternalAccount account, out bool retrievedFromCache)
        {
            var retrieved = true;
            var session = _userSessions.GetOrAdd(account.ExternalID, i =>
            {
                retrieved = false;
                return new TwitchUserSession(account);
            });

            retrievedFromCache = retrieved;
            return session;
        }

        private void DispatchMessage(object sender, FirehoseMessageEventArgs args)
        {
            TwitchStreamSession session;
            if (!_sessions.TryGetValue(args.ExternalID, out session))
            {
                return;
            }

            session.DispatchMessage(args.Message);
        }
    }
}
