﻿using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Linq;
using System.Linq.Expressions;
using System.Threading;
using Curse.Aerospike;
using Curse.Extensions;
using Curse.Friends.Data;
using Curse.Friends.Data.Queues;
using Curse.Friends.Data.Search;
using Curse.Friends.Enums;
using Curse.Friends.NotificationContracts;
using Curse.Friends.Tracing;
using Curse.Friends.TwitchApi;
using Curse.Friends.TwitchService.Chat;
using Curse.Friends.TwitchService.Chat.Firehose;
using Curse.Friends.TwitchService.Chat.Parsing;
using Curse.Logging;

namespace Curse.Friends.TwitchService
{
    public class TwitchStreamSession
    {
        private const int _liveStatusFallbackMilliseconds = 600000;

        private static readonly LogCategory ThrottledLogger = new LogCategory("TwitchStreamSession") { Throttle = TimeSpan.FromSeconds(60) };
        private static readonly FilteredUserLogger FilteredLogger = new FilteredUserLogger("TwitchStreamSession");
        private readonly LogCategory Logger;        
        private readonly ExternalCommunity _stream;
        private readonly CancellationTokenSource _cancellationTokenSource;
        private readonly ExternalAccount _ownerAccount;
        private Group[] _mappedGroups;

        public TwitchStreamSession(string streamID, int bucket)
        {
            Bucket = bucket;
            _cancellationTokenSource = new CancellationTokenSource();

            // Create Bot
            _stream = ExternalCommunity.GetLocal(streamID, AccountType.Twitch);
            if (_stream == null)
            {
                throw new InvalidOperationException("Unable to find Twitch stream for " + streamID);
            }

            _ownerAccount = ExternalAccount.GetLocal(streamID, AccountType.Twitch);
            if (_ownerAccount == null)
            {
                throw new InvalidOperationException("Unable to find stream owner for " + streamID);
            }

            UpdateLinks(_stream.MappedGroups);

            FirehoseManager.Instance.RegisterStream(_stream.ExternalName, _stream.ExternalID);

            Logger = new LogCategory($"Stream {_stream.ExternalDisplayName}/{_stream.ExternalID}");
            Logger.Debug("Session Created!");
        }

        public int Bucket { get; private set; }

        public ExternalCommunity Community { get { return _stream; } }


        public void Shutdown(string newMachineName = null)
        {
            try
            {
                FirehoseManager.Instance.UnregisterStream(_stream.ExternalName);
                _cancellationTokenSource.Cancel();
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to shutdown stream.");
            }
          
            try
            {                

                _stream.MachineName = newMachineName ?? string.Empty;
                _stream.Update(s => s.MachineName);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to unhost stream.");
            }           
        }

        #region Periodic Tasks

        private static readonly LogCategory TaskLogger = new LogCategory("Tasks") { Throttle = TimeSpan.FromMinutes(1) };

        #region Bucket-based

        public void RunBucketedTasks()
        {
            try
            {
                TaskLogger.Debug("Syncing account...");
                AccountSync();
            }
            catch (Exception ex)
            {
                TaskLogger.Warn(ex, "Failed to sync account.");
            }

            try
            {
                TaskLogger.Debug("Syncing subs...");
                SubscriberSync();
            }
            catch (Exception ex)
            {
                TaskLogger.Warn(ex, "Failed to sync subs.");
            }

            try
            {
                TaskLogger.Debug("Syncing mods...");
                ModSync();
            }
            catch (Exception ex)
            {
                Logger.Warn(ex, "Failed to sync mods.");
            }
            
            try
            {
                TaskLogger.Debug("Syncing badges");
                BadgeSync();
            }
            catch (Exception ex)
            {
                TaskLogger.Warn(ex, "Failed to sync badges");
            }
        }
        
        private void AccountSync(bool updateOwner = true)
        {
            // Refresh the owner accunt from the database
            if (updateOwner)
            {                
                UpdateOwner();
            }
            
            // If there is no auth token, we can do nothing
            if (string.IsNullOrEmpty(_ownerAccount.AuthToken))
            {
                FilteredLogger.Log(_ownerAccount.ExternalID, "Skipping account sync: Auth token is null or empty.");
                return;
            }

            // If the owner account does not need reauth and it was last synced in the last 15 mins
            if (!_ownerAccount.NeedsReauthentication && DateTime.UtcNow.Subtract(_ownerAccount.LastSyncTime) < TimeSpan.FromMinutes(15))
            {
                FilteredLogger.Log(_ownerAccount.ExternalID, "Skipping account sync: It is up to date.");
                return;
            }
             
            FilteredLogger.Log(_ownerAccount.ExternalID, "Resyncing account.", new { _ownerAccount.ExternalUsername });
            TwitchAccountSyncWorker.Create(_ownerAccount.ExternalID, true);            
        }

        #region Subscribers

        
        private void SubscriberSync()
        {
            if (!_stream.CanHaveSubs)
            {
                Logger.Trace("Skipping subscriber sync since the community is not partnered");
                return;
            }

            TwitchCommunitySubscriptionsWorker.Create(_stream);
        }

        #endregion

        #region Emoticons

        private void EmoticonSync()
        {
            if (!_stream.CanHaveSubs)
            {
                Logger.Trace("Skipping emoticon sync since the community is not partnered");
                return;
            }

            Logger.Debug("Syncing emoticons");

            var emoticonsResponse = TwitchApiHelper.Default.GetEmoticons(_stream.ExternalID);
            if (emoticonsResponse.Status != TwitchResponseStatus.Success)
            {
                Logger.Warn("Failed to sync emoticons", emoticonsResponse);
                return;
            }

            var twitchEmoticons = emoticonsResponse.Value
                .Emoticons.Where(e => e.SubscriberOnly)
                .ToLookup(e => e.RegexString)
                .ToDictionary(e => e.Key, e => e.First());

            var existingEmotes = ExternalCommunityEmoticon.GetAllByTypeAndSyncID(_stream.Type, _stream.ExternalID).ToDictionary(e => e.Regex);

            foreach (var twitchEmoticon in twitchEmoticons.Values)
            {
                var urlChanged = false;

                var requiredRoles = new HashSet<int> { (int)GroupRoleTag.SyncedSubscriber };

                ExternalCommunityEmoticon emoticon;
                if (!existingEmotes.TryGetValue(twitchEmoticon.RegexString, out emoticon))
                {
                    // New sync
                    if (twitchEmoticon.State == "active")
                    {
                        // create only if it is actually active on Twitch
                        emoticon = new ExternalCommunityEmoticon
                        {
                            CommunityType = _stream.Type,
                            SyncID = _stream.ExternalID,
                            Regex = twitchEmoticon.RegexString,
                            RequiredRoles = requiredRoles,
                            Url = twitchEmoticon.Url
                        };
                        emoticon.InsertLocal();
                        urlChanged = true;
                    }

                }
                else if (twitchEmoticon.State == "active" && emoticon.IsDeleted)
                {
                    // Reactivated emoticon
                    emoticon.IsDeleted = false;
                    emoticon.RequiredRoles = requiredRoles;

                    urlChanged = emoticon.Url != twitchEmoticon.Url;
                    emoticon.Url = twitchEmoticon.Url;

                    emoticon.Update(e => e.IsDeleted, e => e.RequiredRoles, e => e.Url);
                }
                else if (twitchEmoticon.State != "active" && !emoticon.IsDeleted)
                {
                    // Deactivated emoticon
                    emoticon.IsDeleted = true;
                    emoticon.Update(e => e.IsDeleted);
                }
                else if (!requiredRoles.SetEquals(emoticon.RequiredRoles) || emoticon.Url != twitchEmoticon.Url)
                {
                    emoticon.RequiredRoles = requiredRoles;
                    urlChanged = emoticon.Url != twitchEmoticon.Url;
                    emoticon.Url = twitchEmoticon.Url;
                    emoticon.Update(e => e.RequiredRoles, e => e.Url);
                }

                if (urlChanged)
                {
                    var url = emoticon.Url;
                    var entityID = emoticon.GetAvatarEntityID();
                    AvatarUpdatePump.UpdateAvatar(AvatarType.SyncedEmoticon, entityID, url);
                }
            }

            foreach (var emoticon in existingEmotes.Where(kvp => !kvp.Value.IsDeleted && !twitchEmoticons.ContainsKey(kvp.Key)).Select(kvp => kvp.Value))
            {
                emoticon.IsDeleted = true;
                emoticon.Update(e => e.IsDeleted);
                var entityID = emoticon.GetAvatarEntityID();
                AvatarUpdatePump.UpdateAvatar(AvatarType.SyncedEmoticon, entityID, string.Empty);
            }
        }

        #endregion

        #region Badges

        private void BadgeSync()
        {
            if (!_stream.CanHaveSubs)
            {
                Logger.Trace("Skipping badge sync since the community is not partnered");
                return;
            }

            var resp = TwitchApiHelper.Default.GetChannelChatBadges(_stream.ExternalID);
            if (resp.Status != TwitchResponseStatus.Success)
            {
                Logger.Debug("Failed to sync badges for stream", new {_stream, resp});
                return;
            }

            var existingBadges = TwitchChatBadge.GetAllLocal(b => b.ExternalID, _stream.ExternalID).ToDictionary(b => Tuple.Create(b.BadgeSet, b.Version));
            var descriptors = TwitchBadgeDescriptor.GetDescriptors(resp.Value).GroupBy(b => Tuple.Create(b.BadgeSet, b.Version)).ToDictionary(g => g.Key, g => g.First());

            foreach (var descriptor in descriptors)
            {
                descriptor.Value.UpdateModel(existingBadges.GetValueOrDefault(descriptor.Key), _stream.ExternalID);
            }

            foreach (var existingBadge in existingBadges)
            {
                if (!descriptors.ContainsKey(existingBadge.Key))
                {
                    existingBadge.Value.IsDeleted = true;
                    existingBadge.Value.Update(e => e.IsDeleted);
                    var entityID = TwitchChatBadge.GetAvatarEntityID(existingBadge.Key.Item1, existingBadge.Key.Item2, existingBadge.Value.ExternalID);
                    AvatarUpdatePump.UpdateAvatar(AvatarType.TwitchChatBadge, entityID, string.Empty);
                }
            }
        }

        #endregion

        #region Mods

        public DateTime DateLastModSync { get; private set; }

        private void ModSync()
        {
            TwitchCommunityModsWorker.Create(_stream);
        }

        #endregion

        #endregion

        #region Time-based

        private DateTime _lastSubCountSync;

        public void RunPeriodicTasks()
        {
            if (DateTime.UtcNow.Subtract(_lastSubCountSync) >= TimeSpan.FromSeconds(31))
            {
                try
                {
                    TaskLogger.Debug("Syncing sub count");
                    SyncSubCount();
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to sync sub count.");
                }
            }
        }

        #region Sub Count

        private void SyncSubCount()
        {
            _lastSubCountSync = DateTime.UtcNow;

            if (!_stream.CanHaveSubs)
            {
                Logger.Trace("Skipping sub count sync, not a partner!", _stream);
                return;
            }

            Logger.Trace("Syncing sub count...");

            foreach (var mapping in ExternalCommunityMapping.MultiGetLocal(_stream.MappedGroups.Select(id => new KeyInfo(id, _stream.ExternalID, _stream.Type))))
            {
                var group = Group.GetByID(mapping.GroupID);
                if (group == null)
                {
                    Logger.Debug("Group no longer exists for community mapping", new { mapping });
                    continue;
                }

                var roles = group.GetRoles();
                var b = GroupMemberManager.CountMembersByRoles(mapping.GroupID);
                var role = roles.FirstOrDefault(r => r.IsSynced && r.SyncID == _stream.ExternalID && r.Source == _stream.Type && r.Tag == GroupRoleTag.SyncedSubscriber);
                if (role == null)
                {
                    Logger.Debug("Role not found in a mapped group!");
                    continue;
                }

                long count;
                if (!b.TryGetValue(role.RoleID, out count))
                {
                    count = 0L;
                }

                mapping.SyncedPremiumMembers = (int)count;
                mapping.Update(m => m.SyncedPremiumMembers);
            }

        }

        #endregion

        #endregion

        #region Giveaways / Followers

        private bool _giveawaysInitiallyLoaded;
        private DateTime _mostRecentFollowDate;

        public void CheckGiveawaysFollowers()
        {
            try
            {
                if (!_giveawaysInitiallyLoaded)
                {
                    _giveawaysInitiallyLoaded = true;
                    var groups = Group.MultiGetLocal(_stream.MappedGroups.Select(id => new KeyInfo(id)));
                    var giveaways = new List<GroupGiveaway>();
                    foreach (var regionalGroups in groups.GroupBy(g => g.RegionID))
                    {
                        giveaways.AddRange(GroupGiveaway.MultiGet(regionalGroups.Key, regionalGroups.Select(g => new KeyInfo(g.GroupID, g.LatestGiveawayNumber))));
                    }

                    foreach (var giveaway in giveaways.Where(g => g.Status != GroupGiveawayStatus.Ended && g.Status != GroupGiveawayStatus.Inactive))
                    {
                        AddActiveGiveaway(giveaway);
                    }
                }

                // Clean up giveaways that should no longer be tracked
                var activeGiveaways = _activeGiveaways.Values.ToArray();
                foreach (var activeGiveaway in activeGiveaways)
                {
                    if (!_stream.MappedGroups.Contains(activeGiveaway.GroupID))
                    {
                        // If this community is no longer linked
                        GroupGiveaway throwaway;
                        _activeGiveaways.TryRemove(activeGiveaway.GroupID, out throwaway);
                        continue;
                    }

                    activeGiveaway.Refresh();
                    if (activeGiveaway.Status == GroupGiveawayStatus.Ended || activeGiveaway.Status == GroupGiveawayStatus.Inactive)
                    {
                        // If this giveaway is no longer active, 
                        GroupGiveaway throwaway;
                        _activeGiveaways.TryRemove(activeGiveaway.GroupID, out throwaway);
                    }
                }

                activeGiveaways = _activeGiveaways.Values.ToArray();
                if (activeGiveaways.Length == 0)
                {
                    return;
                }

                // Only look back as far as the start of the earliest active giveaway
                var earliestGiveawayStart = activeGiveaways.Min(g => g.DateStarted);
                if (_mostRecentFollowDate < earliestGiveawayStart)
                {
                    _mostRecentFollowDate = earliestGiveawayStart;
                }

                // Only look back at most 4 hours
                var minDate = DateTime.UtcNow.AddHours(-4);
                if (_mostRecentFollowDate < minDate)
                {
                    _mostRecentFollowDate = minDate;
                }

                var recentFollowers = TwitchApiHelper.Default.GetChannelFollowersSince(_stream.ExternalID, _mostRecentFollowDate);
                if (recentFollowers.Status != TwitchResponseStatus.Success)
                {
                    if (recentFollowers.Status != TwitchResponseStatus.Unprocessable &&
                        recentFollowers.Status != TwitchResponseStatus.NotFound &&
                        recentFollowers.Status != TwitchResponseStatus.TwitchServerError)
                    {
                        Logger.Warn("Failed to get recent followers", new { Stream = _stream, Response = recentFollowers });
                    }
                    return;
                }

                if (recentFollowers.Value.Length == 0)
                {
                    // No recent followers, skip
                    return;
                }

                var mostRecentFollowDate = recentFollowers.Value.Max(f => f.CreatedDate);
                if (_mostRecentFollowDate < mostRecentFollowDate)
                {
                    _mostRecentFollowDate = mostRecentFollowDate;
                }

                var followers = recentFollowers.Value.GroupBy(f => f.User.ID).ToDictionary(f => f.Key, f => f.First());
                var existingMemberships = ExternalCommunityMembership.MultiGetLocal(followers.Keys.Select(id => new KeyInfo(id, _stream.ExternalID, _stream.Type, GroupRoleTag.SyncedFollower)))
                    .ToDictionary(f => f.ExternalUserID);

                var missingFollowers = new List<Follower>();
                foreach (var follower in followers)
                {
                    ExternalCommunityMembership existingMembership;
                    if (!existingMemberships.TryGetValue(follower.Key, out existingMembership) || existingMembership.Status != ExternalCommunityMembershipStatus.Active)
                    {
                        missingFollowers.Add(follower.Value);
                    }
                }
                var trackedAccounts = ExternalAccount.MultiGetLocal(missingFollowers.Select(f => new KeyInfo(f.User.ID, AccountType.Twitch))).ToDictionary(a => a.ExternalID);
                var followersToAdd = missingFollowers.Where(f => trackedAccounts.ContainsKey(f.User.ID)).ToArray();

                var updated = followersToAdd.Aggregate(false, (u, f) => u | _stream.AddMemberRole(f.User.ID, GroupRoleTag.SyncedFollower, f.User.Name, f.User.DisplayName, f.CreatedDate.ToUniversalTime()));
                if (updated)
                {
                    Logger.Info("Found recent followers not yet tracked!", new { followersToAdd });
                    ExternalCommunityMemberSyncWorker.Create(_stream.RegionID, _stream.ExternalID, _stream.Type, GroupRoleTag.SyncedFollower);
                }
            }
            catch (Exception ex)
            {
                Logger.Warn(ex, "Failed to check giveaways");
            }
        }

        #endregion

        #region Check Streaming

        public void UpdateStream(Stream stream)
        {
            var oldName = _stream.ExternalName;

            // Refresh from DB
            UpdateOwner();
            _stream.Refresh();
            UpdateLinks(_stream.MappedGroups);

            // Fix stream status if stuck
            var now = DateTime.UtcNow.ToEpochMilliseconds();
            var isStreaming = stream != null;
            if (_stream.IsLive != isStreaming && _stream.LiveTimestamp < now - _liveStatusFallbackMilliseconds)
            {
                ThrottledLogger.Info("Stream status was not updated via Stream Up/Down, fixing its status", new { Stream = _stream.GetLogData() });
                _stream.IsLive = isStreaming;
                _stream.LiveTimestamp = now;
                _stream.Update(s => s.IsLive, s => s.LiveTimestamp);
            }

            if (oldName != _stream.ExternalName)
            {
                Logger.Debug("Stream name changed", new {oldName, newName = _stream.ExternalName, _stream});
                FirehoseManager.Instance.RegisterStream(_stream.ExternalName, _stream.ExternalID);
                FirehoseManager.Instance.UnregisterStream(oldName);
            }

            if (stream != null)
            {
                var previousStatus = _stream.ExternalStatus;
                var previousGame = _stream.ExternalGameName;

                var newStatus = stream.Channel.Status ?? string.Empty;
                var newGame = stream.Channel.Game ?? string.Empty;

                var updates = new List<Expression<Func<ExternalCommunity, object>>>();
                if (_stream.ExternalStatus != newStatus)
                {
                    _stream.ExternalStatus = newStatus;
                    updates.Add(s => s.ExternalStatus);
                }
                if (_stream.ExternalGameName != newGame)
                {
                    _stream.ExternalGameName = newGame;
                    updates.Add(s => s.ExternalGameName);
                }

                if (updates.Count > 0)
                {
                    _stream.Update(updates.ToArray());
                    ExternalCommunityLinkChangedCoordinator.Create(_stream, _mappedGroups, ExternalCommunityLinkChangeType.InfoChanged);
                }
            }

            foreach(var group in _mappedGroups)
            {
                group.UpdateIsStreaming(_stream);
            }
        }

        #endregion

        #endregion

        #region Giveaways

        private readonly ConcurrentDictionary<Guid, GroupGiveaway> _activeGiveaways = new ConcurrentDictionary<Guid, GroupGiveaway>();

        public void UpdateGiveaway(GroupGiveawayCommunityCoordinator coordinator)
        {
            switch (coordinator.Notification.ChangeType)
            {
                case GroupGiveawayChangeType.Started:
                {
                    var group = Group.GetLocal(coordinator.Notification.Giveaway.GroupID);
                    if (group != null)
                    {
                        var giveaway = GroupGiveaway.Get(group.RegionID, group.GroupID, coordinator.Notification.Giveaway.GiveawayID);
                        if (giveaway != null)
                        {
                            AddActiveGiveaway(giveaway);
                        }
                    }
                    break;
                }
                case GroupGiveawayChangeType.Ended:
                case GroupGiveawayChangeType.Removed:
                {
                    GroupGiveaway giveaway;
                    if (_activeGiveaways.TryRemove(coordinator.Notification.Giveaway.GroupID, out giveaway))
                    {
                        Logger.Debug("Removed an ended giveaway, periodic bucketed tasks should stop running", new {_stream, giveaway});
                        if (giveaway.GiveawayID > coordinator.Notification.Giveaway.GiveawayID)
                        {
                            Logger.Debug("Received an ended/removed event for an older giveaway, adding current giveaway back and continuing periodic bucketed tasks",
                                new {_stream, giveaway, coordinator.Notification.Giveaway});
                            _activeGiveaways.TryAdd(giveaway.GroupID, giveaway);
                        }
                    }
                    break;
                }
                case GroupGiveawayChangeType.ParticipantAdded:
                case GroupGiveawayChangeType.ParticipantRemoved:
                case GroupGiveawayChangeType.WinnerSelected:
                case GroupGiveawayChangeType.InvalidWinnerSelected:
                case GroupGiveawayChangeType.PrizeClaimed:
                case GroupGiveawayChangeType.ClaimExpired:
                case GroupGiveawayChangeType.Rolling:
                case GroupGiveawayChangeType.EntriesUpdated:
                case GroupGiveawayChangeType.FakeRoll:
                case GroupGiveawayChangeType.InvalidRoll:
                    break;
                default:
                    Logger.Warn("Unknown GiveawayChangeType cannot be processed.", coordinator);
                    break;
            }
        }

        private void AddActiveGiveaway(GroupGiveaway giveaway)
        {
            if (giveaway.Status == GroupGiveawayStatus.Ended || giveaway.Status == GroupGiveawayStatus.Inactive)
            {
                return;
            }

            if (giveaway.RequiredRoles == null || giveaway.RequiredRoles.Count == 0)
            {
                return;
            }

            var requiredRoles = GroupRole.MultiGetLocal(giveaway.RequiredRoles.Select(id => new KeyInfo(giveaway.GroupID, id)));
            if (!requiredRoles.Any(r => r.IsSynced && r.SyncID == _stream.ExternalID && r.Source == AccountType.Twitch && r.Tag == GroupRoleTag.SyncedFollower))
            {
                return;
            }

            if (_activeGiveaways.TryAdd(giveaway.GroupID, giveaway))
            {
                Logger.Debug("Added an active giveaway, periodic bucketed tasks should begin running", new { _stream, giveaway });
            }
        }

        #endregion

        #region Polls

        public void UpdatePoll(GroupPollChangedNotification notification)
        {
            switch (notification.ChangeType)
            {
                case GroupPollChangeType.Started:
                    break;
                case GroupPollChangeType.Ended:
                    break;
                case GroupPollChangeType.VotesUpdated:
                    break;
                case GroupPollChangeType.Deactivated:
                    break;
                default:
                    Logger.Warn("Unknown Poll change type", notification);
                    break;
            }
        }

        #endregion

        #region Firehose Events

        public void DispatchMessage(TwitchMessage message)
        {
            switch (message.MessageType)
            {
                case IrcMessageType.PrivMsg:
                    ProcessMessage(message);
                    break;
                case IrcMessageType.ClearChat:
                case IrcMessageType.Roomstate:
                case IrcMessageType.UserNotice:
                    NotifyNotice(message);
                    break;
            }
        }

        private void ProcessMessage(TwitchMessage message)
        {
            if (message.Username != "twitchnotify")
            {
                foreach (var mappedGroup in _mappedGroups)
                {
                    var senderMappedIDs = TwitchCaching.GetMappedUserIDsByTwitchID(message.UserID);
                    ExternalMessageCoordinator.Create(mappedGroup, _stream.ExternalID, senderMappedIDs,
                        message.UserID.ToString(), message.Username, message.UserDisplayName, message.UserColor, message.Badges,
                        message.MessageID, message.Data, message.Timestamp.FromEpochMilliconds(), message.EmoteSubstitutions, message.Bits);
                }
            }
        }

        private static readonly System.Text.RegularExpressions.Regex _newSubscriberRegex =
            new System.Text.RegularExpressions.Regex(@"(?<user>.*?) .*?subscribed(?: for ?)?(?<streak>\d+)?.*?$", System.Text.RegularExpressions.RegexOptions.Compiled);

        private void ProcessSubUserNotice(TwitchMessage message)
        {
            var userExternalID = message.UserID.ToString();
            var channelExternalID = message.ChannelID.ToString();
            var username = TwitchMessageParser.GetTagStringValue((string) message.Tags.GetValueOrDefault(TwitchChatTag.Login));
            var userDisplayName = message.UserDisplayName;
            if (channelExternalID != _stream.ExternalID)
            {
                return;
            }

            Logger.Debug("User '" + message.Username + "' subscribed to channel '" + _stream.ExternalName + "'");

            try
            {
                var notification = new TwitchChatNoticeNotification
                {
                    ExternalChannelID = channelExternalID,
                    NoticeText = message.Data,
                    NoticeType = TwitchChatNoticeType.UserSubscribed,
                    Timestamp = DateTime.UtcNow.ToEpochMilliseconds()
                };
                foreach (var mappedGroup in _mappedGroups)
                {
                    TwitchChatNoticeCoordinator.Create(mappedGroup, notification);
                }

                var subPlan = (string)message.Tags.GetValueOrDefault(TwitchChatTag.SubPlan) ?? "1000";
                var subRoles = new List<GroupRoleTag> { GroupRoleTag.SyncedSubscriber };
                if (subPlan == "3000")
                {
                    subRoles.Add(GroupRoleTag.SyncedSubscriberTier2);
                    subRoles.Add(GroupRoleTag.SyncedSubscriberTier3);
                }
                else if (subPlan == "2000")
                {
                    subRoles.Add(GroupRoleTag.SyncedSubscriberTier2);
                }

                var memberships = ExternalCommunityMembership.MultiGetLocal(subRoles.Select(role => new KeyInfo(userExternalID, channelExternalID, AccountType.Twitch, role)));
                var membershipLookup = memberships.ToLookup(m => m.RoleTag);

                if(subRoles.All(r=>membershipLookup[r].FirstOrDefault()?.Status== ExternalCommunityMembershipStatus.Active))
                {
                    Logger.Trace("Skipping sub sync for user '" + message.Username + "'. They already have an active sub roles on the community.");
                }

                var added = false;
                foreach (var role in subRoles)
                {
                    var membership = membershipLookup[role].FirstOrDefault();
                    if (membership != null && membership.Status == ExternalCommunityMembershipStatus.Active)
                    {
                        continue;
                    }

                    Logger.Debug($"Add {subPlan} Sub role for user {message.UserID}");
                    added |= _stream.AddMemberRole(message.UserID.ToString(), role, message.Username, message.UserDisplayName, DateTime.UtcNow);
                }
                if (added)
                {
                    foreach (var link in ExternalAccountMapping.GetAllLocal(l => l.ExternalID, userExternalID).Where(l => !l.IsDeleted && l.Type == AccountType.Twitch))
                    {
                        ExternalUserSyncWorker.Create(link.UserID, link.ExternalID, link.Type);
                    }
                }
            }
            catch (Exception ex)
            {
                Logger.Warn(ex, "Failed to sync a channel subscribed event.");
            }
        }

        private void NotifyNotice(TwitchMessage message)
        {
            if(message.NoticeType == TwitchChatNoticeType.Sub || message.NoticeType==TwitchChatNoticeType.Resub)
            {
                ProcessSubUserNotice(message);
            }

            var notification = new TwitchChatNoticeNotification
            {
                Timestamp = DateTime.UtcNow.ToEpochMilliseconds(),
                NoticeType = message.NoticeType,
                ExternalChannelID = _stream.ExternalID,
                DurationSeconds = ((long?)message.Tags.GetValueOrDefault(TwitchChatTag.BanDuration)) ?? 0L,
                NoticeText = message.Data,
                ExternalUserID = message.UserID.ToString(),
                ExternalUsername = message.Username,
            };

            foreach (var mappedGroup in _mappedGroups)
            {
                TwitchChatNoticeCoordinator.Create(mappedGroup, notification);
            }
        }

        #endregion

        public bool IsFollower(string id)
        {
            return TwitchApiHelper.Default.TestUserFollows(id, _stream.ExternalID);
        }

        public bool IsModerator(string id)
        {
            var membership = ExternalCommunityMembership.GetLocal(id, _stream.ExternalID, _stream.Type, GroupRoleTag.SyncedModerator);
            return membership!=null && membership.Status != ExternalCommunityMembershipStatus.Deleted;
        }

        public bool IsSubscriber(string id)
        {
            if (_ownerAccount.NeedsReauthentication)
            {
                Logger.Info("Can't request subscriber info, account needs reauthentication");
                // Can't get this info, assume false
                return false;
            }
            return TwitchApiHelper.Default.TestUserSubs(id, _stream.ExternalID, _ownerAccount.AuthToken);
        }

        public void UpdateOwner()
        {
            lock (_ownerAccount)
            {
                _ownerAccount.Refresh();
            }
        }

        private void MarkOwnerForReauth(string source)
        {
            lock (_ownerAccount)
            {
                TwitchModelHelper.MarkForReauthorization(_ownerAccount, source);
            }
        }

        public void UpdateLinks(HashSet<Guid> mappedGroups)
        {
            _mappedGroups = Group.MultiGetLocal(mappedGroups.Select(id => new KeyInfo(id))).Select(g=>g.EnsureWritable()).ToArray();
        }
    }
}
