﻿using System;
using System.Linq;
using Curse.Aerospike;
using Curse.Extensions;
using Curse.Friends.Data;
using Curse.Friends.Enums;
using Curse.Friends.NotificationContracts;
using Curse.Friends.TwitchService.Chat;
using Curse.Friends.TwitchService.Chat.Irc;
using Curse.Friends.TwitchService.Chat.Parsing;
using Curse.Logging;

namespace Curse.Friends.TwitchService
{
    public class TwitchUserSession
    {
        private readonly LogCategory Logger;
        private readonly TwitchIrcConnection _client;
        private readonly ExternalAccount _account;
        private bool _isShutdown;

        public TwitchUserSession(ExternalAccount account)
        {
            _account = account;
            Logger = new LogCategory(account.ExternalUsername + " User Session");

            _client = new TwitchIrcConnection(account.ExternalUsername, account.AuthToken);
            _client.MessageReceived += MessageReceived;

            TwitchAccountResolver.HostChanged(account);

            _client.Connect();

            Logger.Debug("Session Created!");
        }

        public string ExternalID { get { return _account.ExternalID; } }

        public bool ShouldShutdown { get { return _isShutdown || _client.IsExpired || _account.NeedsReauthentication; } }

        public MessageFailureReason TrySendMessage(string channelName, string messageBody, out long? retryAfter)
        {
            return _client.TrySendMessage(channelName, messageBody, out retryAfter);
        }

        public void Shutdown(string newMachineName = null)
        {
            if (_isShutdown)
            {
                return;
            }

            _isShutdown = true;
            _client.Close();
            _account.MachineName = newMachineName ?? string.Empty;
            _account.Update(a => a.MachineName);
            TwitchAccountResolver.HostChanged(_account);
        }

        private static readonly System.Text.RegularExpressions.Regex _slowModeRegex = new System.Text.RegularExpressions.Regex(@"talk again in (?<timeout>\d+) seconds", System.Text.RegularExpressions.RegexOptions.Compiled);

        private void MessageReceived(object sender, TwitchMessageEventArgs args)
        {
            if (args.Message.MessageType != IrcMessageType.Notice)
            {
                return;
            }

            if (args.Message.NoticeType == TwitchChatNoticeType.Unknown || IrcMessageParser.IsRoomBasedNotice(args.Message.NoticeType))
            {
                return;
            }

            if (args.Message.NoticeType == TwitchChatNoticeType.AuthenticationFailed)
            {
                TwitchModelHelper.MarkForReauthorization(_account, "ChatLogin");
                Shutdown();
                return;
            }

            var duration = 0L;
            switch (args.Message.NoticeType)
            {
                case TwitchChatNoticeType.MessageSlowMode:
                    // parse
                    var match = _slowModeRegex.Match(args.Message.Data);
                    if (match.Success)
                    {
                        duration = long.Parse(match.Groups["timeout"].Value);
                    }
                    break;
            }

            var notification = new TwitchChatNoticeNotification
            {
                NoticeType = args.Message.NoticeType,
                NoticeText = args.Message.Data,
                Timestamp = DateTime.UtcNow.ToEpochMilliseconds(),
                DurationSeconds = duration,
                ExternalUserID = _account.ExternalID
            };

            DeliverChatNotice(args.Message.ChannelName, notification);
        }

        private void DeliverChatNotice(string channelName, TwitchChatNoticeNotification notification)
        {
            var community = TwitchCaching.GetExternalCommunityByChannelName(channelName);
            if (community == null)
            {
                Logger.Debug("Couldn't find the community", channelName);
                return;
            }
            
            var groups = Group.MultiGetLocal(community.MappedGroups.Select(id => new KeyInfo(id)));
            if (groups.Count == 0)
            {
                return;
            }

            notification.ExternalChannelID = community.ExternalID;

            var targetedUsers = TwitchCaching.GetMappedUserIDsByTwitchID(long.Parse(_account.ExternalID));

            foreach (var group in groups)
            {
                TwitchChatNoticeCoordinator.Create(group, notification, targetedUsers);
            }
        }
    }
}
