﻿using System;
using System.Threading;
using Curse.Friends.Configuration;
using Curse.Friends.WebService.Tests.Utilities;
using Curse.Logging;
using Microsoft.VisualStudio.TestTools.UnitTesting;

namespace Curse.Friends.WebService.Tests
{
    [TestClass]
    public abstract class BaseContext
    {
        protected const int DatabaseUpdateTimeoutMillis = 2000;
        protected const int NotificationClientUserID = Int32.MaxValue - 1;

        protected static NotificationClient NotificationClient;

        [AssemblyInitialize]
        public static void InitializeAssembly(TestContext context)
        {
            Logger.Init(new LoggerConfig(@"C:\Curse\CurseFriendsWebService\Logs") { LogRetainCount = Int32.MaxValue, MaxLogSize = Int32.MaxValue });
            Logger.Info("Initializing Clan Testing...");

            string errorMessage = null;
            try
            {
                StorageConfiguration.Initialize("FriendsWebService");
                NotificationClient = NotificationClient.Connect(NotificationClientUserID, Guid.NewGuid().ToString());

                if (NotificationClient == null)
                {
                    errorMessage = "Failed to connection notification server!";
                    throw new Exception(errorMessage);
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to initialize correctly!");
                Logger.Flush();
            }
            finally
            {
                // Clean up, prevent any tests from running and include the error message.
                if (errorMessage!=null)
                {
                    CleanUpAssembly();
                    throw new Exception(errorMessage);
                }
            }
        }

        [AssemblyCleanup]
        public static void CleanUpAssembly()
        {
            NotificationClient.Disconnect();
            StorageConfiguration.Shutdown();
            Logger.Shutdown();
        }

        protected static bool TryWaitForDatabase(Func<bool> databaseTest = null)
        {
            const int retryMillis = DatabaseUpdateTimeoutMillis / 20;

            var start = DateTime.Now;
            var timeout = start.AddMilliseconds(DatabaseUpdateTimeoutMillis);

            // wait the full duration if no condition is given.
            databaseTest = databaseTest ?? (() => false);
            while (!databaseTest())
            {
                Thread.Sleep(retryMillis);

                if (DateTime.Now > timeout)
                {
                    return false;
                }

            }
            return true;
        }
    }
}
