﻿using System;
using System.Linq;
using Curse.Friends.Data;
using Curse.Friends.Data.DerivedModels;
using Curse.Friends.Data.Models;
using Curse.Friends.Enums;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Newtonsoft.Json;

namespace Curse.Friends.WebService.Tests.Clans
{
    public static class ChannelManagementTests
    {
        [TestClass]
        public class AddChannel : ClansContext
        {
            private const string ClanTitle = "AddChannel Test Clan";
            private const string OfficerChannelTitle = "AddChannel Officer Channel";
            private const string AddedSubchannelTitle = "AddChannel Added Channel";

            private static Group _clanRoot;
            private static Group _officerChannel;
            private static Group _addedChannel;

            [ClassInitialize]
            public static void InitializeAddChannelTests(TestContext context)
            {
                try
                {
                    _clanRoot = CreateRootGroup(ClanTitle, GroupType.Large);
                    _officerChannel = AddSubChannel(_clanRoot, OfficerChannelTitle, Creator);
                    _addedChannel = AddSubChannel(_officerChannel, AddedSubchannelTitle, Creator);
                }
                catch (Exception)
                {
                    CleanUp();
                    throw;
                }
            }

            [TestMethod]
            public void Writes_To_Database()
            {
                Assert.IsNotNull(Group.GetLocal(_addedChannel.GroupID));
            }

            [TestMethod]
            public void Has_Given_Title()
            {
                // DateTime is automatically appended, so the title should contain the expected title.
                Assert.IsTrue(_addedChannel.Title.Contains(AddedSubchannelTitle));
            }

            [TestMethod]
            public void Has_Parent()
            {
                Assert.AreEqual(_officerChannel.GroupID, _addedChannel.ParentGroupID);
            }

            [TestMethod]
            public void Has_Root()
            {
                Assert.AreEqual(_clanRoot.GroupID, _addedChannel.RootGroupID);
            }

            [TestMethod]
            public void Is_Large_Group()
            {
                Assert.AreEqual(GroupType.Large, _addedChannel.Type);
            }

            [TestMethod]
            public void Logs_Subgroup_Created_Event()
            {
                Assert.IsTrue(GroupEventLog.GetLocal(_clanRoot.GroupID).Events.GetValues()
                    .Any(e => e.Type == GroupEventType.SubgroupCreated &&
                              JsonConvert.DeserializeAnonymousType(e.EventData,
                                  new {GroupTitle = ""}).GroupTitle.Contains(_addedChannel.Title)));
            }

            [ClassCleanup]
            public static void CleanUp()
            {
                CleanUpDatabase(_clanRoot);
            }
        }

        [TestClass]
        public class RemoveChannel : ClansContext
        {
            private static Group _clanRoot;
            private static Group _officerChannel;

            [ClassInitialize]
            public static void Initialize(TestContext context)
            {
                try
                {
                    _clanRoot = CreateRootGroup("RemoveChannel Test Clan", GroupType.Large);
                    _officerChannel = AddSubChannel(_clanRoot, "RemoveChannel Officer Channel", Creator);
                    RemoveSubChannel(_officerChannel);
                }
                catch
                {
                    CleanUp();
                    throw;
                }
            }

            [TestMethod]
            public void Channel_Status_Is_Updated()
            {
                var officerChannel = Group.GetLocal(_officerChannel.GroupID);
                Assert.IsNotNull(officerChannel);
                Assert.AreEqual(officerChannel.Status, GroupStatus.Deleted);
            }

            [TestMethod]
            public void Logs_Subgroup_Deleted_Event()
            {
                Assert.IsTrue(GroupEventLog.GetLocal(_clanRoot.GroupID).Events.GetValues()
                    .Any(e => e.Type == GroupEventType.SubgroupRemoved &&
                              JsonConvert.DeserializeAnonymousType(e.EventData,
                                  new {GroupTitle = ""}).GroupTitle.Contains(_officerChannel.Title)));
            }

            [ClassCleanup]
            public static void CleanUp()
            {
                CleanUpDatabase(_clanRoot);
            }
        }

        [TestClass]
        public class RaiseChannelPermissions : ClansContext
        {
            private static Group _clanRoot;
            private static Group _officerChannel;
            private const LegacyGroupRole UpgradedAccessLevel = LegacyGroupRole.Admin;

            [ClassInitialize]
            public static void Initialize(TestContext context)
            {
                try
                {
                    _clanRoot = CreateRootGroup("RaiseChannelPermissions Test Clan", GroupType.Large);
                    _officerChannel = AddSubChannel(_clanRoot, "RaiseChannelPermissions Test Officer Chat", Creator);

                    ChangeAccessLevel(_officerChannel, Creator.UserID, UpgradedAccessLevel);
                }
                catch (Exception)
                {
                    CleanUp();
                    throw;
                }
            }

            [TestMethod]
            public void Writes_To_Database()
            {
                Assert.AreEqual(UpgradedAccessLevel, Group.GetLocal(_officerChannel.GroupID).AccessLevel);
            }

            [TestMethod]
            public void RemovesUsersWithInsufficientAccess()
            {
                Assert.AreEqual(0, GroupMembership.GetAllLocal(p => p.GroupID, _officerChannel.GroupID)
                    .Count(m => m.GroupRole < UpgradedAccessLevel));
            }

            [ClassCleanup]
            public static void CleanUp()
            {
                CleanUpDatabase(_clanRoot);
            }
        }

        [TestClass]
        public class LowerChannelPermissions : ClansContext
        {
            private static Group _clanRoot;
            private static Group _officerChannel;
            private const LegacyGroupRole DowngradedAccessLevel = LegacyGroupRole.Member;

            [ClassInitialize]
            public static void Initialize(TestContext context)
            {
                try
                {
                    _clanRoot = CreateRootGroup("LowerChannelPermissions Test Clan", GroupType.Large);
                    _officerChannel = AddSubChannel(_clanRoot, "LowerChannelPermissions Test Officer Chat", Creator);

                    ChangeAccessLevel(_officerChannel, Creator.UserID, DowngradedAccessLevel);
                }
                catch (Exception)
                {
                    CleanUp();
                    throw;
                }
            }

            [TestMethod]
            public void Writes_To_Database()
            {
                Assert.AreEqual(DowngradedAccessLevel, Group.GetLocal(_officerChannel.GroupID).AccessLevel);
            }

            [TestMethod]
            public void AddsUsersWithSufficientAccess()
            {
                TryWaitForDatabase(() =>
                    GroupMembership.GetAllLocal(p => p.GroupID, _officerChannel.GroupID)
                        .Any(m => m.UserID == Member.UserID));
                Assert.IsNotNull(
                    GroupMembership.GetAllLocal(p => p.GroupID, _officerChannel.GroupID)
                        .FirstOrDefault(m => m.UserID == Member.UserID));
            }

            [ClassCleanup]
            public static void CleanUp()
            {
                CleanUpDatabase(_clanRoot);
            }
        }

        [TestClass]
        public class ReorganizeChannels : ClansContext
        {
            private static Group _clanRoot;
            private static Group _wowChat;
            private static Group _lolChat;
            private static Group _bobsChannel;
            private static Group _marysChannel;
            private static Group _someGuysChannel;
            private static Group _defaultChannel;

            [ClassInitialize]
            public static void Initialize(TestContext context )
            {
                try
                {
                    // Initial Structure
                    //  Clan Root
                    //      WoW Chat
                    //      LoL Chat
                    //          Mary's Channel
                    //          Bob's Channel
                    //          SomeGuy's Channel
                    _clanRoot = CreateRootGroup("RestructureChannels Test Group", GroupType.Large);
                    _wowChat = AddSubChannel(_clanRoot, "RestructureChannels Test Wow Chat");
                    _lolChat = AddSubChannel(_clanRoot, "RestructureChannels Test LoL Chat");
                    _marysChannel = AddSubChannel(_lolChat, "RestructureChannels Test Mary's Channel");
                    _bobsChannel = AddSubChannel(_lolChat, "RestructureChannels Test Bob's Channel");
                    _someGuysChannel = AddSubChannel(_lolChat, "RestructureChannels Test SomeGuy's Channel");

                    //get the default channel
                    _defaultChannel =
                        Group.GetAllLocal(p => p.ParentGroupID, _clanRoot.GroupID).FirstOrDefault(p => p.IsDefaultChannel);
                    Assert.IsNotNull(_defaultChannel);

                    // New Structure
                    //  Clan Root
                    //      WoW Chat
                    //          Bob's Channel (Moved to a different parent)
                    //      LoL Chat
                    //          SomeGuy's Channel (Reordered to be first)
                    //          Mary's Channel
                    _clanRoot.ReorganizeChildren(
                        new GroupTree
                        {
                            GroupID = _clanRoot.GroupID,
                            Children = new[]
                            {
                                new GroupTree
                                {
                                    GroupID = _wowChat.GroupID,
                                    Children = new []
                                    {
                                        new GroupTree
                                        {
                                            GroupID = _bobsChannel.GroupID
                                        }
                                    }
                                }, 
                                new GroupTree
                                {
                                    GroupID = _lolChat.GroupID,
                                    Children = new []
                                    {
                                        new GroupTree
                                        {
                                            GroupID = _someGuysChannel.GroupID
                                        },
                                        new GroupTree
                                        {
                                            GroupID = _marysChannel.GroupID
                                        }
                                    }
                                },
                                new GroupTree
                                {
                                    GroupID = _defaultChannel.GroupID,
                                    Children = new GroupTree[] {}
                                }
                            }
                        }, Creator.UserID);
                }
                catch
                {
                    CleanUp();
                    throw;
                }
            }

            [TestMethod]
            public void Channels_Are_Reparented_As_Needed()
            {
                Assert.AreEqual(_wowChat.GroupID, Group.GetLocal(_bobsChannel.GroupID).ParentGroupID);
            }

            [TestMethod]
            public void Channels_Are_Reordered_AsNeeded()
            {
                Assert.AreEqual(0, Group.GetLocal(_someGuysChannel.GroupID).DisplayOrder);
                Assert.AreEqual(1, Group.GetLocal(_marysChannel.GroupID).DisplayOrder);
            }

            /// <summary>
            /// negative test moving default channel under some other channel
            /// </summary>
            [TestMethod]
            public void TestOrderingPartialtree()
            {
                bool caughtException = false;
                try
                {   
                    //clantoor
                    //  wowchat
                    //      bobschannel
                    //  default channel
                    _clanRoot.ReorganizeChildren(
                        new GroupTree
                        {
                            GroupID = _clanRoot.GroupID,
                            Children = new[]
                            {
                                new GroupTree
                                {
                                    GroupID = _wowChat.GroupID,
                                    Children = new[]
                                    {
                                        new GroupTree
                                        {
                                            GroupID = _bobsChannel.GroupID
                                        }
                                    }
                                },
                                new GroupTree
                                {
                                    GroupID = _defaultChannel.GroupID,
                                    Children = new GroupTree[]{}
                                }
                            }
                        }, Creator.UserID);
                }
                catch (InvalidOperationException ex)
                {
                    caughtException = true;
                }
                Assert.IsTrue(caughtException);
            }

            [TestMethod]
            public void TestinvalidTree()
            {
                bool caughtException = false;
                try
                {
                    //clantoor
                    //  wowchat
                    //      bobschannel
                    //  lolchat
                    //      default channel
                    //      marys channel
                    //      someguys channel
                    _clanRoot.ReorganizeChildren(
                        new GroupTree
                        {
                            GroupID = _clanRoot.GroupID,
                            Children = new[]
                            {
                                new GroupTree
                                {
                                    GroupID = _wowChat.GroupID,
                                    Children = new[]
                                    {
                                        new GroupTree
                                        {
                                            GroupID = _bobsChannel.GroupID
                                        }
                                    }
                                },
                                new GroupTree
                                {
                                    GroupID = _lolChat.GroupID,
                                    Children = new GroupTree[]
                                    {
                                        new GroupTree()
                                        {
                                            GroupID = _defaultChannel.GroupID,
                                            Children = new GroupTree[] {}
                                        },
                                        new GroupTree()
                                        {
                                            GroupID = _marysChannel.GroupID
                                        },
                                        new GroupTree()
                                        {
                                            GroupID = _someGuysChannel.GroupID
                                        }
                                    }
                                }
                            }
                        }, Creator.UserID);
                }
                catch (InvalidOperationException ex)
                {
                    caughtException = true;
                }
                Assert.IsTrue(caughtException);
            }

            [ClassCleanup]
            public static void CleanUp()
            {
                CleanUpDatabase(_clanRoot);
            }
        }

        [TestClass]
        public class ChangeChannelInfo : ClansContext
        {
            private static Group _clanRoot;
            private static Group _subChannel;

            private const string NewTitle = "ChangeChannelInfo Test Channel New Name";
            private const string NewAvatarUrl = "http://nowhere.com";
            private const string NewMotd = "New Message of the Day";
            private const bool NewAllowTempChildren = true;
            private const bool NewForcePTT = true;

            [ClassInitialize]
            public static void Init(TestContext ctx)
            {
                try
                {
                    _clanRoot = CreateRootGroup("ChangeForcePushToTalk Test Root", GroupType.Large);
                    _subChannel = AddSubChannel(_clanRoot, "ChangeChannelInfo Test Channel");

                    ChangeInfo(_subChannel, Admin.UserID, NewTitle, NewAvatarUrl, NewMotd, NewAllowTempChildren,
                        NewForcePTT);
                }
                catch
                {
                    CleanUp();
                    throw;
                }
            }

            [TestMethod]
            public void Changes_Title()
            {
                Assert.AreEqual(NewTitle, Group.GetLocal(_subChannel.GroupID).Title);
            }

            [TestMethod]
            public void Changes_Avatar_Url()
            {
                Assert.AreEqual(NewAvatarUrl, Group.GetLocal(_subChannel.GroupID).AvatarUrl);
            }

            [TestMethod]
            public void Changes_Message_Of_The_Day()
            {
                Assert.AreEqual(NewMotd, Group.GetLocal(_subChannel.GroupID).MessageOfTheDay);
            }

            [TestMethod]
            public void Changes_Allow_Temp_Children()
            {
                Assert.AreEqual(NewAllowTempChildren, Group.GetLocal(_subChannel.GroupID).AllowTempChannels);
            }

            [TestMethod]
            public void Changes_Force_Push_To_Talk()
            {
                Assert.AreEqual(NewForcePTT, Group.GetLocal(_subChannel.GroupID).ForcePushToTalk);
            }

            [TestMethod]
            public void Logs_Title_Changed_Event()
            {
                Assert.IsTrue(GroupEventLog.GetLocal(_clanRoot.GroupID).Events.GetValues()
                    .Any(e => e.Type == GroupEventType.GroupTitleChanged &&
                              JsonConvert.DeserializeAnonymousType(e.EventData,
                                  new {FormerTitle = "", NewTitle = ""})
                                      .NewTitle == NewTitle));
            }

            [TestMethod]
            public void Logs_Avatar_Changed_Event()
            {
                Assert.IsTrue(GroupEventLog.GetLocal(_clanRoot.GroupID).Events.GetValues()
                    .Any(e => e.Type == GroupEventType.GroupAvatarChanged &&
                              JsonConvert.DeserializeAnonymousType(e.EventData,
                                  new {GroupTitle = "", FormerAvatarUrl = "", NewAvatarUrl = ""})
                                      .NewAvatarUrl == NewAvatarUrl));
            }

            [ClassCleanup]
            public static void CleanUp()
            {
                CleanUpDatabase(_clanRoot);
            }
        }

        [TestClass]
        public class AddNormalSubchannel : ClansContext
        {
            private static Group _rootGroup;
            private static InvalidOperationException _expectedException;
            private static Group _subGroup;

            [ClassInitialize]
            public static void Init(TestContext context)
            {
                try
                {
                    _rootGroup = CreateRootGroup("AddNormalSubchannel Test Root", GroupType.Large);

                    try
                    {
                        _subGroup = AddSubChannel(_rootGroup, "AddNormalSubchannel Test Subchannel", Creator,
                            LegacyGroupRole.Member, GroupType.Normal);
                    }
                    catch (InvalidOperationException ex)
                    {
                        _expectedException = ex;
                    }
                }
                catch
                {
                    CleanUp();
                    throw;
                }
            }

            [TestMethod]
            public void Throws_Invalid_Operation_Exception()
            {
                Assert.IsNotNull(_expectedException);
            }

            [TestMethod]
            public void Does_Not_Create_Subchannel()
            {
                Assert.IsNull(_subGroup);
            }

            [ClassCleanup]
            public static void CleanUp()
            {
                CleanUpDatabase(_rootGroup);
            }
        }

        [TestClass]
        public class UpgradeTemproaryChannel : ClansContext
        {
            private static Group _rootGroup;
            private static Group _subGroup;
            private static Group _permGroup;

            [ClassInitialize]
            public static void Initialize(TestContext context)
            {
                try
                {
                    _rootGroup = CreateRootGroup("upgrade channel test", GroupType.Large);
                    ChangeInfo(_rootGroup, Admin.UserID, _rootGroup.Title, _rootGroup.AvatarUrl, _rootGroup.MessageOfTheDay,
                        true, null);
                    _subGroup = AddSubChannel(_rootGroup, "temporary channel to upgrade", Admin, LegacyGroupRole.Member,
                        GroupType.Temporary);
                    _permGroup = AddSubChannel(_rootGroup, "permanent channel to upgrade", Admin, LegacyGroupRole.Member,
                        GroupType.Large);
                    //upgrade temporary channel to permanent one
                    UpgradeTemporaryChannel(_subGroup);
                }
                catch (Exception)
                {
                    CleanUp();
                    throw;
                }
            }

            [ClassCleanup]
            public static void CleanUp()
            {
                CleanUpDatabase(_rootGroup);
            }

            [TestMethod]
            public void TestUpgradedChannelisPermanent()
            {   
                var subgroup = Group.GetLocal(_subGroup.GroupID);
                Assert.IsNotNull(subgroup);
                Assert.AreEqual(subgroup.Type, GroupType.Large);
            }

            [TestMethod]
            public void TestUpgradedChannelhasMemberships()
            {
                var memberships = GroupMembership.GetAllLocal(p => p.GroupID, _subGroup.GroupID);
                Assert.IsTrue(memberships.Any());
            }

            [TestMethod]
            public void TestUpgradedChannelhasNoMemberlist()
            {
                var memberlist = GroupMemberList.GetLocal(_subGroup.GroupID);
                Assert.IsNull(memberlist);
            }

            /// <summary>
            /// negative test
            /// </summary>
            [TestMethod]
            public void TestUpgradePermanentChannel()
            {
                var exceptionRaised = false;
                try
                {
                    UpgradeTemporaryChannel(_permGroup);
                }
                catch (InvalidOperationException ex)
                {
                    exceptionRaised = true;
                }
                Assert.IsTrue(exceptionRaised);
            }
        }
    }
}
