﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading;
using Curse.Friends.Data;
using Curse.Friends.Data.Models;
using Curse.Friends.Enums;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Newtonsoft.Json;
using Newtonsoft.Json.Linq;

namespace Curse.Friends.WebService.Tests.Clans
{
    public static class ClanManagementTests
    {
        [TestClass]
        public class CreateClan : ClansContext
        {
            private const string GroupTitle = "CreateClan Test Clan";
            private const string LobbyTitle = "CreateClan Test Lobby";

            private static Group _clanRoot;

            [ClassInitialize]
            public static void InitializeClanCreationTests(TestContext context)
            {
                try
                {
                    _clanRoot = CreateRootGroup(GroupTitle, GroupType.Large, LobbyTitle);
                }
                catch (Exception)
                {
                    CleanUp();
                    throw;
                }
            }

            [ClassCleanup]
            public static void CleanUp()
            {
                CleanUpDatabase(_clanRoot);
            }

            [TestMethod]
            public void Writes_To_Database()
            {
                Assert.IsNotNull(Group.GetLocal(_clanRoot.GroupID), "New group was not added to the DB.");
            }

            [TestMethod]
            public void Has_Given_Title()
            {
                // Timestamp is appended for testing, so the expected group title should be 
                // contained in the actual title.
                Assert.IsTrue(_clanRoot.Title.Contains(GroupTitle),
                    "The inserted title does not match the requested title");
            }

            [TestMethod]
            public void Has_Initial_Members()
            {
                foreach (var groupMember in DefaultMembers)
                {
                    Assert.IsNotNull(_clanRoot.MemberList.Members.Find(groupMember.UserID),
                        "User {0} was not in the group member list.", groupMember.Username);
                }
            }

            [TestMethod]
            public void Is_Large_Group()
            {
                Assert.AreEqual(GroupType.Large, _clanRoot.Type);
            }

            [TestMethod]
            public void Has_Default_Subchannel()
            {
                var channels = Group.GetAllLocal(p => p.ParentGroupID, _clanRoot.GroupID).Where(p => p.GroupID != _clanRoot.GroupID).ToArray();
                Assert.IsTrue(channels.Count() == 1);
                //Also test if the default channel has the title with "{clan title} Lobby"
                var defaultChannel = channels.FirstOrDefault();
                Assert.AreEqual(LobbyTitle, defaultChannel.Title);
            }

            [TestMethod]
            public void Logs_Creation_Event()
            {
                TryWaitForDatabase(() => GroupEventLog.GetLocal(_clanRoot.GroupID) != null);
                Assert.IsTrue(GroupEventLog.GetLocal(_clanRoot.GroupID)
                    .Events.GetValues()
                    .Any(e => e.Type == GroupEventType.RootGroupCreated));
            }

            [TestMethod]
            public void Logs_Users_Added_Event()
            {
                TryWaitForDatabase(() => GroupEventLog.GetLocal(_clanRoot.GroupID) != null);
                var usersAddedEvent = GroupEventLog.GetLocal(_clanRoot.GroupID)
                    .Events.GetValues()
                    .First(e => e.Type == GroupEventType.UsersAdded);

                var ed = JsonConvert.DeserializeAnonymousType(usersAddedEvent.EventData,
                    new {AddedUsers = new int[0]});
                Assert.IsTrue(
                    DefaultMembers.Where(m => m.UserID != Creator.UserID)
                        .All(m => ed.AddedUsers.Contains(m.UserID)));
            }
        }

        [TestClass]
        public class DeleteClan : ClansContext
        {
            private static Group _clanRoot;
            private static Group _officerChannel;

            [ClassInitialize]
            public static void Initialize(TestContext context)
            {
                try
                {
                    _clanRoot = CreateRootGroup("DeleteClan Test Clan", GroupType.Large);
                    _officerChannel = AddSubChannel(_clanRoot, "DeleteClan Test Officer Channel", Creator);

                    DeleteRootGroup(_clanRoot, Creator.UserID);
                }
                catch (Exception)
                {
                    CleanUp();
                    throw;
                }
            }

            [TestMethod]
            public void Deletes_Clan()
            {
                Assert.IsNull(Group.GetLocal(_clanRoot.GroupID));
                Assert.AreEqual(0, GroupMembership.GetAllLocal(p => p.RootGroupID, _clanRoot.GroupID).Count());
                Assert.AreEqual(0, GroupMemberList.GetAllLocal(p => p.GroupID, _clanRoot.GroupID).Count());
                Assert.IsNull(GroupEventLog.GetLocal(_clanRoot.GroupID));
            }

            [TestMethod]
            public void Deletes_Channels()
            {
                Assert.IsNull(Group.GetLocal(_officerChannel.GroupID));
                Assert.AreEqual(0, GroupMembership.GetAllLocal(p => p.RootGroupID, _officerChannel.GroupID).Count());
            }

            [ClassCleanup]
            public static void CleanUp()
            {
                CleanUpDatabase(_clanRoot);
            }
        }

        [TestClass]
        public class DeleteClanWithoutPermissions : ClansContext
        {
            private static GroupPermissionException _permissionException;
            private static Group _clanRoot;

            [ClassInitialize]
            public static void Initialize(TestContext context)
            {
                try
                {
                    _clanRoot = CreateRootGroup("DeleteClanWithoutPermissions Test Clan", GroupType.Large);

                    try
                    {
                        _clanRoot.DeleteRoot(Officer.UserID);
                        Assert.Fail("Non-owner was able to delete the group");
                    }
                    catch (GroupPermissionException e) // Expected
                    {
                        _permissionException = e;
                    }
                }
                catch (Exception)
                {
                    CleanUp();
                    throw;
                }
            }

            [TestMethod]
            public void Throws_Group_Permission_Exception()
            {
                Assert.IsNotNull(_permissionException);
            }

            [TestMethod]
            public void Does_Not_Delete_Clan()
            {
                Assert.IsNotNull(Group.GetLocal(_clanRoot.GroupID));
            }

            [ClassCleanup]
            public static void CleanUp()
            {
                CleanUpDatabase(_clanRoot);
            }
        }

        [TestClass]
        public class ChangeMessageOfTheDay : ClansContext
        {
            private const string NewMessageOfTheDay = "New Test MOTD";
            private static Group _clanRoot;

            [ClassInitialize]
            public static void Initialize(TestContext context)
            {
                try
                {
                    _clanRoot = CreateRootGroup("ChangeMessageOfTheDay Test Clan", GroupType.Large);

                    ChangeInfo(_clanRoot, Creator.UserID, null, null, NewMessageOfTheDay, null, null);
                }
                catch
                {
                    CleanUp();
                    throw;
                }
            }

            [TestMethod]
            public void Writes_To_Database()
            {
                Assert.AreEqual(NewMessageOfTheDay, Group.GetLocal(_clanRoot.GroupID).MessageOfTheDay);
            }

            [ClassCleanup]
            public static void CleanUp()
            {
                CleanUpDatabase(_clanRoot);
            }
        }

        [TestClass]
        public class ChangePermissionLevels : ClansContext
        {
            private static Group _rootGroup;

            [ClassInitialize]
            public static void Initialize(TestContext context)
            {
                try
                {
                    _rootGroup = CreateRootGroup("ChangePermissionLevels Test Group", GroupType.Large);

                    var permissionsChanges = new Dictionary<GroupPermissions, int>
                    {
                        {GroupPermissions.CreateTemporaryGroup, (int) LegacyGroupRole.Admin},
                        {GroupPermissions.ManageChannels, (int) LegacyGroupRole.Officer}
                    };
                    ChangePermissionLevels(_rootGroup, permissionsChanges, Creator.UserID);
                }
                catch
                {
                    CleanUp();
                    throw;
                }
            }

            [TestMethod]
            public void Changes_Requested_Permissions()
            {
                Assert.AreEqual((int) LegacyGroupRole.Admin,
                    Group.GetLocal(_rootGroup.GroupID).GetAllPermissions()[GroupPermissions.CreateTemporaryGroup]);
                Assert.AreEqual((int) LegacyGroupRole.Officer,
                    Group.GetLocal(_rootGroup.GroupID).GetAllPermissions()[GroupPermissions.ManageChannels]);
            }

            [TestMethod]
            public void Does_Not_Change_Non_Requested_Permissions()
            {
                Assert.AreEqual((int) LegacyGroupRole.Officer,
                    Group.GetLocal(_rootGroup.GroupID).GetAllPermissions()[GroupPermissions.VoiceKickUser]);
            }

            [ClassCleanup]
            public static void CleanUp()
            {
                CleanUpDatabase(_rootGroup);
            }
        }
    }
}
