﻿using System;
using System.Linq;
using System.Runtime.InteropServices;
using System.Threading;
using Curse.Friends.Data;
using Curse.Friends.Data.DerivedModels;
using Curse.Friends.Enums;
using Curse.Logging;
using Microsoft.VisualStudio.TestTools.UnitTesting;

namespace Curse.Friends.WebService.Tests.Clans
{
    [TestClass]
    public class ClanandUserSuiteTestContext : ClansContext
    {
        private Group _rootGroup;
        private Group _childGroup;
        
        [ClassInitialize]
        public static void InitializeSuite(TestContext context)
        {   
        }

        /// <summary>
        /// Initializes all data required to test for this suite
        /// Creates root group, adds default members required to test
        /// </summary>
        [TestInitialize]
        public void TestInit()
        {
            Logger.Info("Test initialization called");
            var title = "Root group for test " + DateTime.UtcNow;
            try
            {
                //create root group to test and add 3 members to group
                _rootGroup = CreateRootGroup("Large root group", GroupType.Large);
                //Create Large sub-group of officer level by admin
                _childGroup = AddSubChannel(_rootGroup, title, Admin);
            }
            catch (Exception ex)
            {
                TestSuiteCleanUp();
                Assert.Fail("Test Initialization failed" + ex.Message);
            }
        }

        /// <summary>
        /// Cleans up any traces left by the test
        /// This method is called after every single test method completes execution (pass/fail)
        /// </summary>
        [TestCleanup]
        public void TestSuiteCleanUp()
        {
            Logger.Info("calling Test Cleanup method");
            //cleans any trace remaining from the test
            //deletes all the groups
            if (_rootGroup != null)
            {
                ClansContext.CleanUpDatabase(_rootGroup);
            }
        }

        /// <summary>
        /// Creates a group and adds members to group, add child group and delete group
        /// Behavior test
        /// </summary>
        [TestMethod]
        public void TestForGroupMembershipCount()
        {
            Logger.Info("running test TestForGroupMembershipCount");
            var expectedMembershipCount = 14; //5 for root group, 5 for default channel, 4 for child group(excluding member)

            //get total memberships
            TryWaitForDatabase(() =>
                GroupMembership.GetAllLocal(p => p.RootGroupID, _rootGroup.GroupID).Count() ==
                expectedMembershipCount);
            var memberships = GroupMembership.GetAllLocal(p => p.RootGroupID, _rootGroup.GroupID);
                
            //compare the total groupmemberships created for this group and child group
            Assert.AreEqual(expectedMembershipCount, memberships.Count(), "groupmembership count not equal");
            var rootGroup = _rootGroup;
            var childGroup = _childGroup;

            //delete the root group and test if all the child and group memberships are deleted
            try
            {
                _rootGroup.DeleteRoot(Creator.UserID);
                TryWaitForDatabase(() => Group.GetLocal(_rootGroup.GroupID) == null);
                rootGroup = Group.GetLocal(_rootGroup.GroupID);
                //check if nothing comes from database
                Assert.IsNull(rootGroup);

                TryWaitForDatabase(() => Group.GetLocal(_childGroup.GroupID) == null);
                childGroup = Group.GetLocal(_childGroup.GroupID);
                Assert.IsNull(childGroup);

                TryWaitForDatabase(() => !GroupMembership.GetAllLocal(p => p.RootGroupID, _rootGroup.GroupID).Any());
                memberships = GroupMembership.GetAllLocal(p => p.RootGroupID, _rootGroup.GroupID);
                Assert.AreEqual(0, memberships.Count(), "groupmembership count not same");
            }
            finally
            {
                _rootGroup = rootGroup;
                _childGroup = childGroup;
                Logger.Info("completed test TestForGroupMembershipCount");
            }
        }

        /// <summary>
        /// Creates all subchannels with different group roles
        /// </summary>
        [TestMethod]
        public void TestPermissionsCreatingSubChannels()
        {
            _rootGroup.ChangeInfo(Creator.UserID, null, null, null, true, null);
            Group officerChannel = null;
            try
            {
                //officer cannot create channel
                officerChannel = AddSubChannel(_rootGroup, "Officer Channel", Officer);
            }
            catch (Exception)
            {
                //permanent channel cannot be created by officer
                Assert.IsNull(officerChannel);
            }
            
            //create and delete channel by admin
            CreateAndDeleteChannel("Admin channel", _rootGroup, LegacyGroupRole.Admin, GroupType.Large, Admin, Admin);

            //create temporary channel by Officer and delete by admin
            CreateAndDeleteChannel("Temporary channel", _rootGroup, LegacyGroupRole.Officer, GroupType.Temporary, Officer, Admin);

            //create and delete temporary channel by member
            CreateAndDeleteChannel("Temporary channel member", _rootGroup, LegacyGroupRole.Member, GroupType.Temporary, Member, Member);

            //create a multi-level channel and try deleting it.
            var permChannel = AddSubChannel(_rootGroup, "random permanent channel", Admin, LegacyGroupRole.Member);
            var permSubChannel = AddSubChannel(permChannel, "random permanent subchannel", Admin, LegacyGroupRole.Member);

            //deleting by officer should pass
            DeleteAndVerifyChannel(permChannel, Admin);

            //create temporary channel by admin and delete by member, it should fail.
            try
            {
                CreateAndDeleteChannel("Temporary channel admin", _rootGroup, LegacyGroupRole.Member, GroupType.Temporary,
                    Admin, Member);
                Assert.Fail("Members should not be able to delete temporary channels they did not create.");
            }
            catch (GroupPermissionException)
            {
                // Expected
            }
        }

        [ClassCleanup]
        public static void CleanupSuite()
        {   
        }

        /// <summary>
        /// Test creating and deleting channel by different user to create and destroy
        /// </summary>
        public static void CreateAndDeleteChannel(string title, Group parentGroup, LegacyGroupRole accessLevel, GroupType type, NewGroupMember creator,
            NewGroupMember destroyer)
        {
            //create channel
            var channel = AddSubChannel(parentGroup, title, creator, accessLevel, type);
            Assert.IsNotNull(channel);
            DeleteAndVerifyChannel(channel, destroyer);
        }

        /// <summary>
        /// delete channel by specified user
        /// </summary>
        public static void DeleteAndVerifyChannel(Group channel, NewGroupMember destroyer)
        {
            //delete channel
            RemoveSubChannel(channel, destroyer);
            //check if groupmemberships does not exist
            TryWaitForDatabase(() => !GroupMembership.GetAllLocal(p => p.GroupID, channel.GroupID).Any());
            var memberships = GroupMembership.GetAllLocal(p => p.GroupID, channel.GroupID);

            channel = Group.GetLocal(channel.GroupID);
            //channel should be there for soft delete
            Assert.IsNotNull(channel, "channel exists");
            Assert.AreEqual(channel.Status, GroupStatus.Deleted, "channel is not deleted");
            
            Assert.AreEqual(memberships.Count(), 0, 0, "groupmemberships exist for deleted channel");
        }
    }

    [TestClass]
    public class TestGroupInfo : ClansContext
    {
        private static Group _clanRoot;
        private static Group _normalGroup;

        [ClassInitialize]
        public static void Initialize(TestContext context)
        {
            _clanRoot = CreateRootGroup("Test invalid string large", GroupType.Large);
            _normalGroup = CreateRootGroup("Test invalid string normal", GroupType.Normal);
        }

        [ClassCleanup]
        public static void Cleanup()
        {
            CleanUpDatabase(new[] {_clanRoot, _normalGroup});
        }

        [TestMethod]
        public void TestInvalidTitle()
        {
            var title = "";
            var exceptionRaised = false;
            try
            {
                ChangeInfo(_clanRoot, Creator.UserID, title, "", null, null, null);
            }
            catch (Exception)
            {
                exceptionRaised = true;
            }
            Assert.IsTrue(exceptionRaised);

            //change title for normal group
            exceptionRaised = false;
            try
            {
                ChangeInfo(_normalGroup, Creator.UserID, title, "", null, null, null);
            }
            catch (Exception)
            {
                exceptionRaised = true;
            }
            Assert.IsTrue(exceptionRaised);
        }
    }
}
