﻿using System;
using System.Collections.Generic;
using System.Linq;
using Curse.Friends.Data;
using Curse.Friends.Data.DerivedModels;
using Curse.Friends.Enums;
using Curse.Friends.WebService.Tests.Groups;

namespace Curse.Friends.WebService.Tests
{
	public class ClansContext : GroupsContext
	{
        public static Group AddSubChannel(Group parent, string title, NewGroupMember channelCreator=null, LegacyGroupRole accessLevel = LegacyGroupRole.Officer, GroupType type = GroupType.Large)
        {
            var timestampedTitle = title + DateTime.UtcNow;
            channelCreator = channelCreator ?? Creator;

            var subgroup = ExecuteGroupChangeWithNotificationTimeout(
                () => parent.CreateChildGroup(channelCreator.UserID, channelCreator.Username,
                    channelCreator.RegionID, type, accessLevel, timestampedTitle, null),

                n => n.ChangeType == GroupChangeType.CreateGroup &&
                     n.Group.GroupTitle == timestampedTitle,

                string.Format("Channel {0} was not created.", timestampedTitle));

            TryWaitForDatabase(() => GroupMembership.GetAllLocal(p => p.GroupID, subgroup.GroupID).Any());

            return subgroup;
        }

        /// <summary>
        /// soft deletes or hard deletes a channel based on the arguments
        /// </summary>
        public static void RemoveSubChannel(Group channel, NewGroupMember member = null)
        {   
            member = member ?? Admin;

            ExecuteGroupChangeWithNotificationTimeout(
                () => channel.DeleteChildGroup(member.UserID),

                n => n.ChangeType == GroupChangeType.RemoveGroup &&
                     n.Group.GroupID == channel.GroupID,

                string.Format("Channel {0} was not deleted.", channel.Title));
        }

	    public static void ChangeUserRole(Group subChannel, int callerUserID, LegacyGroupRole newRole, int affectedUserID)
	    {
	        ExecuteGroupChangeWithNotificationTimeout(
	            () => subChannel.RootGroup.ChangeMemberRole(callerUserID, newRole, affectedUserID),

	            n => n.ChangeType == GroupChangeType.UpdateUsers &&
	                 n.Members.Any(m => m.UserID == affectedUserID && m.Role == newRole),

	            string.Format("User {0}'s role was not changed to {1}", affectedUserID, newRole));
	    }

	    public static void ChangeAccessLevel(Group channel, int callerUserID, LegacyGroupRole newRole)
	    {
	        ExecuteGroupChangeWithNotificationTimeout(
	            () => channel.ChangeAccessLevel(callerUserID, newRole),

	            n => n.ChangeType == GroupChangeType.ChangeInfo &&
                     n.Group.GroupAccessLevel == newRole,

	            string.Format("Channel {0}'s access level was not changed to {1}", channel.Title, newRole));
	    }

        public static void DeleteInvitation(Group channel, GroupInvitation invite, int callerUserID, bool removeMembers)
        {
            if (removeMembers)
            {
                // Get the member list with this invite code:
                var guestsWithCode = channel.MemberList.Members.GetValues()
                                        .Where(p => p.Role == LegacyGroupRole.Guest && p.InviteCode == invite.InviteCode)
                                        .Select(p => p.UserID)
                                        .ToArray();

                ExecuteGroupChangeWithNotificationTimeout(
                    () => channel.DeleteInvitation(invite, callerUserID, removeMembers),

                    n => n.ChangeType == GroupChangeType.RemoveUsers && n.Group.GroupID == channel.GroupID
                         && guestsWithCode.All(m => n.Members.Any(u => u.UserID == m)),

                    string.Format("Group {0}'s member list was not properly updated after deleting an invitation",
                        channel.Title));
            }
            else
            {
                channel.DeleteInvitation(invite, callerUserID, removeMembers);
            }                       
        }

	    public static void ClaimInvitation(Group channel, GroupInvitation invite, User claimUser,
	        UserRegion claimUserRegion)
	    {

	        ExecuteGroupChangeWithNotificationTimeout(
	            () => channel.ProcessInvitation(invite, claimUser, claimUserRegion),

	            n => n.ChangeType == GroupChangeType.AddUsers && n.Group.GroupID == channel.GroupID
	                 && n.Members.Any(p => p.UserID == claimUser.UserID),

	            string.Format("Group {0}'s member list was not properly updated after a user claimed an invite",
	                channel.Title));
	    }

	    public static void ReorganizeChildren(Group root, GroupTree newStructure, int callerUserID)
	    {
	        ExecuteGroupChangeWithNotificationTimeout(
	            () => root.ReorganizeChildren(newStructure, callerUserID),

	            n => n.ChangeType == GroupChangeType.GroupReorganized && n.Group.GroupID == root.GroupID,

	            string.Format("Channel {0}'s children were not restructured.", root.Title));
	    }

	    public static void ChangePermissionLevels(Group root, Dictionary<GroupPermissions,int> permissionsChanges, int requestorID)
	    {
	        ExecuteGroupChangeWithNotificationTimeout(
	            () => root.ChangePermissionLevels(requestorID, permissionsChanges),

	            n => n.ChangeType == GroupChangeType.PermissionsChanged &&
	                 n.Group.GroupID == root.GroupID,

	            string.Format("Group {0}'s permissions were not changed.", root.Title));
	    }
	}
 }

