﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Curse.Friends.Data;
using Curse.Friends.Data.DerivedModels;
using Curse.Friends.Enums;
using Microsoft.VisualStudio.TestTools.UnitTesting;

namespace Curse.Friends.WebService.Tests.Clans.DefaultPermissions
{
    [TestClass]
    public abstract class DefaultPermissionsTest : ClansContext
    {
        protected Group ClanRoot;

        [TestInitialize]
        public void InitTest()
        {
            try
            {
                ClanRoot = CreateRootGroup("DefaultPermissionsTests", GroupType.Large);
                AddMembers(ClanRoot, Creator.UserID, Guest);
            }
            catch
            {
                CleanUpTest();
                throw;
            }
        }

        [TestMethod]
        public void Owner_Test()
        {
            TestPermissions(Permission, Creator, ShouldSucceed[Creator.Role]);
        }

        [TestMethod]
        public void Admin_Test()
        {
            TestPermissions(Permission, Admin, ShouldSucceed[Admin.Role]);
        }

        [TestMethod]
        public void Officer_Test()
        {
            TestPermissions(Permission, Officer, ShouldSucceed[Officer.Role]);
        }

        [TestMethod]
        public void Member_Test()
        {
            TestPermissions(Permission, Member, ShouldSucceed[Member.Role]);
        }

        [TestMethod]
        public void Guest_Test()
        {
            TestPermissions(Permission, Guest, ShouldSucceed[Guest.Role]);
        }

        [TestCleanup]
        public void CleanUpTest()
        {
            CleanUpDatabase(ClanRoot);
        }

        protected void TestPermissions(GroupPermissions permission, NewGroupMember requestor, bool shouldSucceed)
        {
            try
            {
                TestAction(requestor);

                if (!shouldSucceed)
                {
                    Assert.Fail("Permission {0} check should have failed for user {1}.", permission, requestor.Username);
                }
            }
            catch (GroupPermissionException)
            {
                if (shouldSucceed)
                {
                    Assert.Fail("Permission {0} check should have succeeded for user {1}.", permission, requestor.Username);
                }
            }
        }

        protected abstract void TestAction(NewGroupMember requestor);
        protected abstract GroupPermissions Permission { get; }
        protected abstract Dictionary<LegacyGroupRole, bool> ShouldSucceed { get; } 
    }

    [TestClass]
    public class AllDefaultPermissionsMapped
    {
        [TestMethod]
        public void All_Permission_Types_Are_Defaulted()
        {
            var defaultPermissions = new Group().GetAllPermissions();
            foreach (GroupPermissions permission in Enum.GetValues(typeof(GroupPermissions)))
            {
                Assert.IsTrue(defaultPermissions.ContainsKey(permission));
            }
        }
    }

    [TestClass]
    public class DeleteClan : DefaultPermissionsTest
    {
        protected override void TestAction(NewGroupMember requestor)
        {
            DeleteRootGroup(ClanRoot, requestor.UserID);
        }

        protected override GroupPermissions Permission
        {
            get { return GroupPermissions.ManageChannels; }
        }

        protected override Dictionary<LegacyGroupRole, bool> ShouldSucceed
        {
            get
            {
                return new Dictionary<LegacyGroupRole, bool>
                {
                    {LegacyGroupRole.Owner, true},
                    {LegacyGroupRole.Admin, false},
                    {LegacyGroupRole.Officer, false},
                    {LegacyGroupRole.Member, false},
                    {LegacyGroupRole.Guest, false}
                };
            }
        }
    }

    [TestClass]
    public class SetClanMotd : DefaultPermissionsTest
    {
        protected override void TestAction(NewGroupMember requestor)
        {
            ChangeInfo(ClanRoot, requestor.UserID, null, null, "New MOTD", null, null);
        }

        protected override GroupPermissions Permission
        {
            get { return GroupPermissions.ManageChannels; }
        }

        protected override Dictionary<LegacyGroupRole, bool> ShouldSucceed
        {
            get
            {
                return new Dictionary<LegacyGroupRole, bool>
                {
                    {LegacyGroupRole.Owner, true},
                    {LegacyGroupRole.Admin, true},
                    {LegacyGroupRole.Officer, false},
                    {LegacyGroupRole.Member, false},
                    {LegacyGroupRole.Guest, false}
                };
            }
        }
    }

    [TestClass]
    public class ModifyRoles : DefaultPermissionsTest
    {
        protected override void TestAction(NewGroupMember requestor)
        {
            ClanRoot.ChangePermissionLevels(requestor.UserID, new Dictionary<GroupPermissions, int>
            {
                {GroupPermissions.CreateTemporaryGroup, (int) LegacyGroupRole.Guest}
            });
        }

        protected override GroupPermissions Permission
        {
            get { return GroupPermissions.ManageServer; }
        }

        protected override Dictionary<LegacyGroupRole, bool> ShouldSucceed
        {
            get
            {
                return new Dictionary<LegacyGroupRole, bool>
                {
                    {LegacyGroupRole.Owner, true},
                    {LegacyGroupRole.Admin, true},
                    {LegacyGroupRole.Officer, false},
                    {LegacyGroupRole.Member, false},
                    {LegacyGroupRole.Guest, false}
                };
            }
        }
    }

    [TestClass]
    public class AddPermanentChildGroup : DefaultPermissionsTest
    {
        protected override GroupPermissions Permission
        {
            get { return GroupPermissions.ManageChannels; }
        }

        protected override Dictionary<LegacyGroupRole, bool> ShouldSucceed
        {
            get
            {
                return new Dictionary<LegacyGroupRole, bool>
                {
                    {LegacyGroupRole.Owner, true},
                    {LegacyGroupRole.Admin, true},
                    {LegacyGroupRole.Officer, false},
                    {LegacyGroupRole.Member, false},
                    {LegacyGroupRole.Guest, false}
                };
            }
        }

        protected override void TestAction(NewGroupMember requestor)
        {
            AddSubChannel(ClanRoot, "AddedGroup", requestor, requestor.Role);
        }
    }

    [TestClass]
    public class DeletePermanentGroup : DefaultPermissionsTest
    {
        private Group _channelToDelete;

        [TestInitialize]
        public void Init()
        {
            _channelToDelete = AddSubChannel(ClanRoot, "ChannelToDelete", Creator);
        }

        protected override void TestAction(NewGroupMember requestor)
        {
            RemoveSubChannel(_channelToDelete, requestor);
        }

        protected override GroupPermissions Permission
        {
            get { return GroupPermissions.ManageChannels; }
        }

        protected override Dictionary<LegacyGroupRole, bool> ShouldSucceed
        {
            get
            {
                return new Dictionary<LegacyGroupRole, bool>
                {
                    {LegacyGroupRole.Owner, true},
                    {LegacyGroupRole.Admin, true},
                    {LegacyGroupRole.Officer, false},
                    {LegacyGroupRole.Member, false},
                    {LegacyGroupRole.Guest, false}
                };
            }
        }
    }

    [TestClass]
    public class CreateTemporaryGroup : DefaultPermissionsTest
    {
        [TestInitialize]
        public void Init()
        {
            ChangeInfo(ClanRoot, Creator.UserID, null, null, null, true, null);
        }

        protected override void TestAction(NewGroupMember requestor)
        {
            AddSubChannel(ClanRoot, "AddedTempGroup", requestor, requestor.Role, GroupType.Temporary);
        }

        protected override GroupPermissions Permission
        {
            get { return GroupPermissions.CreateTemporaryGroup; }
        }

        protected override Dictionary<LegacyGroupRole, bool> ShouldSucceed
        {
            get
            {
                return new Dictionary<LegacyGroupRole, bool>
                {
                    {LegacyGroupRole.Owner, true},
                    {LegacyGroupRole.Admin, true},
                    {LegacyGroupRole.Officer, true},
                    {LegacyGroupRole.Member, true},
                    {LegacyGroupRole.Guest, false}
                };
            }
        }
    }

    [TestClass]
    public class DeleteTemporaryGroup : DefaultPermissionsTest
    {
        private Group _tempChannelToDelete;

        [TestInitialize]
        public void Init()
        {
            ChangeInfo(ClanRoot, Creator.UserID, null, null, null, true, null);
            _tempChannelToDelete = AddSubChannel(ClanRoot, "TempChannel To Delete", Creator, LegacyGroupRole.Guest, GroupType.Temporary);
        }

        protected override void TestAction(NewGroupMember requestor)
        {
            RemoveSubChannel(_tempChannelToDelete, requestor);
        }

        protected override GroupPermissions Permission
        {
            get { return GroupPermissions.ManageChannels; }
        }

        protected override Dictionary<LegacyGroupRole, bool> ShouldSucceed
        {
            get
            {
                return new Dictionary<LegacyGroupRole, bool>
                {
                    {LegacyGroupRole.Owner, true},
                    {LegacyGroupRole.Admin, true},
                    {LegacyGroupRole.Officer, true},
                    {LegacyGroupRole.Member, false},
                    {LegacyGroupRole.Guest, false}
                };
            }
        }
    }

    [TestClass]
    public class ModifyGroup : DefaultPermissionsTest
    {
        private Group _channel;

        [TestInitialize]
        public void Init()
        {
            _channel = AddSubChannel(ClanRoot, "ModifyGroup Permission Test Channel", Creator, LegacyGroupRole.Guest);
        }
        protected override void TestAction(NewGroupMember requestor)
        {
            ChangeInfo(_channel, requestor.UserID, string.Format("Channel for {0}", requestor.Username), "http://new/url", null,
                true, true);
        }

        protected override GroupPermissions Permission
        {
            get { return GroupPermissions.ManageChannels; }
        }

        protected override Dictionary<LegacyGroupRole, bool> ShouldSucceed
        {
            get
            {
                return new Dictionary<LegacyGroupRole, bool>
                {
                    {LegacyGroupRole.Owner, true},
                    {LegacyGroupRole.Admin, true},
                    {LegacyGroupRole.Officer, false},
                    {LegacyGroupRole.Member, false},
                    {LegacyGroupRole.Guest, false}
                };
            }
        }
    }

    [TestClass]
    public class AddUser : DefaultPermissionsTest
    {
        // TODO: Use a role value between guest and member to really test AddUser for member.
        //private static readonly NewGroupMember NewSubMember = new NewGroupMember(LowestReservedUserID - 1, 1,
        //    "New Sub-Member", GroupRole.Member - 1);

        protected override void TestAction(NewGroupMember requestor)
        {
            AddMembers(ClanRoot, requestor.UserID, NewMember);
        }

        protected override GroupPermissions Permission
        {
            get
            {
                return GroupPermissions.InviteUsers;
            }
        }

        protected override Dictionary<LegacyGroupRole, bool> ShouldSucceed
        {
            get
            {
                return new Dictionary<LegacyGroupRole, bool>
                {
                    {LegacyGroupRole.Owner, true},
                    {LegacyGroupRole.Admin, true},
                    {LegacyGroupRole.Officer, true},
                    {LegacyGroupRole.Member, false},
                    {LegacyGroupRole.Guest, false}
                };
            }
        }
    }

    [TestClass]
    public class RemoveUser : DefaultPermissionsTest
    {
        [TestInitialize]
        public void Init()
        {
            AddMembers(ClanRoot, Creator.UserID, NewMember);
        }

        protected override void TestAction(NewGroupMember requestor)
        {
            RemoveMembers(ClanRoot, requestor.UserID, NewMember.UserID);
        }

        protected override GroupPermissions Permission
        {
            get { return GroupPermissions.RemoveUser; }
        }

        protected override Dictionary<LegacyGroupRole, bool> ShouldSucceed
        {
            get
            {
                return new Dictionary<LegacyGroupRole, bool>
                {
                    {LegacyGroupRole.Owner, true},
                    {LegacyGroupRole.Admin, true},
                    {LegacyGroupRole.Officer, false},
                    {LegacyGroupRole.Member, false},
                    {LegacyGroupRole.Guest, false}
                };
            }
        }
    }

    [TestClass]
    public class InviteGuests : DefaultPermissionsTest
    {
        private GroupInvitation _groupInvitation;

        [TestCleanup]
        public void Cleanup()
        {
            if (_groupInvitation != null)
            {
                _groupInvitation.Delete();
            }
        }

        protected override void TestAction(NewGroupMember requestor)
        {
            _groupInvitation = ClanRoot.CreateGroupInvitation(requestor.UserID, false, null);
        }

        protected override GroupPermissions Permission
        {
            get { return GroupPermissions.InviteUsers; }
        }

        protected override Dictionary<LegacyGroupRole, bool> ShouldSucceed
        {
            get
            {
                return new Dictionary<LegacyGroupRole, bool>
                {
                    {LegacyGroupRole.Owner, true},
                    {LegacyGroupRole.Admin, true},
                    {LegacyGroupRole.Officer, true},
                    {LegacyGroupRole.Member, true},
                    {LegacyGroupRole.Guest, false}
                };
            }
        }
    }

    [TestClass]
    public class ManageInvitations : DefaultPermissionsTest
    {
        protected override void TestAction(NewGroupMember requestor)
        {
            ClanRoot.GetGroupInvitations(requestor.UserID, false, true);
        }

        protected override GroupPermissions Permission
        {
            get { return GroupPermissions.ManageInvitations; }
        }

        protected override Dictionary<LegacyGroupRole, bool> ShouldSucceed
        {
            get
            {
                return new Dictionary<LegacyGroupRole, bool>
                {
                    {LegacyGroupRole.Owner, true},
                    {LegacyGroupRole.Admin, true},
                    {LegacyGroupRole.Officer, true},
                    {LegacyGroupRole.Member, false},
                    {LegacyGroupRole.Guest, false}
                };
            }
        }
    }

    [TestClass]
    public class AddGuest : DefaultPermissionsTest
    {
        private static readonly NewGroupMember NewGuest = new NewGroupMember(LowestReservedUserID - 1, 1, "NewGuest",
            LegacyGroupRole.Guest);

        protected override void TestAction(NewGroupMember requestor)
        {
            AddMembers(ClanRoot, requestor.UserID, NewGuest);
        }

        protected override GroupPermissions Permission
        {
            get { return GroupPermissions.InviteUsers; }
        }

        protected override Dictionary<LegacyGroupRole, bool> ShouldSucceed
        {
            get
            {
                return new Dictionary<LegacyGroupRole, bool>
                {
                    {LegacyGroupRole.Owner, true},
                    {LegacyGroupRole.Admin, true},
                    {LegacyGroupRole.Officer, true},
                    {LegacyGroupRole.Member, true},
                    {LegacyGroupRole.Guest, false}
                };
            }
        }
    }

    [TestClass]
    public class RemoveGuest : DefaultPermissionsTest
    {
        private static readonly NewGroupMember NewGuest = new NewGroupMember(LowestReservedUserID - 1, 1, "NewGuest", LegacyGroupRole.Guest);

        [TestInitialize]
        public void Init()
        {
            AddMembers(ClanRoot, Creator.UserID, NewGuest);
        }


        protected override void TestAction(NewGroupMember requestor)
        {
            RemoveMembers(ClanRoot, requestor.UserID, NewGuest.UserID);
        }

        protected override GroupPermissions Permission
        {
            get { return GroupPermissions.RemoveUser; }
        }

        protected override Dictionary<LegacyGroupRole, bool> ShouldSucceed
        {
            get
            {
                return new Dictionary<LegacyGroupRole, bool>
                {
                    {LegacyGroupRole.Owner, true},
                    {LegacyGroupRole.Admin, true},
                    {LegacyGroupRole.Officer, true},
                    {LegacyGroupRole.Member, true},
                    {LegacyGroupRole.Guest, false}
                };
            }
        }
    }

    //public class KickUserFromVoiceSession : DefaultPermissionsTest
    //{

    //}

    //public class MuteUserGlobally : DefaultPermissionsTest
    //{

    //}
}
