﻿using System;
using System.Linq;
using Curse.Friends.Data.DerivedModels;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Curse.Friends.Data;
using Curse.Friends.Data.Models;
using Curse.Friends.Enums;
using Newtonsoft.Json;

namespace Curse.Friends.WebService.Tests.Clans
{
    public static class GroupInvitationTests
    {
        [TestClass]
        public class CreateInvitation : ClansContext
        {
            private static Group _clanRoot;
            private static GroupInvitation _groupInvite;
            private static NewGroupMember _inviteCreator;

            [ClassInitialize]
            public static void Initialize(TestContext context)
            {
                try
                {
                    _inviteCreator = Officer;
                    _clanRoot = CreateRootGroup("AddUserToClan Test Clan", GroupType.Large);
                    _groupInvite = _clanRoot.CreateGroupInvitation(_inviteCreator.UserID, false, TimeSpan.FromDays(1));
                }
                catch (Exception)
                {
                    CleanUp();
                    throw;
                }
            }

            [TestMethod]
            public void Creates_Invitation()
            {
                var allInvites = GroupInvitation.GetAllByGroupID(_clanRoot.GroupID);

                // Ensure 1 and only 1 invite exists in the DB
                Assert.AreEqual(1, allInvites.Count());

                var invite = allInvites.FirstOrDefault();

                Assert.IsNotNull(invite);

                Assert.IsFalse(invite.IsExpired);

                Assert.IsFalse(invite.AutoRemoveMembers);

                Assert.IsFalse(string.IsNullOrEmpty(invite.InviteCode));

                Assert.AreEqual(invite.CreatorID, _inviteCreator.UserID);
            }
            

            [ClassCleanup]
            public static void CleanUp()
            {
                CleanUpDatabase(_clanRoot);
                if (_groupInvite != null)
                {
                    _groupInvite.Delete();
                }

            }
        }

        [TestClass]
        public class ClaimInvitation : ClansContext
        {
            private static Group _clanRoot;
            private static GroupInvitation _groupInvite;
            private static NewGroupMember _inviteCreator;

            [ClassInitialize]
            public static void Initialize(TestContext context)
            {
                try
                {
                    _inviteCreator = Officer;
                    _clanRoot = CreateRootGroup("AddUserToClan Test Clan", GroupType.Large);
                    _groupInvite = _clanRoot.CreateGroupInvitation(_inviteCreator.UserID, false, TimeSpan.FromDays(1));

                    var testClaimUser = new User { UserID = NewMember.UserID, Username = NewMember.Username };
                    var testClaimUserRegion = new UserRegion { UserID = NewMember.UserID, RegionID = NewMember.RegionID };

                    ClaimInvitation(_clanRoot, _groupInvite, testClaimUser, testClaimUserRegion);
                }
                catch (Exception)
                {
                    CleanUp();
                    throw;
                }
            }

            [TestMethod]
            public void Adds_Member_To_Group()
            {
                Assert.IsNotNull(_clanRoot.MemberList.Members.Find(NewMember.UserID));
            }

            [TestMethod]
            public void Adds_Group_Membership()
            {
                var allMembers = GroupMembership.GetAllLocal(p => p.GroupID, _clanRoot.GroupID);                
                Assert.IsNotNull(allMembers.FirstOrDefault(m => m.UserID == NewMember.UserID));
            }

            [TestMethod]
            public void Has_Group_Access()
            {
                _clanRoot.CheckViewPermission(NewMember.UserID);
            }

            [TestMethod]
            public void Logs_User_Invited_Event()
            {
                Assert.IsTrue(GroupEventLog.GetLocal(_clanRoot.GroupID).Events.GetValues()
                    .Any(e => e.Type == GroupEventType.UserInvited &&
                              e.InitiatingUserID == _groupInvite.CreatorID &&
                              JsonConvert.DeserializeAnonymousType(e.EventData,
                                  new {InvitedUser = 0}).InvitedUser == NewMember.UserID));
            }

            [ClassCleanup]
            public static void CleanUp()
            {
                CleanUpDatabase(_clanRoot);
                if (_groupInvite != null)
                {
                    _groupInvite.Delete();
                }

            }
        }

        [TestClass]
        public class DeleteInvitation : ClansContext
        {
            private static Group _clanRoot;
            private static GroupInvitation _groupInvite;
            private static NewGroupMember _inviteCreator;

            [ClassInitialize]
            public static void Initialize(TestContext context)
            {
                try
                {
                    _inviteCreator = Officer;
                    _clanRoot = CreateRootGroup("AddUserToClan Test Clan", GroupType.Large);
                    _groupInvite = _clanRoot.CreateGroupInvitation(_inviteCreator.UserID, false, TimeSpan.FromDays(1));

                    var testClaimUser = new User { UserID = NewMember.UserID, Username = NewMember.Username };
                    var testClaimUserRegion = new UserRegion { UserID = NewMember.UserID, RegionID = NewMember.RegionID };

                    ClaimInvitation(_clanRoot, _groupInvite, testClaimUser, testClaimUserRegion);

                    DeleteInvitation(_clanRoot, _groupInvite, _inviteCreator.UserID, true);
                }
                catch
                {
                    CleanUp();
                    throw;
                }
            }

            [TestMethod]
            public void Remove_Member_From_Group()
            {
                Assert.IsNull(_clanRoot.MemberList.Members.Find(NewMember.UserID));
            }

            [TestMethod]
            public void Removes_Group_Membership()
            {
                var allMembers = GroupMembership.GetAllLocal(p => p.GroupID, _clanRoot.GroupID);
                Assert.IsNull(allMembers.FirstOrDefault(m => m.UserID == NewMember.UserID));
            }

            [TestMethod]
            public void No_Group_Access()
            {
                try
                {
                    _clanRoot.CheckViewPermission(NewMember.UserID);
                    Assert.Fail("User still has group access, even though invite deleted!");
                }
                catch (GroupPermissionException)
                {
                    
                }
                
            }

            [ClassCleanup]
            public static void CleanUp()
            {
                CleanUpDatabase(_clanRoot);
                if (_groupInvite != null)
                {
                    _groupInvite.Delete();
                }

            }
        }

    }
}
