﻿using System;
using System.Collections.Generic;
using System.Linq;
using Curse.Friends.Data;
using Curse.Friends.Data.DerivedModels;
using Curse.Friends.Data.Models;
using Curse.Friends.Enums;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Newtonsoft.Json;

namespace Curse.Friends.WebService.Tests.Clans
{
    public static class UserManagementTests
    {
        [TestClass]
        public class AddUserToClan : ClansContext
        {
            private static Group _clanRoot;
            private static Group _memberChannel;

            [ClassInitialize]
            public static void TestAddingUsers(TestContext context)
            {
                try
                {
                    _clanRoot = CreateRootGroup("AddUserToClan Test Clan", GroupType.Large);
                    _memberChannel = AddSubChannel(_clanRoot, "AddUserToClan Test Member Channel", Creator,
                        LegacyGroupRole.Member);

                    AddMembers(_clanRoot, Admin.UserID, NewMember);
                }
                catch (Exception)
                {
                    CleanUp();
                    throw;
                }
            }

            [TestMethod]
            public void Adds_Member_To_Clan()
            {
                Assert.IsNotNull(_clanRoot.MemberList.Members.Find(NewMember.UserID));
            }

            [TestMethod]
            public void Adds_Group_Membership_To_Clan_Root()
            {
                TryWaitForDatabase(() =>
                    GroupMembership.GetByUserAndGroupID(NewMember.RegionID, NewMember.UserID, _clanRoot.GroupID) != null);
                Assert.IsNotNull(GroupMembership.GetByUserAndGroupID(NewMember.RegionID, NewMember.UserID,
                    _clanRoot.GroupID));
            }

            [TestMethod]
            public void Adds_Group_Membership_To_Clan_Subchannels()
            {
                Assert.IsNotNull(GroupMembership.GetAllLocal(p => p.GroupID, _memberChannel.GroupID)
                    .FirstOrDefault(m => m.UserID == NewMember.UserID));
            }

            [TestMethod]
            public void Logs_Users_Added_Event()
            {
                Assert.IsTrue(GroupEventLog.GetLocal(_clanRoot.GroupID).Events.GetValues()
                    .Any(e => e.Type == GroupEventType.UsersAdded &&
                              JsonConvert.DeserializeAnonymousType(e.EventData, new {AddedUsers = new int[0]})
                                  .AddedUsers.Contains(NewMember.UserID)));
            }

	    [TestMethod]
            public void Updates_Root_Group_Membership_User_Counts()
            {
                foreach (var membership in GroupMembership.GetAllLocal(p => p.GroupID, _clanRoot.GroupID))
                {
                    Assert.AreEqual(DefaultMembers.Length + 1, membership.GroupMemberCount);
                }
            }

            [ClassCleanup]
            public static void CleanUp()
            {
                CleanUpDatabase(_clanRoot);
            }
        }

        [TestClass]
        public class RemoveUserFromClan : ClansContext
        {
            private static Group _clanRoot;
            private static Group _membersChannel;

            [ClassInitialize]
            public static void TestRemovingUsers(TestContext context)
            {
                try
                {
                    _clanRoot = CreateRootGroup("RemoveUserFromClan Test Clan", GroupType.Large);
                    _membersChannel = AddSubChannel(_clanRoot, "RemoveUserFromClan Member Channel", Creator,
                        LegacyGroupRole.Member);
                    RemoveMembers(_clanRoot, Creator.UserID, Member.UserID);
                }
                catch (Exception)
                {
                    CleanUp();
                    throw;
                }
            }

            [TestMethod]
            public void Removes_Member_From_Clan()
            {
                Assert.IsNull(_clanRoot.MemberList.Members.Find(Member.UserID));
            }

            [TestMethod]
            public void Removes_Membership_From_Clan_Root()
            {
                TryWaitForDatabase(() =>
                    GroupMembership.GetByUserAndGroupID(Member.RegionID, Member.UserID, _clanRoot.GroupID) == null);
                Assert.IsNull(GroupMembership.GetByUserAndGroupID(Member.RegionID, Member.UserID,
                    _clanRoot.GroupID));
            }

            [TestMethod]
            public void Removes_Membership_From_Clan_Channels()
            {
                TryWaitForDatabase(() =>
                    GroupMembership.GetByUserAndGroupID(Member.RegionID, Member.UserID, _membersChannel.GroupID) == null);
                Assert.IsNull(GroupMembership.GetByUserAndGroupID(Member.RegionID, Member.UserID,
                    _membersChannel.GroupID));
            }

            [TestMethod]
            public void Updates_Root_Group_Membership_User_Counts()
            {
                foreach (var membership in GroupMembership.GetAllLocal(p => p.GroupID, _clanRoot.GroupID))
                {
                    Assert.AreEqual(DefaultMembers.Length - 1, membership.GroupMemberCount);
                }
            }

            [TestMethod]
            public void Logs_Users_Removed_Event()
            {
                Assert.IsTrue(GroupEventLog.GetLocal(_clanRoot.GroupID).Events.GetValues()
                                    .Any(e => e.Type == GroupEventType.UsersRemoved &&
                                              JsonConvert.DeserializeAnonymousType(e.EventData, new { RemovedUsers = new int[0] })
                                                  .RemovedUsers.Contains(Member.UserID)));
            }

            [ClassCleanup]
            public static void CleanUp()
            {
                CleanUpDatabase(_clanRoot);
            }
        }

        [TestClass]
        public class PromoteUser : ClansContext
        {
            private static Group _clanRoot;
            private static Group _officerChannel;

            [ClassInitialize]
            public static void Initialize(TestContext context)
            {
                try
                {
                    _clanRoot = CreateRootGroup("PromoteUser Test Clan", GroupType.Large);
                    _officerChannel = AddSubChannel(_clanRoot, "PromoteUser Test Officer Channel", Creator);
                    ChangeUserRole(_officerChannel, Admin.UserID, LegacyGroupRole.Officer, Member.UserID);
                }
                catch (Exception)
                {
                    CleanUp();
                    throw;
                }
            }

            [TestMethod]
            public void Adds_Memberships_For_Subchannels()
            {
                TryWaitForDatabase(() =>
                    GroupMembership.GetByUserAndGroupID(Member.RegionID, Member.UserID, _officerChannel.GroupID) != null);
                Assert.IsNotNull(GroupMembership.GetByUserAndGroupID(Member.RegionID, Member.UserID, _officerChannel.GroupID));
            }

            [TestMethod]
            public void Changes_Member_Role()
            {
                Assert.AreEqual(LegacyGroupRole.Officer,
                    GroupMemberList.GetLocal(_clanRoot.GroupID).Members.Find(Member.UserID).Role);
            }

            [TestMethod]
            public void Logs_User_Promoted_Event()
            {
                Assert.IsTrue(GroupEventLog.GetLocal(_clanRoot.GroupID).Events.GetValues()
                    .Any(e => e.Type == GroupEventType.UserPromoted &&
                              JsonConvert.DeserializeAnonymousType(e.EventData,
                                  new {PromotedUser = 0, FormerRole = "", NewRole = ""})
                                  .PromotedUser == Member.UserID));
            }

            [ClassCleanup]
            public static void CleanUp()
            {
                CleanUpDatabase(_clanRoot);
            }
        }

        [TestClass]
        public class DemoteUser : ClansContext
        {
            private static Group _clanRoot;
            private static Group _officerChannel;

            [ClassInitialize]
            public static void Initialize(TestContext context)
            {
                try
                {
                    _clanRoot = CreateRootGroup("DemoteUser Test Clan", GroupType.Large);
                    _officerChannel = AddSubChannel(_clanRoot, "DemoteUser Test Officer Channel", Creator);
                    ChangeUserRole(_officerChannel, Creator.UserID, LegacyGroupRole.Member, Officer.UserID);

                    // sleep for up to DatabaseTimeoutMillis milliseconds
                    TryWaitForDatabase(() =>
                        GroupMembership.GetAllLocal(p => p.GroupID, _officerChannel.GroupID)
                            .All(m => m.UserID != Officer.UserID));
                }
                catch (Exception)
                {
                    CleanUp();
                    throw;
                }
            }

            [TestMethod]
            public void Removes_Memberships_From_Subchannels()
            {
                Assert.IsNull(GroupMembership.GetAllLocal(p => p.GroupID, _officerChannel.GroupID)
                    .FirstOrDefault(m => m.UserID == Officer.UserID));
            }

            [TestMethod]
            public void Changes_User_Role()
            {
                Assert.AreEqual(LegacyGroupRole.Member,
                    GroupMemberList.GetLocal(_clanRoot.GroupID).Members.Find(Officer.UserID).Role);
            }

            [TestMethod]
            public void Logs_User_Demoted_Event()
            {
                Assert.IsTrue(GroupEventLog.GetLocal(_clanRoot.GroupID).Events.GetValues()
                    .Any(e => e.Type == GroupEventType.UserDemoted &&
                              JsonConvert.DeserializeAnonymousType(e.EventData,
                                  new {DemotedUser = 0, FormerRole = "", NewRole = ""})
                                  .DemotedUser == Officer.UserID));
            }

            [ClassCleanup]
            public static void CleanUp()
            {
                CleanUpDatabase(_clanRoot);
            }
        }

        [TestClass]
        public class PromoteUserToPeer : ClansContext
        {
            private static Group _clanRoot;
            private static GroupPermissionException _permissionException;

            [ClassInitialize]
            public static void Initialize(TestContext context)
            {
                try
                {
                    _clanRoot = CreateRootGroup("PromoteUserToPeer Test Clan", GroupType.Large);

                    try
                    {
                        ChangeUserRole(_clanRoot, Officer.UserID, Officer.Role, Member.UserID);
                    }
                    catch (GroupPermissionException ex)
                    {
                        _permissionException = ex;
                    }
                }
                catch (Exception)
                {
                    CleanUp();
                    throw;
                }
            }

            [TestMethod]
            public void Throws_Permission_Exception()
            {
                Assert.IsNotNull(_permissionException);
            }

            [TestMethod]
            public void Does_Not_Change_Permission()
            {
                Assert.AreEqual(Member.Role, _clanRoot.MemberList.Members.Find(Member.UserID).Role);
            }

            [ClassCleanup]
            public static void CleanUp()
            {
                CleanUpDatabase(_clanRoot);
            }
        }

        [TestClass]
        public class DemotePeerUser : ClansContext
        {
            private static Group _rootClan;
            private static GroupPermissionException _permissionException;

            [ClassInitialize]
            public static void Initialize(TestContext context)
            {
                try
                {
                    _rootClan = CreateRootGroup("DemotePeerUser Test Clan", GroupType.Large);

                    try
                    {
                        ChangeUserRole(_rootClan, Officer.UserID, LegacyGroupRole.Member, PeerOfficer.UserID);
                    }
                    catch (GroupPermissionException ex)
                    {
                        _permissionException = ex;
                    }
                }
                catch (Exception)
                {
                    CleanUp();
                    throw;
                }
            }

            [TestMethod]
            public void Throws_Permission_Exception()
            {
                Assert.IsNotNull(_permissionException);
            }

            [TestMethod]
            public void Does_Not_Demote_User()
            {
                Assert.AreEqual(PeerOfficer.Role, _rootClan.MemberList.Members.Find(PeerOfficer.UserID).Role);
            }

            [ClassCleanup]
            public static void CleanUp()
            {
                CleanUpDatabase(_rootClan);
            }
        }

        [TestClass]
        public class UserNotificationPreferences : ClansContext
        {
            private static Group _rootClan;
            private static GroupMembership _adminMembership;
            private static GroupMembership _officerMembership;
            
            [ClassInitialize]
            public static void Initialize(TestContext context)
            {
                try
                {
                    _rootClan = CreateRootGroup("User Preference testing", GroupType.Large);
                    _adminMembership = GroupMembership.GetByUserAndGroupID(new UserRegion() { UserID = Admin.UserID, RegionID = Admin.RegionID }, _rootClan.GroupID);
                    _officerMembership = GroupMembership.GetByUserAndGroupID(new UserRegion() { UserID = Officer.UserID, RegionID = Officer.RegionID }, _rootClan.GroupID);
                }
                catch (Exception)
                {
                    CleanUp();
                    throw;
                }
            }

            [TestMethod]
            public void TestDefaultPreferenceMember()
            {
                //A user should not be notified to the group by default.
                CompareUserNotificationPreference(_rootClan, Member, NotificationPreference.Disabled);
            }

            [TestMethod]
            public void TestEnableNotificationPreferenceofAdmin()
            {
                //enable notification
                var preference = NotificationPreference.Enabled;
                _adminMembership.UpdateUserPreference(preference, null);
                CompareUserNotificationPreference(_rootClan, Admin, preference);
            }

            [TestMethod]
            public void TestFilteredNotificationOfficer()
            {
                var preference = NotificationPreference.Filtered;
                _officerMembership.UpdateUserPreference(preference, new HashSet<string>(){"lunch", "wow", "lol"});
                CompareUserNotificationPreference(_rootClan, Officer, preference);
            }

            //test if the member is notified by default for normal group
            [TestMethod]
            public void TestNormalGroupNotification()
            {
                var normalGroup = CreateRootGroup("User Preference Normal group", GroupType.Normal);
                CompareUserNotificationPreference(normalGroup, Admin, NotificationPreference.Enabled);
                CleanUpDatabase(normalGroup);
            }

            private static void CompareUserNotificationPreference(Group group, NewGroupMember member, NotificationPreference preference)
            {
                var memberMembership = GroupMembership.GetByUserAndGroupID(new UserRegion() { UserID = member.UserID, RegionID = member.RegionID }, group.GroupID);
                //check for membership
                Assert.IsNotNull(memberMembership);
                //compare the membership notification
                Assert.AreEqual(memberMembership.NotificationPreference, preference);
            }

            [ClassCleanup]
            public static void CleanUp()
            {
                CleanUpDatabase(_rootClan);
            }
        }

        [TestClass]
        public class InvalidateExpiredInvitations : ClansContext
        {
            private static Group _clanRoot;
            private static GroupInvitation _expiredInvite;
            private static GroupInvitation _activeInvite;

            private static UserAndRegion _guestWithExpiredInvite;
            private static UserAndRegion _memberWithExpiredInvite;
            private static UserAndRegion _guestWithActiveInvite;

            [ClassInitialize]
            public static void Init(TestContext context)
            {
                try
                {
                    _clanRoot = CreateRootGroup("ProcessExpiredInvitations Test Root", GroupType.Large);

                    // Create Invitations and add members
                    _expiredInvite = _clanRoot.CreateGroupInvitation(Creator.UserID, true, TimeSpan.FromDays(1));
                    _activeInvite = _clanRoot.CreateGroupInvitation(Creator.UserID, true, TimeSpan.FromDays(2));

                    // Add some guests
                    _guestWithExpiredInvite = CreateUser(LowestReservedUserID - 1, "ExpiredGuest");
                    _memberWithExpiredInvite = CreateUser(LowestReservedUserID - 2, "ExpiredMember");
                    _guestWithActiveInvite = CreateUser(LowestReservedUserID - 3, "ActiveGuest");

                    ClaimInvitation(_clanRoot, _expiredInvite, _guestWithExpiredInvite.User,
                        _guestWithExpiredInvite.Region);
                    ClaimInvitation(_clanRoot, _expiredInvite, _memberWithExpiredInvite.User,
                        _memberWithExpiredInvite.Region);
                    ClaimInvitation(_clanRoot, _activeInvite, _guestWithActiveInvite.User, _guestWithActiveInvite.Region);
                    
                    // Promote the member
                    ChangeUserRole(_clanRoot, Creator.UserID, LegacyGroupRole.Member, _memberWithExpiredInvite.User.UserID);

                    // Expire the invite
                    _expiredInvite.DateCreated = DateTime.UtcNow.AddDays(-1);
                    _expiredInvite.DateExpires = DateTime.UtcNow.AddHours(-1);
                    _expiredInvite.Update(p => p.DateCreated, p => p.DateExpires);

                    // Purge Guests
                    _clanRoot.InvalidateExpiredInvitations();
                }
                catch
                {
                    CleanUp();
                    throw;
                }
            }

            [TestMethod]
            public void Removes_Guests_With_Expired_Invite()
            {
                Assert.IsNull(Group.GetLocal(_clanRoot.GroupID).MemberList.Members.Find(_guestWithExpiredInvite.User.UserID));
            }

            [TestMethod]
            public void Ignores_Members_With_Expired_Invite()
            {
                Assert.IsNotNull(Group.GetLocal(_clanRoot.GroupID).MemberList.Members.Find(_memberWithExpiredInvite.User.UserID));
            }

            [TestMethod]
            public void Ignores_Guests_With_Active_Invite()
            {
                Assert.IsNotNull(Group.GetLocal(_clanRoot.GroupID).MemberList.Members.Find(_guestWithActiveInvite.User.UserID));
            }

            private static UserAndRegion CreateUser(int userID, string userName)
            {
                return new UserAndRegion
                {
                    User = new User
                    {
                        UserID = userID,
                        Username = userName
                    },
                    Region = new UserRegion
                    {
                        UserID = userID,
                        RegionID = 1
                    }
                };
            }

            private struct UserAndRegion
            {
                public User User { get; set; }
                public UserRegion Region { get; set; }
            }

            [ClassCleanup]
            public static void CleanUp()
            {
                if (_expiredInvite != null)
                {
                    _expiredInvite.Delete();
                }
                if (_activeInvite != null)
                {
                    _activeInvite.Delete();
                }
                CleanUpDatabase(_clanRoot);
            }
        }

        [TestClass]
        public class PromoteGuestToMember : ClansContext
        {
            private static Group _clanRoot;
            private static GroupInvitation _invitation;

            [ClassInitialize]
            public static void Init(TestContext ctx)
            {
                try
                {
                    _clanRoot = CreateRootGroup("PromoteGuestToMember Test Root", GroupType.Large);
                    _invitation = _clanRoot.CreateGroupInvitation(Creator.UserID, false, null);

                    var user = new User
                    {
                        UserID = Guest.UserID,
                        Username = Guest.Username,
                    };
                    var region = new UserRegion
                    {
                        UserID = Guest.UserID,
                        RegionID = Guest.RegionID
                    };
                    ClaimInvitation(_clanRoot, _invitation, user, region);

                    ChangeUserRole(_clanRoot, Creator.UserID, LegacyGroupRole.Member, Guest.UserID);
                }
                catch
                {
                    CleanUp();
                    throw;
                }
            }

            [TestMethod]
            public void User_Is_Promoted_To_Member()
            {
                Assert.AreEqual(LegacyGroupRole.Member, Group.GetLocal(_clanRoot.GroupID).MemberList.Members.Find(Guest.UserID).Role);
            }

            [TestMethod]
            public void User_No_Longer_Has_Invite_Code()
            {
                Assert.IsNull(Group.GetLocal(_clanRoot.GroupID).MemberList.Members.Find(Guest.UserID).InviteCode);
            }

            [ClassCleanup]
            public static void CleanUp()
            {
                if (_invitation != null)
                {
                    _invitation.Delete();
                }
                CleanUpDatabase(_clanRoot);
            }
        }

        [TestClass]
        public class LeaveClan : ClansContext
        {
            private static Group _clanRoot;

            [ClassInitialize]
            public static void Init(TestContext ctx)
            {
                try
                {
                    _clanRoot = CreateRootGroup("LeaveClan Test Clan", GroupType.Large);
                    _clanRoot.Leave(Member.UserID, Member.RegionID);
                }
                catch
                {
                    CleanUp();
                    throw;
                }
            }

            [TestMethod]
            public void Logs_User_Left_Event()
            {
                Assert.IsTrue(GroupEventLog.GetLocal(_clanRoot.GroupID).Events.GetValues()
                    .Any(e => e.Type == GroupEventType.UserLeft && e.InitiatingUserID == Member.UserID));
            }

            [ClassCleanup]
            public static void CleanUp()
            {
                CleanUpDatabase(_clanRoot);
            }
        }

        [TestClass]
        public class TestAddRemoveGuests : ClansContext
        {
            private static Group _clanRoot;
            
            [ClassInitialize]
            public static void Initialize(TestContext context)
            {
                _clanRoot = CreateRootGroup("Test add remove guest", GroupType.Large);
                //Add a guest by member
                AddMembers(_clanRoot, Member.UserID, Guest);
            }

            [ClassCleanup]
            public static void CleanUp()
            {
                CleanUpDatabase(_clanRoot);
            }

            [TestMethod]
            public void TestGuestMembershipExists()
            {
                //get
                TryWaitForDatabase(() =>
                {
                    var membership =
                        GroupMembership.GetByUserAndGroupID(
                            new UserRegion {UserID = Guest.UserID, RegionID = Guest.RegionID}, _clanRoot.GroupID);
                    return membership != null && membership.GroupID == _clanRoot.GroupID;
                });
                var groupMembership =
                    GroupMembership.GetByUserAndGroupID(
                        new UserRegion() {UserID = Guest.UserID, RegionID = Guest.RegionID}, _clanRoot.GroupID);
                Assert.IsNotNull(groupMembership);
                //check if the guest is in groupmemberlist
                Assert.IsTrue(_clanRoot.MemberList.Members.Exists(Guest.UserID));
            }
        }
    }
}
