﻿using System;
using System.Linq;
using Curse.Friends.Data;
using Curse.Friends.Enums;
using Curse.Friends.WebService.Tests.Utilities;
using Microsoft.VisualStudio.TestTools.UnitTesting;

namespace Curse.Friends.WebService.Tests.Friends
{
    public static class FriendshipTests
    {
        [TestClass]
        public class TestFriendRequest : FriendContext
        {
            private static TestUser _user1, _user2;
            private static string _message;
            private static Friendship _friendship1, _friendship2;

            [ClassInitialize]
            public static void Initialize(TestContext context)
            {
                try
                {
                    _user1 = CreateUserandUserRegion(1, "Test User1 " + DateTime.UtcNow);
                    _user2 = CreateUserandUserRegion(1, "Test User1 " + DateTime.UtcNow, _user1.UserID);

                    _message = "Friendship Create test " + DateTime.UtcNow;
                    
                    //create friednship request from user1 to user2
                    var friendship = CreateFriendship(_user1.User, _user1.Region, _user2.User, _user2.Region, _message);
                    _friendship1 = friendship.Item1;
                    _friendship2 = friendship.Item2;
                }
                catch (Exception ex)
                {
                    CleanUp();
                    Assert.Fail(ex.Message);
                }
            }

            /// <summary>
            /// Test to see if the friendships are created for both users in database
            /// </summary>
            [TestMethod]
            public void Test_FriendhipStatus()
            {   
                //check friendship status
                CheckFriendshipStatus(_user1, _user2.UserID, FriendshipStatus.AwaitingThem);
                CheckFriendshipStatus(_user2, _user1.UserID, FriendshipStatus.AwaitingMe);
            }

            /// <summary>
            /// Test if the connectionstatus are offline for unconfirmed requests
            /// </summary>
            [TestMethod]
            public void Test_FriendshipConnectionStatus()
            {
                Assert.AreEqual(_friendship1.OtherUserConnectionStatus, UserConnectionStatus.Offline);
                Assert.AreEqual(_friendship2.OtherUserConnectionStatus, UserConnectionStatus.Offline);
            }

            /// <summary>
            /// Test other regionId is stored (although its same for both regions)
            /// </summary>
            [TestMethod]
            public void CheckForOtherRegionID()
            {
                Assert.AreEqual(_friendship1.OtherUserRegionID, _user2.RegionID);
                Assert.AreEqual(_friendship2.OtherUserRegionID, _user1.RegionID);
            }

            [TestMethod]
            public void Test_InvitationMessageProperlySet()
            {
                var msg = _friendship1.InvitationMessage;
                if (!string.IsNullOrEmpty(msg))
                {
                    Assert.Fail("Invitation Message should be empty for requestor friendship");
                }
                Assert.AreEqual(_friendship2.InvitationMessage, _message);
            }

            [ClassCleanup]
            public static void CleanUp()
            {   
                CleanupFriendsDatabase(new[] { _user1, _user2 });
            }
        }

        [TestClass]
        public class TestConfirmFriendship : FriendContext
        {   
            private static string _message;
            private static TestUser _user1, _user2, _user3;

            [ClassInitialize]
            public static void Initialize(TestContext context)
            {
                try
                {
                    _user1 = CreateUserandUserRegion(1, "user1 " + DateTime.UtcNow);
                    _user2 = CreateUserandUserRegion(1, "user2 " + DateTime.UtcNow, _user1.UserID);
                    _user3 = CreateUserandUserRegion(1, "user2 " + DateTime.UtcNow, _user2.UserID);

                    _message = "Friendship Confirm test " + DateTime.UtcNow;

                    //create req from user1 to user2
                    var friendship12 = CreateFriendship(_user1.User, _user1.Region, _user2.User, _user2.Region, _message);
                    //create req from user1 to user3
                    var friendship13 = CreateFriendship(_user1.User, _user1.Region, _user3.User, _user3.Region, _message);
                    
                    
                    //confirm request by user2
                    Friendship.Confirm(_user2.User, _user2.Region, friendship12.Item2, _user1.User, _user1.Region, friendship12.Item1);
                    //confirm request by user3
                    Friendship.Confirm(_user3.User, _user3.Region, friendship13.Item2, _user1.User, _user1.Region, friendship13.Item1);
                }
                catch (Exception ex)
                {
                    CleanUp();
                    Assert.Fail(ex.Message);
                }
            }

            [TestMethod]
            public void TestFriendshipStatus()
            {
                CheckFriendshipStatus(_user1, _user2.UserID, FriendshipStatus.Confirmed);
                CheckFriendshipStatus(_user2, _user1.UserID, FriendshipStatus.Confirmed);
                CheckFriendshipStatus(_user1, _user3.UserID, FriendshipStatus.Confirmed);
                CheckFriendshipStatus(_user3, _user1.UserID, FriendshipStatus.Confirmed);
            }

            [TestMethod]
            public void TestFriendsCount()
            {
                Assert.AreEqual(_user1.User.FriendCount, 2);
                Assert.AreEqual(_user2.User.FriendCount, 1);
                Assert.AreEqual(_user3.User.FriendCount, 1);
            }

            [TestMethod]
            public void TestRenameFriend()
            {
                var alias = "New user1 name";
                //user2 changes user1 name
                Friendship.UpdateNickname(_user2.RegionID, _user2.UserID, _user1.UserID, alias);
                var friendship = Friendship.Get(_user2.RegionID, _user2.UserID, _user1.UserID);
                Assert.AreEqual(friendship.DisplayName, alias);

                alias = "New user3 name";
                Friendship.UpdateNickname(_user2.RegionID, _user2.UserID, _user1.UserID, alias);
                friendship = Friendship.Get(_user2.RegionID, _user2.UserID, _user1.UserID);
                Assert.AreEqual(friendship.DisplayName, alias);
            }

            /// <summary>
            /// make the user favorite
            /// </summary>
            [TestMethod]
            public void TestFavoriteFriend()
            {
                //user2 makes user1 as favorite user
                Friendship.UpdateIsFavorite(_user2.RegionID, _user2.UserID, _user1.UserID, true);
                var friendship = Friendship.Get(_user2.RegionID, _user2.UserID, _user1.UserID);
                Assert.IsTrue(friendship.IsFavorite);

                //user1 makes user3 favorite user
                Friendship.UpdateIsFavorite(_user1.RegionID, _user1.UserID, _user3.UserID, true);
                friendship = Friendship.Get(_user1.RegionID, _user1.UserID, _user3.UserID);
                Assert.IsTrue(friendship.IsFavorite);

                //user2 should not be favorite of user1
                friendship = Friendship.Get(_user1.RegionID, _user1.UserID, _user2.UserID);
                Assert.AreNotEqual(friendship.IsFavorite, true);

                //user1 unfavorites user3
                Friendship.UpdateIsFavorite(_user1.RegionID, _user1.UserID, _user3.UserID, false);
                friendship = Friendship.Get(_user1.RegionID, _user1.UserID, _user3.UserID);
                Assert.IsFalse(friendship.IsFavorite);
            }

            [ClassCleanup]
            public static void CleanUp()
            {   
                CleanupFriendsDatabase(new[] { _user1, _user2, _user3 });
            }
        }

        [TestClass]
        public class TestDeclineFriendship : FriendContext
        {   
            private static string _message;
            private static TestUser _user1, _user2, _user3;

            [ClassInitialize]
            public static void Initialize(TestContext context)
            {
                try
                {
                    _user1 = CreateUserandUserRegion(1, "Test User1 " + DateTime.UtcNow);
                    _user2 = CreateUserandUserRegion(1, "Test User2 " + DateTime.UtcNow, _user1.UserID);
                    _user3 = CreateUserandUserRegion(1, "Test user3 " + DateTime.UtcNow, _user2.UserID);
                    _message = "Friendship Decline test " + DateTime.UtcNow;

                    //create req from user1 to user2
                    var friendship12 = CreateFriendship(_user1.User, _user1.Region, _user2.User, _user2.Region, _message);
                    //create req from user1 to user3
                    var friendship13 = CreateFriendship(_user1.User, _user1.Region, _user3.User, _user3.Region, _message);
                    //user1 makes user2 favorite
                    Friendship.UpdateIsFavorite(_user1.RegionID, _user1.UserID, _user2.UserID, true);
                    //user2 permanently declines request from user1
                    Friendship.Remove(true, _user2.User, friendship12.Item2, _user2.Region, _user1.User, friendship12.Item1, _user1.Region);
                    //user3 temporarily declines friendshpi from user1
                    Friendship.Remove(false, _user3.User, friendship13.Item2, _user3.Region, _user1.User, friendship13.Item1, _user1.Region);
                }
                catch (Exception ex)
                {
                    CleanUp();
                    Assert.Fail(ex.Message);
                }
            }

            [TestMethod]
            public void TestFriendshipStatus()
            {
                //friendship of user1 and user2
                var friendship = Friendship.Get(_user1.RegionID, _user1.UserID, _user2.UserID);
                Assert.AreEqual(friendship.Status, FriendshipStatus.DeclinedByThem);
                //check if the user2 is still a favorite friend for user1
                Assert.IsTrue(friendship.IsFavorite);

                //friendship of user2 and user1
                friendship = Friendship.Get(_user2.RegionID, _user2.UserID, _user1.UserID);
                Assert.AreEqual(friendship.Status, FriendshipStatus.DeclinedByMe);

                //there should not be ay friendhsip existing between user1 and user3
                friendship = Friendship.Get(_user3.RegionID, _user3.UserID, _user1.UserID);
                Assert.IsNull(friendship);
                friendship = Friendship.Get(_user1.RegionID, _user1.UserID, _user3.UserID);
                Assert.IsNull(friendship);
            }

            [TestMethod]
            public void TestFriendsCount()
            {
                Assert.AreEqual(_user1.User.FriendCount, 0);
                Assert.AreEqual(_user2.User.FriendCount, 0);
                Assert.AreEqual(_user3.User.FriendCount, 0);
            }

            [ClassCleanup]
            public static void CleanUp()
            {
                CleanupFriendsDatabase(new[] { _user1, _user2, _user3 });
            }

        }

        /// <summary>
        /// test case to check if the user declines friendsship request and accepts new one
        /// </summary>
        [TestClass]
        public class TestDeclineandAddFriendship : FriendContext
        {
            private static TestUser _user1, _user2;
            private static string _message;
            private static Tuple<Friendship, Friendship> _friendship;
            
            [ClassInitialize]
            public static void Initialize(TestContext context)
            {
                try
                {
                    _user1 = CreateUserandUserRegion(1, "Test User1 " + DateTime.UtcNow);
                    _user2 = CreateUserandUserRegion(1, "Test User2 " + DateTime.UtcNow, _user1.UserID);
                    
                    _message = "Friendship Decline and confirm test " + DateTime.UtcNow;

                    //create req from user1 to user2
                    _friendship = CreateFriendship(_user1.User, _user1.Region, _user2.User, _user2.Region, _message);
                    //user2 declines request from user1
                    Friendship.Remove(false, _user2.User, _friendship.Item2, _user2.Region, _user1.User, _friendship.Item1, _user1.Region);
                    //user1 sends request to user2 again
                    _friendship = CreateFriendship(_user1.User, _user1.Region, _user2.User, _user2.Region, _message);
                    //user2 accepts friendship from user1
                    Friendship.Confirm(_user2.User, _user2.Region, _friendship.Item2, _user1.User, _user1.Region, _friendship.Item1);
                }
                catch (Exception ex)
                {
                    CleanUp();
                    Assert.Fail(ex.Message);
                }
            }

            [TestMethod]
            public void TestFriendshipStatus()
            {
                CheckFriendshipStatus(_user1, _user2.UserID, FriendshipStatus.Confirmed);
                CheckFriendshipStatus(_user2, _user1.UserID, FriendshipStatus.Confirmed);
            }

            [TestMethod]
            public void TestFriendsCount()
            {
                Assert.AreEqual(_user1.User.FriendCount, 1);
                Assert.AreEqual(_user2.User.FriendCount, 1);
            }

            [ClassCleanup]
            public static void CleanUp()
            {
                CleanupFriendsDatabase(new[] { _user1, _user2 });
            }
        }
    }
}
