﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading;
using Curse.Friends.Data;
using Curse.Friends.Data.DerivedModels;
using Curse.Friends.Enums;
using Curse.Friends.NotificationContracts;
using Curse.SocketMessages;

namespace Curse.Friends.WebService.Tests.Groups
{
    public abstract class GroupsContext : BaseContext
    {
        protected const int GroupNotificationTimeoutMillis = 2000;

        public static readonly NewGroupMember Creator = new NewGroupMember(NotificationClientUserID, 1,
            "ClanTestsCreator", LegacyGroupRole.Owner);

        public static readonly NewGroupMember Officer = new NewGroupMember(Creator.UserID - 1, 1, "ClanTestsOfficer",
            LegacyGroupRole.Officer);

        public static readonly NewGroupMember Member = new NewGroupMember(Officer.UserID - 1, 1, "ClanTestsMember",
            LegacyGroupRole.Member);

        public static readonly NewGroupMember Admin = new NewGroupMember(Member.UserID - 1, 1, "ClanTestsAdmin",
            LegacyGroupRole.Admin);

        public static readonly NewGroupMember Guest = new NewGroupMember(Admin.UserID - 1, 1, "ClanTestsGuest",
            LegacyGroupRole.Guest);

        public static readonly NewGroupMember NewMember = new NewGroupMember(Guest.UserID - 1, 1, "ClanTestsNewMember",
            LegacyGroupRole.Member);

        public static readonly NewGroupMember PeerOfficer = new NewGroupMember(NewMember.UserID - 1, 1,
            "ClanTestsPeerOfficer", LegacyGroupRole.Officer);

        public static readonly NewGroupMember[] DefaultMembers = {Creator, Officer, Member, Admin, PeerOfficer};

        protected static int LowestReservedUserID = PeerOfficer.UserID;

        public static void CleanUpDatabase(params Group[] groups)
        {
            foreach (var @group in groups)
            {
                if (@group != null && @group.IsRootGroup && Group.GetLocal(@group.GroupID) != null)
                {
                    // root group that is still in the database
                    @group.DeleteRoot(Creator.UserID);
                    //clean up all memberships and memberList if any for the group
                    var memberships = GroupMembership.GetAllLocal(p => p.RootGroupID, @group.GroupID);
                    foreach (var membership in memberships)
                    {
                        membership.Delete();
                    }
                    var memberlists = GroupMemberList.GetAllLocal(p => p.GroupID, @group.GroupID);
                    foreach (var memberlist in memberlists)
                    {
                        memberlist.Delete();
                    }
                }
            }
            
        }

        protected static void ExecuteGroupChangeWithNotificationTimeout(Action action,
            Func<GroupChangeNotification, bool> validityCheck, string errorMessage)
        {
            ExecuteGroupChangeWithNotificationTimeout(
                () =>
                {
                    action();
                    return true;
                },
                validityCheck, errorMessage);
        }

        protected static T ExecuteGroupChangeWithNotificationTimeout<T>(Func<T> func,
            Func<GroupChangeNotification, bool> validityCheck,
            string errorMessage)
        {
            var mre = new ManualResetEvent(false);
            EventHandler<EventArgs<GroupChangeNotification>> handler = (o, e) =>
            {
                if (validityCheck(e.Value))
                {
                    mre.Set();
                }
            };

            NotificationClient.GroupChanged += handler;
            var result = func();
            var success = mre.WaitOne(GroupNotificationTimeoutMillis);
            NotificationClient.GroupChanged -= handler;

            if (!success)
            {
                throw new TimeoutException(
                    string.Format("Expected notification was not received within {0} ms. Message: {1}",
                        GroupNotificationTimeoutMillis, errorMessage));
            }

            return result;
        }

        protected static Group CreateRootGroup(string title, GroupType type, string lobbyName=null)
        {
            var timestampedTitle = title + DateTime.UtcNow;

            lobbyName = lobbyName ?? (title + " Lobby");
            var roles = Enum.GetValues(typeof (LegacyGroupRole))
                .Cast<LegacyGroupRole>()
                .Where(v => v > 0)
                .ToDictionary(r => r, r => Enum.GetName(typeof (LegacyGroupRole), r));

            var root = ExecuteGroupChangeWithNotificationTimeout(
                () => Group.CreateRootGroup(type, Creator.UserID, Creator.RegionID, Creator.Username,
                    timestampedTitle, null, DefaultMembers.Where(m => m.UserID != Creator.UserID).ToArray(),
                    lobbyName,roles),

                n => n.ChangeType == GroupChangeType.CreateGroup &&
                     n.Group.GroupTitle == timestampedTitle,

                string.Format("Group {0} was not created.", timestampedTitle));

            TryWaitForDatabase(() => GroupMembership.GetAllLocal(p => p.GroupID, root.GroupID).Any());

            return root;
        }

        protected static void DeleteRootGroup(Group clan, int requestorUserID)
        {
            ExecuteGroupChangeWithNotificationTimeout(
                () => clan.DeleteRoot(requestorUserID),
                n => n.ChangeType == GroupChangeType.RemoveGroup &&
                     n.Group.GroupID == clan.GroupID,
                string.Format("Group {0} was not deleted.", clan.Title));
            TryWaitForDatabase(() => !GroupMembership.GetAllLocal(p => p.RootGroupID, clan.GroupID).Any());
        }

        protected static void AddMembers(Group channel, int callerUserID, params NewGroupMember[] newMembers)
        {
            ExecuteGroupChangeWithNotificationTimeout(
                () => channel.RootGroup.AddUsers(callerUserID, newMembers),

                n => n.ChangeType == GroupChangeType.AddUsers &&
                     newMembers.All(m => n.Members.Any(u => u.UserID == m.UserID)),

                "The new members were not added.");
        }

        protected static void RemoveMembers(Group channel, int callerUserId, params int[] userIdsToRemove)
        {
            ExecuteGroupChangeWithNotificationTimeout(
                () => channel.RootGroup.RemoveUsers(callerUserId, new HashSet<int>(userIdsToRemove)),

                n => n.ChangeType == GroupChangeType.RemoveUsers &&
                     userIdsToRemove.All(id => n.Members.Any(u => u.UserID == id)),

                "The members were not removed.");
        }

        protected static void ChangeInfo(Group group, int userID, string title, string avatarUrl, string motd,
            bool? allowTempChildren, bool? forcePTT)
        {
            ExecuteGroupChangeWithNotificationTimeout(
                () => group.ChangeInfo(userID, title, avatarUrl, motd, allowTempChildren, forcePTT),

                n => n.ChangeType == GroupChangeType.ChangeInfo,

                string.Format("Group {0}'s information was not changed.", group.Title));
        }



        protected static void SendGroupMessage(Group group, string message)
        {
            var mre = new ManualResetEvent(false);
            EventHandler<EventArgs<GroupMessageNotification>> handler = (o, e) =>
            {
                if (e.Value.GroupID == group.GroupID && e.Value.Message == message)
                {
                    mre.Set();
                }
            };

            NotificationClient.GroupMessaged += handler;
            NotificationClient.SendGroupMessage(group.GroupID, message);
            var success = mre.WaitOne(GroupNotificationTimeoutMillis);
            NotificationClient.GroupMessaged -= handler;

            if (!success)
            {
                throw new TimeoutException(string.Format("Group message notification was not received within {0} ms.",
                    GroupNotificationTimeoutMillis));
            }

            TryWaitForDatabase(
                () => Conversation.GetByTypeAndID(ConversationType.Group, group.GroupID.ToString()) != null);
        }

        protected static void UpgradeTemporaryChannel(Group group, NewGroupMember requestor = null)
        {
            requestor = requestor ?? Admin;
            ExecuteGroupChangeWithNotificationTimeout(
                () => group.UpgradeToPermanent(requestor.UserID),

                n => n.ChangeType == GroupChangeType.ChangeInfo,

                string.Format("Group {0}'s information was not changed.", group.Title));
        }
    }
}
