﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Net;
using System.Security.Authentication;
using System.Security.Cryptography;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using Curse.Friends.Data;
using Curse.Friends.Enums;
using Curse.Friends.NotificationContracts;
using Curse.Logging;
using Curse.SocketInterface;
using Curse.SocketMessages;

namespace Curse.Friends.WebService.Tests.Utilities
{
    public class NotificationClient : ClientSocketInterface
    {
        private readonly Dictionary<int, IContractDispatcher> _contractDispatchers =
            new Dictionary<int, IContractDispatcher>();

        private static NotificationClient _instance;
        private static ClientEndpoint _endpoint;
        private static UserRegion _userRegion;
        private static User _user;

        private static ManualResetEvent _joinMre;
        private static JoinResponse _joinResponse;

        public static NotificationClient Connect(int userID, string machineKey)
        {
            if (_instance == null)
            {
                var sessionID = Guid.NewGuid().ToString(); 
                
                _endpoint = new ClientEndpoint
                {
                    UserID = userID,
                    MachineKey = machineKey,
                    ServerName = Environment.MachineName,
                    SessionID =  sessionID
                };
                _endpoint.InsertLocal();

                _userRegion = new UserRegion
                {
                    UserID = userID,
                    RegionID = 1
                };
                _userRegion.InsertLocal();

                _user = new User
                {
                    UserID = userID,
                    ConnectionStatus = UserConnectionStatus.Online,
                };
                _user.InsertLocal();

                _instance = new NotificationClient(IPAddress.Loopback, 443, ClientSocketOptions.TcpOnly);
                if (!_instance.Connect() || !_instance.Join(machineKey, userID, sessionID))
                {
                    Disconnect();
                }
            }

            return _instance;
        }

        private bool Join(string machineKey, int userID, string sessionID)
        {
            bool joinSuccessful = false;

            try
            {
                using (var rsa = CryptoHelper.NewRsaProvider())
                {
                    var encryptionAlgorithm = new AesCryptoServiceProvider();

                    var joinRequest = new JoinRequest
                    {
                        SessionID = sessionID,
                        MachineKey = machineKey,
                        UserID = userID,
                        Status = UserConnectionStatus.Online,
                        ClientVersion = "6.1.12345.54321",
                        PublicKey = rsa.ExportPublicKey(),
                        CipherAlgorithm = CipherAlgorithmType.Aes,
                        CipherStrength = encryptionAlgorithm.BlockSize,
                    };

                    Logger.Info("Sending join session request...", joinRequest);
                    _joinMre = new ManualResetEvent(false);
                    SendContract(joinRequest);
                    _joinMre.WaitOne(2000);
                    // Wait up to 2 seconds for the join call to complete                                
                    _joinMre.Dispose();
                    _joinMre = null;

                    if (_joinResponse != null && _joinResponse.Status == JoinStatus.Successful)
                    {
                        CryptoHelper.SetSecret(encryptionAlgorithm, rsa, _joinResponse.EncryptedSessionKey);
                        Session = new ClientSession(sessionID, encryptionAlgorithm);

                        joinSuccessful = true;
                    }
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex);
            }
            finally
            {
                _joinMre = null;
            }

            return joinSuccessful;
        }

        public static void Disconnect()
        {
            if (_instance != null)
            {
                _instance.Disconnect(SocketDisconnectReason.UserInitiated);
                _instance = null;
            }
            if (_endpoint != null)
            {
                _endpoint.Delete();
                _endpoint = null;
            }
            if (_userRegion != null)
            {
                _userRegion.Delete();
                _userRegion = null;
            }
            if (_user != null)
            {
                _user.Delete();
                _user = null;
            }
        }

        private NotificationClient(IPAddress remoteAddress, int port, ClientSocketOptions options = ClientSocketOptions.TcpAndUdp) : base(remoteAddress, port, options)
        {
            MessageReceived += OnMessageReceived;
            AddContractDispatcher<GroupChangeNotification>((si, @event) => TriggerEvent(GroupChanged, @event));
            AddContractDispatcher<GroupMessageNotification>((si, @event) => TriggerEvent(GroupMessaged, @event));

            AddContractDispatcher<JoinResponse>((si, @event) =>
            {
                if (_joinMre != null)
                {
                    _joinResponse = @event;
                    _joinMre.Set();
                }
            });
        }

        private void AddContractDispatcher<T>(Action<ISocketInterface, T> handler) where T : Contract<T>, new()
        {
            int messageType = Contract<T>.MessageType;
            IContractDispatcher dispatcher = new ContractDispatcher<T>(handler);
            _contractDispatchers.Add(messageType, dispatcher);

        }

        private void OnMessageReceived(object sender, MessageEventArgs messageEventArgs)
        {
            if (!IsConnected)
            {
                return;
            }

            IContractDispatcher dispatcher = null;
            if (_contractDispatchers.TryGetValue(messageEventArgs.Message.Header.MessageType, out dispatcher))
            {
                try
                {
                    dispatcher.Dispatch(this, messageEventArgs.Message);
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Dispatcher callback failed for message type: " + dispatcher.TypeName);
                }

            }
        }

        public event EventHandler<EventArgs<GroupChangeNotification>> GroupChanged;
        public event EventHandler<EventArgs<GroupMessageNotification>> GroupMessaged;

        private void TriggerEvent<T>(EventHandler<EventArgs<T>> handler, T args)
        {
            if (handler != null)
            {
                handler(this, new EventArgs<T>(args));
            }
        }

        public void SendGroupMessage(Guid groupID, string message)
        {
            SendContract(new GroupMessageRequest
            {
                Message = message,
                GroupID = groupID,
                ClientID = Guid.NewGuid()
            });
        }
    }
}
