﻿using Curse.Aerospike;
using Curse.CloudSearch;
using Curse.CloudServices;
using Curse.CloudServices.Authentication;
using Curse.CloudServices.Models;
using Curse.Friends.Configuration;
using Curse.Friends.Configuration.CurseVoiceService;
using Curse.Friends.Data;
using Curse.Friends.Data.DerivedModels;
using Curse.Friends.Data.Queues;
using Curse.Friends.Data.Search;
using Curse.Friends.Enums;
using Curse.Friends.Statistics;
using Curse.Friends.WebService.Exceptions;
using Curse.Friends.WebService.Requests;
using Curse.Friends.WebService.Responses;
using Curse.Logging;
using Curse.ServiceEncryption;
using Curse.Voice.Helpers;
using Nest;
using Newtonsoft.Json;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Runtime.Caching;
using System.ServiceModel;
using Curse.Extensions;
using Curse.Friends.Data.Messaging;
using Curse.Friends.NotificationContracts;
using Curse.Logging.Uploader;

namespace Curse.Friends.WebService
{
    [ServiceBehavior(Namespace = "http://friends.cursevoice.com/", AddressFilterMode = AddressFilterMode.Any, ConcurrencyMode = ConcurrencyMode.Multiple, InstanceContextMode = InstanceContextMode.PerSession)]
    [FriendsServiceErrorHandler]
    public class FriendsService : IFriendsService
    {

        private static readonly ObjectCache TokenCache = MemoryCache.Default;

        private class CachedAuthTokenStorageProvider : IAuthTokenStorageProvider
        {
            private static string GetCacheKey(string token)
            {
                return "AuthToken:" + token;
            }

            public AuthenticationTokenData FindToken(string token)
            {
                var cacheKey = GetCacheKey(token);
                AuthenticationTokenData value = null;
                if (TokenCache.Contains(cacheKey))
                {
                    value = TokenCache.Get(cacheKey) as AuthenticationTokenData;
                }

                return value;
            }

            public void StoreToken(string token, AuthenticationTokenData data)
            {
                var cacheKey = GetCacheKey(token);
                TokenCache.Set(cacheKey, data, new DateTimeOffset(DateTime.UtcNow.AddMonths(1)));
            }
        }

        public static void Startup()
        {
            Logger.Init(new LoggerConfig(@"C:\Curse\CurseFriendsWebService\Logs"));

            Logger.Info("Service Starting...");

            try
            {
                LogUploader.Initialize((int)ServiceHostType.FriendsWebService, FriendsServiceConfiguration.Instance.CentralLogServiceUrl, FriendsServiceConfiguration.Instance.CentralServiceApiKey);
                FriendsStatsManager.Initialize((int)ServiceHostType.FriendsWebService, ServiceHostType.FriendsWebService.ToString(), HostCounter.All);
                FriendsStatsManager.BeginStartup();
                AuthenticationConfiguration.ApiKey = FriendsServiceConfiguration.Instance.NotificationServiceApiKey;
                AuthenticationConfiguration.TokenLifespan = TimeSpan.FromSeconds(FriendsServiceConfiguration.Instance.AuthTokenLifespanSeconds);
                AuthenticationConfiguration.SetTokenProvider(new CachedAuthTokenStorageProvider());
                EncryptionToken.Initialize(FriendsServiceConfiguration.Instance.EncryptionKey);
                StorageConfiguration.Initialize("FriendsWebService");
                NotificationHostManager.Initialize();
                Logger.Info("Service Started!");
                FriendsStatsManager.Started();
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to start service!");
                Logger.Flush();
            }
        }

        public static void Shutdown()
        {
            Logger.Info("Service Stopping...");
            FriendsStatsManager.BeginShutdown();
            StorageConfiguration.Shutdown();
            Logger.Info("Service Stopped!");
            FriendsStatsManager.Shutdown();
            LogUploader.Shutdown();
        }

        public BasicServiceResponse UploadFriendHints(string payload)
        {
            try
            {
                var authToken = AuthenticationContext.Current;
                var json = EncryptionProvider.Instance.Decrypt(payload, "password");
                var submittedHints = JsonConvert.DeserializeObject<FriendHintNotification[]>(json);

                if (!submittedHints.Any())
                {
                    return new BasicServiceResponse { Status = BasicServiceResponseStatus.Invalid };
                }

                var hints = submittedHints.Select(FriendHint.FromLegacyNotification).ToArray();

                foreach (var hint in hints)
                {
                    string validationReason;

                    if (!hint.Validate(out validationReason))
                    {
                        Logger.Warn("Uploaded friend hint failed validation: " + validationReason, hint);
                        continue;
                    }

                    if (hint.Type != FriendHintType.Game && hint.Type != FriendHintType.Platform)
                    {
                        Logger.Warn("Uploaded friend hint is not a suitable type: " + hint.Type, hint);
                        continue;
                    }

                    hint.UserID = authToken.UserID;

                    var friendHint = FriendHint.GetLocal(hint.GetGeneratedKey());
                    var trackStat = false;
                    if (friendHint == null) // This is a new hint
                    {
                        hint.InsertLocal();
                        trackStat = true;
                    }
                    else
                    {
                        friendHint.DisplayName = hint.DisplayName;

                        if (friendHint.Status != hint.Status)
                        {
                            trackStat = true;
                        }

                        friendHint.Status = hint.Status;
                        friendHint.Visibility = hint.Visibility;
                        friendHint.AvatarUrl = hint.AvatarUrl;
                        if (hint.Verification == FriendHintVerification.ClientObserved &&
                            friendHint.Verification < FriendHintVerification.ClientObserved)
                        {
                            friendHint.Verification = FriendHintVerification.ClientObserved;
                        }
                        friendHint.Update();
                    }

                    if (trackStat)
                    {
                        if (hint.Status == FriendHintStatus.Normal)
                        {
                            FriendsStatsManager.Current.IdentitiesCreated.Track(authToken.UserID);
                            if (hint.Type == FriendHintType.Platform)
                            {
                                FriendsStatsManager.Current.PlatformIdentitiesCreatedByType.Track((int)hint.Platform);
                            }
                        }
                        else
                        {
                            FriendsStatsManager.Current.IdentitiesDeclined.Track(authToken.UserID);
                            if (hint.Type == FriendHintType.Platform)
                            {
                                FriendsStatsManager.Current.PlatformIdentitiesDeclinedByType.Track((int)hint.Platform);
                            }
                        }
                    }
                }

                new FriendHintSearchIndexer { UserID = authToken.UserID }.Enqueue();
                return new BasicServiceResponse { Status = BasicServiceResponseStatus.Successful };
            }
            catch (UserNotFoundException ex)
            {
                Logger.Warn(ex, "[UploadFriendHints] Attempt made to upload friend hints from an unregistered user.");
                return new BasicServiceResponse { Status = BasicServiceResponseStatus.Forbidden };
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "[UploadFriendHints] Unhandled exception");
                return new BasicServiceResponse { Status = BasicServiceResponseStatus.Error };
            }

        }

        public AddFriendHintResponse AddFriendHint(FriendHintRequest request)
        {
            var response = new AddFriendHintResponse();

            try
            {
                if (request.Identity.Type != FriendHintType.Game && request.Identity.Type != FriendHintType.Platform)
                {
                    response.Status = AddFriendHintResponseStatus.Unauthorized;
                    return response;
                }

                if (!request.Validate())
                {
                    response.Status = AddFriendHintResponseStatus.Invalid;
                    return response;
                }

                var identity = FriendHint.FromLegacyNotification(request.Identity);

                identity.InsertLocal();

                new FriendHintSearchIndexer { UserID = request.Identity.UserID }.Enqueue();

                response.Status = AddFriendHintResponseStatus.Successful;

            }
            catch (Exception ex)
            {
                Logger.Error(ex, "[AddFriendHint] Unhandled exception");
                response.Status = AddFriendHintResponseStatus.Error;
            }

            return response;
        }

        public FriendSearchResponse FindFriends(FriendSearchRequest request)
        {
            if (!request.Validate())
            {
                return new FriendSearchResponse() { Status = FriendSearchStatus.Invalid };
            }

            request.QueryString = request.QueryString.ToLowerInvariant();

            // If the input contains an @ symbol, always perform an email search
            if (request.QueryString.Contains("@"))
            {
                var emailSearchClient = EmailSearchModel.GetClient();
                var emailResults = emailSearchClient.Search<EmailSearchModel>(s => s.Query(q => q.Match(m => m.OnField(p => p.EmailAddress).Query(request.QueryString))).Take(1));

                if (!emailResults.IsValid)
                {
                    return new FriendSearchResponse() { Status = FriendSearchStatus.Invalid };
                }

                if (emailResults.Total > 0)
                {

                    var searchResult = CloudSearchResult<EmailSearchModel>.CreateFromQuery(emailResults);

                    return new FriendSearchResponse()
                    {
                        Status = FriendSearchStatus.Successful,
                        EmailMatches = searchResult,
                        ElapsedMilliseconds = emailResults.ElapsedMilliseconds
                    };
                }

                return new FriendSearchResponse()
                {
                    Status = FriendSearchStatus.Successful,
                    EmailMatches = CloudSearchResult<EmailSearchModel>.EmptyList,
                    CharacterMatches = CloudSearchResult<CharacterSearchModel>.EmptyList,
                    PlatformMatches = CloudSearchResult<PlatformSearchModel>.EmptyList,
                    UserMatches = CloudSearchResult<UsernameSearchModel>.EmptyList
                };
            }


            // User Results
            var usernameSearchClient = UsernameSearchModel.GetClient();
            var userResults = usernameSearchClient.Search<UsernameSearchModel>(s =>
                s.Query(q =>
                    q.Term(t =>
                        t.Username.Suffix("autocomplete"), request.QueryString)).Take(5));

            if (!userResults.IsValid)
            {
                return new FriendSearchResponse() { Status = FriendSearchStatus.Invalid };
            }

            // PLatform searches
            var platformSearchClient = PlatformSearchModel.GetClient();

            var platformResults = platformSearchClient.Search<PlatformSearchModel>(s => s.Query(q =>
                q.Bool(b => b.Must(qd => qd.Term(t => t.SearchTerm.Suffix("lowercase"), request.QueryString)))
                && q.Bool(b => b.Must(qd => qd.Term(t => t.Visibility, FriendHintVisibility.VisibleToEveryone)))));


            if (!platformResults.IsValid)
            {
                return new FriendSearchResponse() { Status = FriendSearchStatus.Invalid };
            }

            var characterResults = GetCharacterSearchResults(AuthenticationContext.Current.UserID, request.QueryString);

            if (!characterResults.IsValid)
            {
                return new FriendSearchResponse() { Status = FriendSearchStatus.Invalid };
            }

            var resp = new FriendSearchResponse
            {
                Status = FriendSearchStatus.Successful,
                CharacterMatches = CloudSearchResult<CharacterSearchModel>.CreateFromQuery(characterResults),
                UserMatches = CloudSearchResult<UsernameSearchModel>.CreateFromQuery(userResults),
                PlatformMatches = CloudSearchResult<PlatformSearchModel>.CreateFromQuery(platformResults),
                EmailMatches = CloudSearchResult<EmailSearchModel>.EmptyList,
            };

            resp.AddSearchTime(userResults.ElapsedMilliseconds);
            resp.AddSearchTime(characterResults.ElapsedMilliseconds);

            FriendsStatsManager.Current.SearchesPerformed.Track();

            return resp;
        }

        static ISearchResponse<CharacterSearchModel> GetCharacterSearchResults(int userID, string queryString)
        {

            // Remove any invalid characters
            queryString = queryString.Replace(" ", string.Empty).Replace("_", string.Empty);

            // Get the current user's character search hints
            var searchHints = FriendHint.GetAllLocal(p => p.UserID, userID);
            var characterHints = searchHints.Where(p => p.Type == FriendHintType.Game).ToArray();
            var distinctRegions = characterHints.Select(p => p.Region).Distinct().ToArray();
            var distinctServers = characterHints.Select(p => p.Server).Distinct().ToArray();

            var shouldFilters = new List<Func<QueryDescriptor<CharacterSearchModel>, QueryContainer>>();

            foreach (var region in distinctRegions)
            {
                shouldFilters.Add(qd => qd.Term(t => t.ServerRegion, region, 2.0));
            }

            shouldFilters.Add(qd => qd.Terms(t => t.ServerName, distinctServers));

            var characterClient = CharacterSearchModel.GetClient();

            return characterClient.Search<CharacterSearchModel>(s => s
              .Query(q => q
                 .Bool(b => b
                     .Must(
                          qd => qd.Term(t => t.SearchTerm.Suffix("autocomplete"), queryString)
                      )
                     .Should(shouldFilters.ToArray())
                     .MinimumShouldMatch(0)
                     .Boost(2.0)
                  )
              )
              .Take(10)
            );
        }

        public GetSelfHintsResponse GetSelfHints()
        {
            var response = new GetSelfHintsResponse();

            try
            {
                var authToken = AuthenticationContext.Current;
                var hints = FriendHint.GetAllLocal(p => p.UserID, authToken.UserID);
                return new GetSelfHintsResponse
                {
                    Status = GetSelfHintsResponseStatus.Successful,
                    Hints = hints.Where(p => p.Type == FriendHintType.Game || p.Type == FriendHintType.Platform).ToArray()
                };
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "[GetSelfHints] Unhandled exception");
                response.Status = GetSelfHintsResponseStatus.Error;
            }

            return response;
        }

        public GetMyFriendsResponse GetMyFriends()
        {
            try
            {
                var authToken = AuthenticationContext.Current;
                var userAndRegion = GetCurrentUserAndRegion();

                var myFriends = Friendship.GetAll(userAndRegion.Region.RegionID, p => p.UserID, authToken.UserID).Where(p => p.Status != FriendshipStatus.DeclinedByThem).ToArray();
                var userStats = UserStatistics.GetDictionaryByUserIDs(myFriends.Select(p => p.OtherUserID));
                
                foreach (var friend in myFriends)
                {
                    var friendStats = userStats.GetValueOrDefault(friend.OtherUserID);
                    if (friend.Status == FriendshipStatus.Confirmed)
                    {
                        // A shim for the legacy API
                        friend.OtherUserAvatarUrl = GetNewFriendAvatarUrl(friend.OtherUserID);                        
                        
                        if (friendStats != null && friendStats.ConnectStatusTimestamp > 0)
                        {
                            friend.OtherUserConnectionStatus = friendStats.ConnectionStatus;
                            friend.OtherUserStatusTimestamp = friendStats.ConnectStatusTimestamp.FromEpochMilliconds();
                            friend.OtherUserGameID = friendStats.CurrentGameID;
                            friend.OtherUserGameStatusMessage = friendStats.StatusMessage;
                            friend.OtherUserGameState = friendStats.GameState;
                        }
                    }                   
                    friend.DateRead = friend.DateRead.RoundToMillisecond().AddMilliseconds(100); // This is a hack
                }

                return new GetMyFriendsResponse() { Status = GetMyFriendsStatus.Successful, Friends = myFriends };

            }
            catch (UserNotFoundException)
            {
                return new GetMyFriendsResponse() { Status = GetMyFriendsStatus.Unauthorized };
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "[GetMyFriends] Unhandled exception");
                return new GetMyFriendsResponse() { Status = GetMyFriendsStatus.Error };
            }
        }

        public GetMyFriendSuggestionsResponse GetMyFriendSuggestions()
        {
            try
            {
                var userAndRegion = GetCurrentUserAndRegion();
                var activeSuggestions = FriendSuggestion.GetAllLocal(fs => fs.UserID, userAndRegion.User.UserID)
                    .Where(fs => fs.Status == FriendSuggestionStatus.Pending).Select(p => p.ToNotification()).ToArray();

                return new GetMyFriendSuggestionsResponse
                {
                    Status = GetMyFriendSuggestionsStatus.Successful,
                    Suggestions = activeSuggestions
                };
            }
            catch (UserNotFoundException)
            {
                return new GetMyFriendSuggestionsResponse { Status = GetMyFriendSuggestionsStatus.Unauthorized };
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "[GetMyFriendSuggestions] Unhandled exception");
                return new GetMyFriendSuggestionsResponse { Status = GetMyFriendSuggestionsStatus.Error };
            }
        }

        public GetFriendsOfFriendResponse GetFriendsOfFriend(int friendID)
        {
            try
            {
                var me = GetCurrentUserAndRegion();
                var myFriendship = Friendship.Get(me.Region.RegionID, me.User.UserID, friendID);

                if (myFriendship == null || myFriendship.Status != FriendshipStatus.Confirmed)
                {
                    return new GetFriendsOfFriendResponse { Status = BasicServiceResponseStatus.Forbidden };
                }

                var theirFriends = Friendship.GetAllConfirmed(myFriendship.OtherUserRegionID, friendID);

                return new GetFriendsOfFriendResponse
                {
                    Status = BasicServiceResponseStatus.Successful,
                    FriendsOfFriend = theirFriends.Select(f => new FriendOfFriendDetails
                    {
                        UserID = f.OtherUserID,
                        Username = f.OtherUsername
                    }).ToArray()
                };
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "[GetFriendsOfFriend] Error retrieving friends of a friend.", new { Requestor = AuthenticationContext.Current.UserID, FriendID = friendID });
                return new GetFriendsOfFriendResponse { Status = BasicServiceResponseStatus.Error };
            }
        }

        public GetMyPushNotificationPreferencesResponse GetMyPushNotificationPreferences()
        {
            try
            {
                var me = GetCurrentUserAndRegion();
                return new GetMyPushNotificationPreferencesResponse
                {
                    Status = GetMyPushNotificationPreferencesStatus.Success,
                    FriendMessagePushPreference = me.User.FriendMessagePushPreference,
                    GroupMessagePushPreference = me.User.GroupMessagePushPreference,
                    FriendRequestPushEnabled = me.User.FriendRequestPushEnabled,
                    MentionsPushEnabled = !me.User.MentionsPushEnabled.HasValue || me.User.MentionsPushEnabled.Value
                };
            }
            catch (UserNotFoundException)
            {
                return new GetMyPushNotificationPreferencesResponse
                {
                    Status = GetMyPushNotificationPreferencesStatus.Unauthorized
                };
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "[GetMyPushNotificationPreferences] Unhandled exception");
                return new GetMyPushNotificationPreferencesResponse
                {
                    Status = GetMyPushNotificationPreferencesStatus.Error
                };
            }
        }

        public BasicServiceResponse ChangePushNotificationPreferences(ChangePushNotificationPreferencesRequest request)
        {
            try
            {
                var user = GetCurrentUserAndRegion().User;
                user.GroupMessagePushPreference = request.GroupMessagePushPreference;
                user.FriendMessagePushPreference = request.FriendMessagePushPreference;
                user.FriendRequestPushEnabled = request.FriendRequestPushEnabled;
                user.MentionsPushEnabled = request.MentionsPushEnabled;
                user.Update(u => u.GroupMessagePushPreference,
                    u => u.FriendMessagePushPreference,
                    u => u.FriendRequestPushEnabled,
                    u => u.MentionsPushEnabled);

                return new BasicServiceResponse { Status = BasicServiceResponseStatus.Successful };
            }
            catch (UserNotFoundException)
            {
                return new BasicServiceResponse { Status = BasicServiceResponseStatus.Forbidden };
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "[ChangePushNotificationPreferences] Unhandled exception");
                return new BasicServiceResponse { Status = BasicServiceResponseStatus.Error };
            }
        }


        public ConfirmFriendshipResponse ConfirmFriendship(ConfirmFriendshipRequest request)
        {
            try
            {
                var authToken = AuthenticationContext.Current;
                var friendshipContext = new FriendshipContext(authToken.UserID, request.FriendID);

                // Try to get the friendship requested                
                var myFriendship = friendshipContext.GetMyFriendship();
                var theirFriendship = friendshipContext.GetTheirFriendship();

                if (myFriendship == null && theirFriendship == null)
                {
                    Logger.Debug("[ConfirmFriendship] Not found", request);
                    return new ConfirmFriendshipResponse(ConfirmFriendshipStatus.NotFound);
                }

                if (myFriendship == null && theirFriendship != null)
                {
                    Logger.Debug("[ConfirmFriendship] Data issue detected, myFriendship is null", request);
                    theirFriendship.Delete();
                    return new ConfirmFriendshipResponse(ConfirmFriendshipStatus.NotFound);
                }

                if (theirFriendship == null && myFriendship != null)
                {
                    Logger.Debug("[ConfirmFriendship] Data issue detected, theirFriendship is null", request);
                    myFriendship.Delete();
                    return new ConfirmFriendshipResponse(ConfirmFriendshipStatus.NotFound);
                }

                // Ensure that the far end friendship is in the 'AwaitingConfirmation' state.
                if (theirFriendship.Status != FriendshipStatus.AwaitingThem)
                {
                    Logger.Debug("[ConfirmFriendship] Other user's status is not AwaitingThem", request);
                    return new ConfirmFriendshipResponse(ConfirmFriendshipStatus.NotFound);
                }

                // Finalize the friendship
                Friendship.Confirm(friendshipContext.Me, friendshipContext.MyRegion, myFriendship, friendshipContext.Them, friendshipContext.TheirRegion, theirFriendship);

                // Track this, for stats
                FriendsStatsManager.Current.FriendRequestsConfirmed.Track();

                return new ConfirmFriendshipResponse(ConfirmFriendshipStatus.Successful);
            }
            catch (FriendshipContextException ex)
            {
                Logger.Debug(ex, "[ConfirmFriendship] Unable to create a friendship context.", request);
                return new ConfirmFriendshipResponse(ConfirmFriendshipStatus.NotFound);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "[ConfirmFriendship] Unhandled exception");
                return new ConfirmFriendshipResponse(ConfirmFriendshipStatus.Error);
            }
        }

        public RequestFriendshipResponse RequestFriendship(FriendshipRequest request)
        {
            try
            {
                if (!request.Validate())
                {
                    return new RequestFriendshipResponse { Status = RequestFriendshipStatus.FailedValidation };
                }

                var authToken = AuthenticationContext.Current;
                var friendshipContext = new FriendshipContext(authToken.UserID, request.FriendID);

                var myFriendship = friendshipContext.GetMyFriendship();
                var theirFriendship = friendshipContext.GetTheirFriendship();


                // Already requested
                if (myFriendship != null && theirFriendship != null)
                {
                    if (request.IsFromSuggestion)
                    {
                        var suggestion = FriendSuggestion.Get(friendshipContext.MyRegion.RegionID,
                            friendshipContext.Me.UserID, request.FriendID);
                        if (suggestion != null && suggestion.Status == FriendSuggestionStatus.Pending)
                        {
                            suggestion.Status = FriendSuggestionStatus.Accepted;
                            suggestion.Update();
                        }
                    }

                    if (myFriendship.Status == FriendshipStatus.Confirmed ||
                        myFriendship.Status == FriendshipStatus.AwaitingThem ||
                        myFriendship.Status == FriendshipStatus.DeclinedByThem)
                    {
                        return new RequestFriendshipResponse(RequestFriendshipStatus.AlreadyRequested);
                    }
                }

                // Ensure this user doesn't have a ton of friendship records already. We check the actual friendship set, instead of user statistics, to ensure consistency.
                var myFriendCount = Friendship.GetConfirmedOrRequestedCount(friendshipContext.MyRegion.RegionID, friendshipContext.Me.UserID);

                if (myFriendCount > Friendship.MaxFriendCount)
                {
                    return new RequestFriendshipResponse { Status = RequestFriendshipStatus.MaximumFriends };
                }

                var theirFriendCount = Friendship.GetConfirmedOrRequestedCount(friendshipContext.TheirRegion.RegionID, friendshipContext.Them.UserID);

                if (theirFriendCount > Friendship.MaxFriendCount)
                {
                    return new RequestFriendshipResponse { Status = RequestFriendshipStatus.MaximumFriends };
                }

                if (theirFriendship != null && myFriendship != null && myFriendship.Status == FriendshipStatus.AwaitingMe) // Confirm it
                {
                    Friendship.Confirm(friendshipContext.Me, friendshipContext.MyRegion, myFriendship,
                        friendshipContext.Them, friendshipContext.TheirRegion, theirFriendship);
                }
                else // Create the request on both sides
                {
                    var result = Friendship.CreateRequest(friendshipContext.Me, friendshipContext.MyRegion,
                        friendshipContext.Them, friendshipContext.TheirRegion, request.KnownIdentity,
                        request.InvitationMessage, request.IsFromSuggestion);
                }

                // Track this, for stats
                if (request.IsFromSuggestion)
                {
                    FriendsStatsManager.Current.SuggestionsAccepted.Track();
                }
                else
                {
                    FriendsStatsManager.Current.FriendRequestsSent.Track();
                }

                return new RequestFriendshipResponse(RequestFriendshipStatus.Successful);
            }
            catch (FriendshipContextException ex)
            {
                Logger.Trace(ex, "[RequestFriendship] Unable to create a friendship context.", request);
                return new RequestFriendshipResponse(RequestFriendshipStatus.NotFound);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "[RequestFriendship] Unhandled exception", request);

                return new RequestFriendshipResponse(RequestFriendshipStatus.Error);
            }
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        public BasicServiceResponse RemoveFriendship(RemoveFriendshipRequest request)
        {
            try
            {
                if (!request.Validate())
                {
                    Logger.Debug("[RemoveFriendship] Invalid", request);
                    return new BasicServiceResponse { Status = BasicServiceResponseStatus.Invalid };
                }

                // Ensure that the friendship exists
                var authToken = AuthenticationContext.Current;
                var friendshipContext = new FriendshipContext(authToken.UserID, request.FriendID);

                var myFriendship = friendshipContext.GetMyFriendship();
                var theirFriendship = friendshipContext.GetTheirFriendship();

                // If neither side of the friendship exists, this is an invalid request
                if (myFriendship == null && theirFriendship == null)
                {
                    return new BasicServiceResponse { Status = BasicServiceResponseStatus.Invalid };
                }

                // If only their side exists, remove it (this is a data issue)
                if (myFriendship == null && theirFriendship != null)
                {
                    Logger.Debug("[RemoveFriendship] Detected data issue, myFriendshup is null");
                    theirFriendship.Delete();
                    return new BasicServiceResponse { Status = BasicServiceResponseStatus.Successful };
                }

                // If only my side exists, remove it (this is a data issue)
                if (theirFriendship == null && myFriendship != null)
                {
                    Logger.Debug("[RemoveFriendship] Detected data issue, theirFriendship is null");
                    myFriendship.Delete();
                    return new BasicServiceResponse { Status = BasicServiceResponseStatus.Successful };
                }

                var canRemove = (myFriendship.Status == FriendshipStatus.Confirmed && theirFriendship.Status == FriendshipStatus.Confirmed)
                    || (myFriendship.Status == FriendshipStatus.AwaitingThem);

                if (!canRemove)
                {
                    Logger.Debug("[RemoveFriendship] Unable to remove friendship", new { request, myFriendship, theirFriendship });
                    return new BasicServiceResponse { Status = BasicServiceResponseStatus.NotFound };
                }

                Friendship.Remove(false, friendshipContext.Me, myFriendship, friendshipContext.MyRegion, friendshipContext.Them, theirFriendship, friendshipContext.TheirRegion);


                // Notify Myself        
                new FriendshipRemovedResolver { UserID = friendshipContext.Me.UserID, FriendID = friendshipContext.Them.UserID }.Enqueue();

                // Notify Them
                new FriendshipRemovedResolver { UserID = friendshipContext.Them.UserID, FriendID = friendshipContext.Me.UserID }.Enqueue();

                // Track this, for stats
                FriendsStatsManager.Current.FriendRequestsRemoved.Track();

                return new BasicServiceResponse() { Status = BasicServiceResponseStatus.Successful };

            }
            catch (FriendshipContextException ex)
            {
                Logger.Debug(ex, "[RemoveFriendship] Failed to create friendship context.");
                return new BasicServiceResponse { Status = BasicServiceResponseStatus.Invalid };
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "[RemoveFriendship] Unhandled exception");
                return new BasicServiceResponse { Status = BasicServiceResponseStatus.Error };
            }

        }

        /// <summary>
        /// Unblocks a user from sending future friend requests
        /// </summary>
        /// <param name="request"></param>
        /// <returns></returns>
        public BasicServiceResponse UnblockFriendship(UnblockFriendshipRequest request)
        {
            try
            {
                if (!request.Validate())
                {
                    return new BasicServiceResponse { Status = BasicServiceResponseStatus.Invalid };
                }

                // Ensure that the friendship exists
                var authToken = AuthenticationContext.Current;
                var friendshipContext = new FriendshipContext(authToken.UserID, request.FriendID);

                var myFriendship = friendshipContext.GetMyFriendship(true);
                var theirFriendship = friendshipContext.GetTheirFriendship(true);

                if (myFriendship.Status != FriendshipStatus.DeclinedByMe)
                {
                    return new BasicServiceResponse { Status = BasicServiceResponseStatus.Invalid };
                }

                Friendship.Remove(false, friendshipContext.Me, myFriendship, friendshipContext.MyRegion, friendshipContext.Them, theirFriendship, friendshipContext.TheirRegion);

                return new BasicServiceResponse { Status = BasicServiceResponseStatus.Successful };

            }
            catch (FriendshipContextException ex)
            {
                Logger.Trace(ex, "[UnblockFriendship] Failed to create friendship context.");
                return new BasicServiceResponse { Status = BasicServiceResponseStatus.Invalid };
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "[UnblockFriendship] Unhandled exception.");
                return new BasicServiceResponse { Status = BasicServiceResponseStatus.Error };
            }
        }

        public BasicServiceResponse ChangeFriendNotificationPreferences(ChangeFriendNotificationPreferencesRequest request)
        {
            if (!request.Validate())
            {
                Logger.Debug("[ChangeUserNotificationPreference] Failed to validate: " + request.ValidationMessage);
                return new BasicServiceResponse { Status = BasicServiceResponseStatus.Invalid };
            }

            try
            {
                var authToken = AuthenticationContext.Current;
                var friendshipContext = new FriendshipContext(authToken.UserID, request.FriendID);
                var myFriendship = friendshipContext.GetMyFriendship(true);
                myFriendship.UpdateNotificationPreferences(request.Preference, request.FilterSet);
                RefreshFriendshipAndNotify(authToken.UserID, friendshipContext);
                return new BasicServiceResponse { Status = BasicServiceResponseStatus.Successful };
            }
            catch (FriendshipContextException ex)
            {
                Logger.Debug(ex, "[ChangeUserNotificationPreference] Unable to create a friendship context.", request);
                return new BasicServiceResponse { Status = BasicServiceResponseStatus.Forbidden };
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "[ChangeUserNotificationPreference] Unhandled exception");
                return new BasicServiceResponse { Status = BasicServiceResponseStatus.Error };
            }
        }

        public DeclineFriendshipResponse DeclineFriendship(DeclineFriendshipRequest request)
        {
            try
            {
                if (!request.Validate())
                {
                    Logger.Debug("[DeclineFriendship] Failed validation: " + request.ValidationMessage, request);
                    return new DeclineFriendshipResponse { Status = DeclineFriendshipStatus.FailedValidation };
                }

                var authToken = AuthenticationContext.Current;
                var friendshipContext = new FriendshipContext(authToken.UserID, request.FriendID);

                var myFriendship = friendshipContext.GetMyFriendship();
                var theirFriendship = friendshipContext.GetTheirFriendship();

                if (myFriendship == null && theirFriendship == null)
                {
                    return new DeclineFriendshipResponse { Status = DeclineFriendshipStatus.NotFound };
                }

                if (myFriendship == null && theirFriendship != null)
                {
                    theirFriendship.Delete();
                    return new DeclineFriendshipResponse { Status = DeclineFriendshipStatus.NotFound };
                }

                if (theirFriendship == null && myFriendship != null)
                {
                    myFriendship.Delete();
                    return new DeclineFriendshipResponse { Status = DeclineFriendshipStatus.NotFound };
                }

                // If this friendship isn't awaiting me, consider it to be not found
                if (myFriendship.Status != FriendshipStatus.AwaitingMe)
                {
                    return new DeclineFriendshipResponse { Status = DeclineFriendshipStatus.NotFound };
                }

                // Create a deny and decline status, and then notify the client
                Friendship.Remove(request.BlockFutureRequests, friendshipContext.Me, myFriendship,
                    friendshipContext.MyRegion, friendshipContext.Them, theirFriendship, friendshipContext.TheirRegion);

                new FriendshipRemovedResolver()
                {
                    UserID = friendshipContext.Me.UserID,
                    FriendID = myFriendship.OtherUserID
                }.Enqueue();
                new FriendshipRemovedResolver()
                {
                    UserID = friendshipContext.Them.UserID,
                    FriendID = theirFriendship.OtherUserID
                }.Enqueue();

                // Track this, for stats
                FriendsStatsManager.Current.FriendRequestsDeclined.Track();

                return new DeclineFriendshipResponse { Status = DeclineFriendshipStatus.Successful };
            }
            catch (FriendshipContextException)
            {
                return new DeclineFriendshipResponse { Status = DeclineFriendshipStatus.NotFound };
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "[DeclineFriendship] Unhandled exception");
                return new DeclineFriendshipResponse { Status = DeclineFriendshipStatus.Error };
            }
        }

        private UserRegionInfo GetCurrentUserAndRegion(bool createRegion = false)
        {
            var authToken = AuthenticationContext.Current;

            return GetUserAndRegion(authToken.UserID, createRegion);
        }

        private UserRegionInfo GetUserAndRegion(int userID, bool createRegion = false)
        {
            // See if the user has registered before
            var userRegion = UserRegion.GetLocal(userID);

            // If not, register them to the local storage node
            if (userRegion == null)
            {
                if (!createRegion)
                {
                    throw new UserNotFoundException(userID);
                }

                userRegion = new UserRegion { UserID = userID, RegionID = UserRegion.LocalConfigID };
                userRegion.InsertLocal();
                userRegion = UserRegion.GetLocal(userID);
                if (userRegion == null)
                {
                    throw new InvalidOperationException("Failed to read back UserRegion after inserting it!");
                }
            }

            // From their storage region, get their record and update their status            
            var user = User.Get(userRegion.RegionID, userID);

            if (user == null && !createRegion)
            {
                throw new UserNotFoundException(userID);
            }

            return new UserRegionInfo(user, userRegion);
        }

        public RegisterSelfResponse RegisterSelf(RegisterSelfRequest request)
        {
            try
            {

                if (request.MachineKey == Guid.Empty)
                {
                    Logger.Warn("Attempt to register an invalid endpoint");
                    return new RegisterSelfResponse { Status = RegisterSelfStatus.Error };
                }

                var authToken = AuthenticationContext.Current;
                var userAndRegion = RegisterUser(authToken.UserID, authToken.Username, authToken.Email);

                // Create a new session for this
                var sessionID = Guid.NewGuid().ToString();

                // Register the client endpoint
                var endpoint = ClientEndpoint.GetLocal(authToken.UserID, request.MachineKey.ToString());

                if (endpoint == null) // Create a new endpoint and session
                {
                    endpoint = new ClientEndpoint
                    {
                        UserID = authToken.UserID,
                        MachineKey = request.MachineKey.ToString(),
                        SessionDate = DateTime.UtcNow,
                        IsConnected = false,
                        RegionID = ClientEndpoint.LocalConfigID,
                        SessionID = sessionID,
                        Platform = request.Platform,
                        DeviceID = request.DeviceID,
                        PushKitToken = request.PushKitToken
                    };

                    endpoint.InsertLocal();
                }
                else // Endpoint already exists, so just update it
                {
                    if (DateTime.UtcNow - endpoint.SessionDate > TimeSpan.FromDays(1)) // Issue a new session ID
                    {
                        ClientEndpoint.UpdateSession(sessionID, authToken.UserID, request.MachineKey.ToString());
                    }
                    else
                    {
                        sessionID = endpoint.SessionID;
                    }

                    // Ensure that this endpoint is indexed
                    endpoint.ValidateMap();
                }


                if (endpoint.Platform != request.Platform)
                {
                    ClientEndpoint.UpdatePlatform(request.Platform, authToken.UserID, request.MachineKey.ToString());
                }

                if (endpoint.DeviceID != request.DeviceID)
                {
                    Logger.Trace("Updating endpoint device ID", endpoint);
                    ClientEndpoint.UpdateDeviceID(request.DeviceID, authToken.UserID, request.MachineKey.ToString());
                }

                if (endpoint.PushKitToken != request.PushKitToken)
                {
                    Logger.Trace("Updating endpoint Voice Token", new { endpoint, request.PushKitToken });
                    ClientEndpoint.UpdatePushKitToken(request.PushKitToken, authToken.UserID, request.MachineKey.ToString());
                }

                var user = userAndRegion.User;
                user.AvatarUrl = GetNewFriendAvatarUrl(user.UserID);

                var resp = new RegisterSelfResponse
                {
                    Status = RegisterSelfStatus.Successful,
                    SessionID = sessionID,
                    User = user
                };

                if (endpoint.Platform != DevicePlatform.Chrome)
                {
                    resp.NotificationHostPort = FriendsServiceConfiguration.Instance.NotificationServicePorts.FirstOrDefault();
                    resp.NotificationHostPorts = FriendsServiceConfiguration.Instance.NotificationServicePorts;
                    resp.NotificationHostList = NotificationHostManager.GetHostList();
                }

                return resp;
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "[RegisterSelf] Unhandled exception");
                return new RegisterSelfResponse { Status = RegisterSelfStatus.Error };
            }

        }

        public BasicServiceResponse UpdateDeviceTokens(UpdateDeviceTokensRequest request)
        {
            try
            {
                if (!ClientEndpoint.UpdateDeviceTokens(request.DeviceID, request.PushKitID, AuthenticationContext.Current.UserID, request.MachineKey))
                {
                    return new BasicServiceResponse { Status = BasicServiceResponseStatus.NotFound };
                }

                return new BasicServiceResponse { Status = BasicServiceResponseStatus.Successful };
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "[UpdateDeviceTokens] Unhandled exception");
                return new BasicServiceResponse { Status = BasicServiceResponseStatus.Error };
            }
        }

        private UserRegionInfo RegisterUser(int userID, string username, string email)
        {
            // From their storage region, get their record and update their status            
            var userAndRegion = GetUserAndRegion(userID, true);
            var reindexSearch = false;

            if (userAndRegion.User == null) // If it doesn't exist, insert or replace
            {
                userAndRegion.User = new User
                {
                    UserID = userID,
                    Username = username,
                };

                userAndRegion.User.Insert(userAndRegion.Region.RegionID);

                userAndRegion.User = User.Get(userAndRegion.Region.RegionID, userID);

                if (userAndRegion.User == null)
                {
                    throw new InvalidOperationException("Failed to retrieve user after inserting it.");
                }

                reindexSearch = true;
            }
            else // Otherwise, update the status
            {
                if (!string.IsNullOrEmpty(username) && userAndRegion.User.Username != username)
                {
                    reindexSearch = true;
                    userAndRegion.User.Username = username;
                    userAndRegion.User.Update(p => p.Username);
                }
            }

            // Upsert their friend hints
            if (reindexSearch || FriendsServiceConfiguration.Instance.AlwaysIndexUserSearch)
            {
                try
                {
                    if (!string.IsNullOrEmpty(username))
                    {
                        new FriendHint() { UserID = userID, Type = FriendHintType.Username, SearchTerm = username, AvatarUrl = userAndRegion.User.AvatarUrl, Verification = FriendHintVerification.Verified }.InsertLocal();
                    }

                    if (!string.IsNullOrEmpty(email))
                    {
                        new FriendHint() { UserID = userID, Type = FriendHintType.Email, SearchTerm = email, AvatarUrl = userAndRegion.User.AvatarUrl, Verification = FriendHintVerification.Verified }.InsertLocal();
                    }

                    // Queue up a job to reindex the user
                    FriendHintSearchIndexer.CreateForUser(userAndRegion.User.UserID);
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to create friend hint search indexer for registering user.");
                }
            }

            return userAndRegion;
        }

        public BasicServiceResponse UnregisterEndpoint(UnregisterEndpointRequest request)
        {
            try
            {
                var authToken = AuthenticationContext.Current;

                var endpoint = ClientEndpoint.GetLocal(authToken.UserID, request.MachineKey.ToString());

                if (endpoint == null)
                {
                    return new BasicServiceResponse { Status = BasicServiceResponseStatus.NotFound };
                }

                if (endpoint.DeviceID != request.DeviceID)
                {
                    return new BasicServiceResponse { Status = BasicServiceResponseStatus.Forbidden };
                }

                // Update any endpoints with this device ID, for this user
                var allDeviceEndpoints = ClientEndpoint.GetAllLocal(p => p.UserID, authToken.UserID).Where(p => p.DeviceID == request.DeviceID);

                Logger.Trace("Unregistering endpoint", request);

                foreach (var deviceEndpoint in allDeviceEndpoints)
                {
                    deviceEndpoint.DeviceID = null;
                    deviceEndpoint.PushKitToken = null;
                    deviceEndpoint.Update(p => p.DeviceID, p => p.PushKitToken);
                }

                return new BasicServiceResponse { Status = BasicServiceResponseStatus.Successful };

            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to unregister endpoint", request);
                return new BasicServiceResponse { Status = BasicServiceResponseStatus.Error };
            }

        }

        public GetHostListResponse GetHostList()
        {
            try
            {
                return new GetHostListResponse
                {
                    NotificationHostPort = FriendsServiceConfiguration.Instance.NotificationServicePorts.FirstOrDefault(),
                    NotificationHostPorts = FriendsServiceConfiguration.Instance.NotificationServicePorts,
                    NotificationHostList = NotificationHostManager.GetHostList(),
                };
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "[GetHostList] Unhandled exception");
                return null;
            }
        }

        public ToggleFavoriteResponse ToggleFavorite(ToggleFavoriteRequest request)
        {
            try
            {
                var authToken = AuthenticationContext.Current;
                var friendshipContext = new FriendshipContext(authToken.UserID, request.FriendID);
                var myFriendship = friendshipContext.GetMyFriendship();
                if (myFriendship == null)
                {
                    return new ToggleFavoriteResponse { Status = ToggleFavoriteStatus.NotFound };
                }

                Friendship.UpdateIsFavorite(friendshipContext.MyRegion.RegionID, authToken.UserID, request.FriendID,
                    request.IsFavorite);

                RefreshFriendshipAndNotify(authToken.UserID, friendshipContext);

                return new ToggleFavoriteResponse { Status = ToggleFavoriteStatus.Successful };
            }
            catch (FriendshipContextException ex)
            {
                Logger.Warn(ex, "[ToggleFavorite] Friendship exception");
                return new ToggleFavoriteResponse { Status = ToggleFavoriteStatus.Error };
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "[ToggleFavorite] Unhandled exception");
                return new ToggleFavoriteResponse { Status = ToggleFavoriteStatus.Error };
            }
        }

        /// <summary>
        /// Notify a user about a change to one of their friendships
        /// </summary>
        /// <param name="userID"></param>
        /// <param name="friendshipContext"></param>
        private void RefreshFriendshipAndNotify(int userID, FriendshipContext friendshipContext)
        {
            var myFriendship = friendshipContext.GetMyFriendship(); // Refresh

            // Let all the current users endpoints know to refresh this friendship
            var endpoints = ClientEndpoint.GetAllConnected(userID);

            foreach (var endpoint in endpoints)
            {
                new FriendshipChangeNotifier(endpoint) { Friend = myFriendship }.Enqueue();
            }
        }

        public RenameFriendResponse RenameFriend(RenameFriendRequest request)
        {
            try
            {

                if (!request.Validate())
                {
                    return new RenameFriendResponse { Status = RenameFriendStatus.Error };
                }

                var authToken = AuthenticationContext.Current;
                var friendshipContext = new FriendshipContext(authToken.UserID, request.FriendID);
                var myFriendship = friendshipContext.GetMyFriendship();
                if (myFriendship == null)
                {
                    return new RenameFriendResponse { Status = RenameFriendStatus.NotFound };
                }

                Friendship.UpdateNickname(friendshipContext.MyRegion.RegionID, authToken.UserID, request.FriendID, request.Nickname);

                RefreshFriendshipAndNotify(authToken.UserID, friendshipContext);

                return new RenameFriendResponse { Status = RenameFriendStatus.Successful };

            }
            catch (Exception ex)
            {
                Logger.Error(ex, "[RenameFriend] Unhandled exception");
                return new RenameFriendResponse { Status = RenameFriendStatus.Error };
            }
        }

        public ChangeStatusResponse ChangeStatus(ChangeStatusRequest request)
        {
            try
            {
                if (!request.Validate())
                {
                    return new ChangeStatusResponse { Status = ChangeStatusStatus.Invalid, ValidationMessage = request.ValidationMessage };
                }

                // From their storage region, get their record and update their status            
                var userAndRegion = GetCurrentUserAndRegion();
                var endpoint = ClientEndpoint.GetLocal(userAndRegion.User.UserID, request.MachineKey);

                if (endpoint == null)
                {
                    throw new UserNotFoundException(userAndRegion.User.UserID);
                }

                // If the user is offline, and they have no connected endpoint, do not let them change their status manually.
                if (userAndRegion.User.ConnectionStatus == UserConnectionStatus.Offline && !endpoint.IsConnected)
                {
                    return new ChangeStatusResponse { Status = ChangeStatusStatus.Successful }; // For now, to stop all the spam!                
                }

                var statusChanged = userAndRegion.User.ConnectionStatus != request.Status;

                // Update their status                
                if (userAndRegion.User.CustomStatusMessage != request.CustomStatusMessage)
                {
                    statusChanged = true;
                    userAndRegion.User.CustomStatusTimestamp = DateTime.UtcNow;
                    userAndRegion.User.CustomStatusMessage = request.CustomStatusMessage;
                    userAndRegion.User.Update(p => p.CustomStatusTimestamp, p => p.CustomStatusMessage);
                    userAndRegion.User.UpdateStatistics();
                }

                if (request.Status == UserConnectionStatus.Idle)
                {
                    if (!endpoint.IsIdle)
                    {
                        statusChanged = true;
                        endpoint.ToggleIdle(true, userAndRegion.Region.RegionID);
                    }

                }
                else if (endpoint.IsIdle)
                {
                    statusChanged = true;
                    endpoint.ToggleIdle(false, userAndRegion.Region.RegionID);
                }

                // Queue off a job to let their friends know their status
                if (statusChanged)
                {
                    UserStatusResolver.CreateStatusChangeResolver(userAndRegion.User.UserID, request.Status, endpoint.MachineKey, endpoint.SessionID);
                }

                return new ChangeStatusResponse() { Status = ChangeStatusStatus.Successful };

            }
            catch (UserNotFoundException)
            {
                return new ChangeStatusResponse { Status = ChangeStatusStatus.NotRegistered };
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "[ChangeStatus] Unhandled exception");
                return new ChangeStatusResponse { Status = ChangeStatusStatus.Error };
            }
        }

        public ChangeGameStatusResponse ChangeGameStatus(ChangeGameStatusRequest request)
        {
            try
            {
                if (!request.Validate())
                {
                    return new ChangeGameStatusResponse { Status = ChangeGameStatusStatus.Invalid };
                }

                // From their storage region, get their record and update their status
                var userAndRegion = GetCurrentUserAndRegion();

                // No GameID implies not running and visa versa
                var newGameID = request.GameID ?? 0;
                if (newGameID == 0)
                {
                    request.IsRunning = false;
                }
                else if (!request.IsRunning)
                {
                    newGameID = 0;
                }

                // Update current game stats
                if (newGameID != userAndRegion.User.CurrentGameID)
                {
                    if (userAndRegion.User.CurrentGameID > 0)
                    {
                        // No longer playing the previous game
                        FriendsStatsManager.Current.CurrentGames.Decrement(userAndRegion.User.CurrentGameID);
                    }

                    if (newGameID > 0)
                    {
                        // Playing a new game now
                        FriendsStatsManager.Current.CurrentGames.Track(newGameID);

                        // Detect game changes
                        FriendsStatsManager.Current.GameNotifications.Track(newGameID);
                    }
                }

                // Update machine key as needed
                if (!request.IsRunning && userAndRegion.User.CurrentGameClientMachineKey != null)
                {
                    userAndRegion.User.CurrentGameClientMachineKey = null;
                    userAndRegion.User.Update(u => u.CurrentGameClientMachineKey);
                }
                else if (userAndRegion.User.CurrentGameClientMachineKey != request.MachineKey)
                {
                    userAndRegion.User.CurrentGameClientMachineKey = request.MachineKey;
                    userAndRegion.User.Update(u => u.CurrentGameClientMachineKey);
                }

                // Only process this if it is a change!
                if (newGameID != userAndRegion.User.CurrentGameID ||
                    request.GameState != userAndRegion.User.CurrentGameState ||
                    request.GameStatusMessage != userAndRegion.User.CurrentGameStatusMessage)
                {
                    // Update their status
                    userAndRegion.User.CurrentGameID = newGameID;

                    if (newGameID > 0)
                    {
                        userAndRegion.User.LastGameID = newGameID;
                    }

                    userAndRegion.User.CurrentGameTimestamp = DateTime.UtcNow;
                    userAndRegion.User.CurrentGameState = request.GameState;
                    userAndRegion.User.CurrentGameStatusMessage = request.GameStatusMessage;
                    userAndRegion.User.Update(p => p.CurrentGameID, p => p.LastGameID, p => p.CurrentGameTimestamp,
                        p => p.CurrentGameState, p => p.CurrentGameStatusMessage);
                    
                    userAndRegion.User.UpdateStatistics();

                    // Queue off a job to let their friends know that they are playing a game
                    RegionalUserChangeResolver.Create(userAndRegion.User.UserID);
                }

                return new ChangeGameStatusResponse { Status = ChangeGameStatusStatus.Successful };

            }
            catch (UserNotFoundException)
            {
                return new ChangeGameStatusResponse { Status = ChangeGameStatusStatus.NotRegistered };
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "[ChangeGameStatus] Unhandled exception");
                return new ChangeGameStatusResponse { Status = ChangeGameStatusStatus.Successful };
            }
        }

        public ChangeProfileResponse ChangeProfile(ChangeProfileRequest request)
        {
            try
            {
                if (!request.Validate())
                {
                    Logger.Debug("[ChangeProfile] Failed Validation: " + request.ValidationMessage, request);
                    return new ChangeProfileResponse { Status = ChangeProfileStatus.Invalid };
                }

                var authToken = AuthenticationContext.Current;

                // From their storage region, get their record and update their status            
                var userAndRegion = GetCurrentUserAndRegion();
                var updateSearchHint = userAndRegion.User.AvatarUrl != request.AvatarUrl;

                // Update the new avatar model
                if(!string.IsNullOrWhiteSpace(request.AvatarUrl))
                {
                    SaveLegacyAvatar(request.AvatarUrl, userAndRegion.User.UserID);
                }
               
                // Update their profile info
                userAndRegion.User.AvatarUrl = request.AvatarUrl;
                userAndRegion.User.AboutMe = request.AboutMe;
                userAndRegion.User.City = request.City;
                userAndRegion.User.State = request.State;
                userAndRegion.User.CountryCode = request.CountryCode;
                userAndRegion.User.Name = request.Name;
                userAndRegion.User.AboutMe = request.AboutMe;
                userAndRegion.User.Update(p => p.AvatarUrl, p => p.AboutMe, p => p.City, p => p.State, p => p.CountryCode, p => p.Name, p => p.AboutMe);

                // Fan this change out to their friends
                RegionalUserChangeResolver.Create(userAndRegion.User.UserID);

                // Update their friend hints
                if (updateSearchHint)
                {
                    var hints = FriendHint.GetAllLocal(p => p.UserID, userAndRegion.User.UserID).Where(p => p.Type == FriendHintType.Email || p.Type == FriendHintType.Username).ToArray();

                    foreach (var hint in hints)
                    {
                        hint.AvatarUrl = request.AvatarUrl;
                        hint.Update();
                    }

                    new FriendHintSearchIndexer { UserID = userAndRegion.User.UserID }.Enqueue();
                }

                var hasAnyData = !string.IsNullOrEmpty(request.AboutMe)
                                 || !string.IsNullOrEmpty(request.City)
                                 || !string.IsNullOrEmpty(request.CountryCode)
                                 || !string.IsNullOrEmpty(request.Name)
                                 || !string.IsNullOrEmpty(request.State);

                if (hasAnyData)
                {
                    FriendsStatsManager.Current.ProfilesUpdated.Track(authToken.UserID);
                }

                return new ChangeProfileResponse() { Status = ChangeProfileStatus.Successful };

            }
            catch (UserNotFoundException)
            {
                return new ChangeProfileResponse { Status = ChangeProfileStatus.NotRegistered };
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "[ChangeProfile] Unhandled exception");
                return new ChangeProfileResponse() { Status = ChangeProfileStatus.Error };
            }
        }

        private void SaveLegacyAvatar(string avatarUrl, int userID)
        {
            
            try
            {
                if (string.IsNullOrWhiteSpace(avatarUrl))
                {
                    return;
                }

                if (!avatarUrl.Contains("curseapp"))
                {
                    var avatar = Avatar.GetByTypeAndID(AvatarType.User, userID.ToString());
                    if (avatar == null)
                    {
                        avatar = new Avatar
                        {
                            AvatarType = (int)AvatarType.User,
                            EntityID = userID.ToString(),
                            Timestamp = DateTime.UtcNow.ToEpochMilliseconds(),
                            Url = avatarUrl
                        };

                        avatar.InsertLocal();
                    }
                    else if (avatar.Url == null || !avatar.Url.Equals(avatarUrl))
                    {
                        avatar.Url = avatarUrl;
                        avatar.StorageKey = string.Empty;
                        avatar.Filename = string.Empty;
                        avatar.FileRegionID = 0;
                        avatar.Filename = string.Empty;
                        avatar.Update();
                    }
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to update legacy avatar.");
            }
           
        }

        public BasicServiceResponse DeclineFriendSuggestion(DeclineFriendSuggestionRequest request)
        {
            try
            {
                if (!request.Validate())
                {
                    Logger.Debug("[DeclineFriendSuggestion] Failed Validation: " + request.ValidationMessage, request);
                    return new BasicServiceResponse { Status = BasicServiceResponseStatus.Invalid };
                }

                var userAndRegion = GetCurrentUserAndRegion();
                var suggestion = FriendSuggestion.Get(userAndRegion.Region.RegionID, userAndRegion.User.UserID, request.FriendID);
                if (suggestion != null && suggestion.Status == FriendSuggestionStatus.Pending)
                {
                    suggestion.Status = FriendSuggestionStatus.Declined;
                    suggestion.Update();
                    FriendsStatsManager.Current.SuggestionsDeclined.Track();
                }

                return new BasicServiceResponse() { Status = BasicServiceResponseStatus.Successful };
            }
            catch (UserNotFoundException)
            {
                return new BasicServiceResponse { Status = BasicServiceResponseStatus.NotFound };
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "[DeclineFriendSuggestion] Unhandled exception");
                return new BasicServiceResponse { Status = BasicServiceResponseStatus.Error };
            }
        }

        public BasicServiceResponse FriendListSearch(FriendListSearchRequest request)
        {
            try
            {
                if (!request.Validate())
                {
                    Logger.Debug("[FriendListSearch] Failed Validation: " + request.ValidationMessage, request);
                    FriendsStatsManager.Current.FriendSyncsByResult.Track((int)BasicServiceResponseStatus.Invalid);
                    return new BasicServiceResponse { Status = BasicServiceResponseStatus.Invalid };
                }

                var user = GetCurrentUserAndRegion();

                if (user.User.LastFriendSyncSearch != DateTime.MinValue
                    && DateTime.UtcNow.Subtract(user.User.LastFriendSyncSearch) < TimeSpan.FromMinutes(5))
                {
                    return new BasicServiceResponse {Status = BasicServiceResponseStatus.Invalid};
                }

                user.User.LastFriendSyncSearch = DateTime.Now;
                user.User.Update(p => p.LastFriendSyncSearch);

                new FriendListSearchWorker
                {
                    UserID = user.User.UserID,
                    Identity = request.Identity,
                    FriendsList = request.FriendsList
                }.Enqueue();

                FriendsStatsManager.Current.FriendSyncsByResult.Track((int)BasicServiceResponseStatus.Successful);
                return new BasicServiceResponse() { Status = BasicServiceResponseStatus.Successful };
            }
            catch (Exception ex)
            {
                FriendsStatsManager.Current.FriendSyncsByResult.Track((int)BasicServiceResponseStatus.Error);
                Logger.Error(ex, "[DeclineFriendSuggestion] Unhandled exception");
                return new BasicServiceResponse { Status = BasicServiceResponseStatus.Error };
            }
        }

        private static bool CanViewProfile(UserRegionInfo me, UserRegionInfo other)
        {
            if (me.User.UserID == other.User.UserID)
            {
                return true;
            }

            var suggestion = FriendSuggestion.Get(me.Region.RegionID, me.User.UserID, other.User.UserID);

            if (suggestion != null && suggestion.Status != FriendSuggestionStatus.Declined)
            {
                return true;
            }

            var friendship = Friendship.Get(me.Region.RegionID, me.User.UserID, other.User.UserID);

            if (friendship == null)
            {
                return false;
            }

            return friendship.Status == FriendshipStatus.AwaitingMe || friendship.Status == FriendshipStatus.Confirmed;
        }

        public UserProfileResponse GetUserProfile(int otherUserID)
        {
            try
            {
                // Get the user and region for both sides
                var me = GetCurrentUserAndRegion();
                var other = GetUserAndRegion(otherUserID);

                // Ensure that the requesting user is friends with the target user, or is a pending suggestion
                if (!CanViewProfile(me, other))
                {
                    return new UserProfileResponse {Status = BasicServiceResponseStatus.Forbidden};
                }

                var profile = DoGetUserProfile(otherUserID);

                if (profile.Status != BasicServiceResponseStatus.Successful)
                {
                    return profile;
                }

                if (otherUserID != me.User.UserID)
                {
                    profile.MutualFriends = User.GetMutualFriendIDs(me.User.UserID, otherUserID);
                }

                FriendsStatsManager.Current.ProfilesViewed.Track(otherUserID);

                return profile;
            }
            catch (InvalidOperationException ex)
            {
                Logger.Warn(ex, "[GetUserProfile] Unknown region!", otherUserID);
                return new UserProfileResponse { Status = BasicServiceResponseStatus.NotFound };
            }
            catch (UserNotFoundException)
            {
                return new UserProfileResponse { Status = BasicServiceResponseStatus.NotFound };
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "[GetUserProfile] Unhandled exception.");
                return new UserProfileResponse { Status = BasicServiceResponseStatus.Error };
            }
        }

        public UserProfileResponse GetUserInfo(int userID, string apiKey)
        {
            if (apiKey != FriendsServiceConfiguration.Instance.CentralServiceApiKey)
            {
                return new UserProfileResponse
                {
                    Status = BasicServiceResponseStatus.Forbidden
                };
            }

            return DoGetUserProfile(userID);
        }

        private UserProfileResponse DoGetUserProfile(int otherUserID)
        {
            try
            {
                // Get the user and region for both sides
                var other = GetUserAndRegion(otherUserID);

                // Get the target user's friend hints
                var hints = FriendHint.GetAllLocal(p => p.UserID, otherUserID)
                            .Where(p => (p.Type == FriendHintType.Game || p.Type == FriendHintType.Platform) && p.Status == FriendHintStatus.Normal
                                && (p.Visibility == FriendHintVisibility.VisibleToEveryone || p.Visibility == FriendHintVisibility.VisibleToFriends))
                            .ToArray();


                // Change out the display name to the appropriate public name
                foreach (var hint in hints)
                {
                    hint.DisplayName = hint.GetPublicName();
                }


                return new UserProfileResponse
                {
                    Status = BasicServiceResponseStatus.Successful,
                    AboutMe = other.User.AboutMe,
                    AvatarUrl = GetNewFriendAvatarUrl(other.User.UserID),
                    City = other.User.City,
                    CountryCode = other.User.CountryCode,
                    Name = other.User.Name,
                    State = other.User.State,
                    UserID = otherUserID,
                    Username = other.User.Username,
                    MutualFriends = new MutualFriend[0],
                    FriendCount = other.User.FriendCount,
                    LastGameID = other.User.LastGameID,
                    Identities = hints
                };
            }
            catch (UserNotFoundException)
            {
                return new UserProfileResponse { Status = BasicServiceResponseStatus.NotFound };
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "[GetUserProfile] Unhandled exception.");
                return new UserProfileResponse { Status = BasicServiceResponseStatus.Error };
            }
        }

        public void ReindexSearch(bool rebuildIndex)
        {
            try
            {
                Logger.Info("Received command to reindex search!");
                new FriendHintSearchIndexer { RebuildIndex = rebuildIndex }.Enqueue();
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to queue reindex search job!");
            }

        }

        /// <remarks>
        /// TODO: Deprecate/Remove this method.
        /// </remarks>
        public BasicServiceResponse RenameGroup(ChangeGroupRequest request)
        {
            return ChangeGroupInfo(request);
        }

        public BasicServiceResponse ChangeGroupInfo(ChangeGroupRequest request)
        {
            try
            {
                if (!request.Validate())
                {
                    Logger.Debug("[ChangeGroupInfo] Failed Validation: " + request.ValidationMessage, request);
                    return new BasicServiceResponse() { Status = BasicServiceResponseStatus.Invalid };
                }

                var group = Group.GetLocal(request.GroupID);
                if (group == null)
                {
                    return new BasicServiceResponse() { Status = BasicServiceResponseStatus.NotFound };
                }

                var authContext = AuthenticationContext.Current;
                group.ChangeInfo(authContext.UserID, request.Title, request.MessageOfTheDay,
                    request.AllowTemporaryChildGroups, request.ForcePushToTalk);
                return new BasicServiceResponse() { Status = BasicServiceResponseStatus.Successful };
            }
            catch (GroupPermissionException)
            {
                return new BasicServiceResponse { Status = BasicServiceResponseStatus.Forbidden };
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to change group!");
                return new BasicServiceResponse { Status = BasicServiceResponseStatus.Error };
            }
        }

        /// <summary>
        /// Adds the user list provided in the request from the group and notifies the current users that belong to group
        /// </summary>
        public BasicServiceResponse AddUsersToGroup(AddUserToGroupRequest request)
        {
            try
            {
                if (!request.Validate())
                {
                    Logger.Debug("[AddUsersToGroup] Failed Validation: " + request.ValidationMessage, request);
                    return new BasicServiceResponse() { Status = BasicServiceResponseStatus.Invalid };
                }

                var group = Group.GetLocal(request.GroupID);
                if (group == null)
                {
                    //if the requested group does not exist
                    return new BasicServiceResponse() { Status = BasicServiceResponseStatus.NotFound };
                }

                var authContext = AuthenticationContext.Current;
                var me = GetCurrentUserAndRegion();

                group.CheckPermission(GroupPermissions.InviteUsers, authContext.UserID);

                // Ensure that every user being added to the group is a friend:
                var filteredParticipants = Friendship.GetAllConfirmed(me.Region.RegionID, authContext.UserID)
                    .Where(p => request.UserIDs.Contains(p.OtherUserID))
                    .ToArray();

                //if users are not friends
                if (!filteredParticipants.Any())
                {
                    return new BasicServiceResponse() { Status = BasicServiceResponseStatus.Forbidden };
                }


                filteredParticipants = filteredParticipants.Where(p => !group.IsMember(p.OtherUserID)).ToArray();

                // Remove any users are already exist
                if (!filteredParticipants.Any())
                {
                    //If there are no users to add, do not service the request
                    return new BasicServiceResponse { Status = BasicServiceResponseStatus.Successful };
                }

                // Add the invited users
                group.AddUsers(authContext.UserID, filteredParticipants.Select(p => new NewGroupMember(p, group.DefaultRole)).ToArray());

                return new BasicServiceResponse() { Status = BasicServiceResponseStatus.Successful };
            }
            catch (GroupPermissionException ex)
            {
                Logger.Warn(ex, "Failed to add users to Group");
                return new BasicServiceResponse() { Status = BasicServiceResponseStatus.Forbidden };
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to add users to Group");
                return new BasicServiceResponse() { Status = BasicServiceResponseStatus.Error };
            }
        }

        /// <summary>
        /// Removes the user list provided in the request from the group and notifies the current users that belong to group
        /// </summary>
        public BasicServiceResponse RemoveUsersFromGroup(RemoveUsersFromGroupRequest request)
        {
            try
            {
                if (!request.Validate())
                {
                    Logger.Debug("[RemoveUsersFromGroup] Failed Validation: " + request.ValidationMessage, request);
                    return new BasicServiceResponse { Status = BasicServiceResponseStatus.Invalid };
                }
                var group = Group.GetLocal(request.GroupID);
                if (group == null)
                {
                    //if the requested group does not exist
                    return new BasicServiceResponse { Status = BasicServiceResponseStatus.NotFound };
                }

                var me = GetCurrentUserAndRegion();


                group.RemoveUsers(me.User.UserID, request.UserIDs);

                var removeUsersResult = RemoveUsersFromGroupCall(group, me.User.UserID, request.UserIDs.ToArray(), UserDisconnectReason.Kicked);
                if (removeUsersResult != RemoveUsersFromGroupCallStatus.Successful && removeUsersResult != RemoveUsersFromGroupCallStatus.NotFound)
                {
                    Logger.Warn("[RemoveUsersFromGroup] Failed to remove users from group call: " + removeUsersResult);
                }

                return new BasicServiceResponse { Status = BasicServiceResponseStatus.Successful };
            }
            catch (GroupPermissionException)
            {
                return new BasicServiceResponse { Status = BasicServiceResponseStatus.Forbidden };
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to remove users from Group");
                return new BasicServiceResponse { Status = BasicServiceResponseStatus.Error };
            }
        }

        private RemoveUsersFromGroupCallStatus RemoveUsersFromGroupCall(Group group, int kicker, int[] removedUsers, UserDisconnectReason reason)
        {
            if (group.VoiceSessionCode != null)
            {

                try
                {
                    var resp = CurseVoiceServiceClient.TryServiceCall("RemoveUsersFromGroupCall",
                        c => c.RemoveUsersFromGroupCall(FriendsServiceConfiguration.Instance.CentralServiceApiKey, OperationContext.Current.GetClientIPAddress(false),
                            group.GroupID, kicker, removedUsers, reason));

                    if (resp.Status != RemoveUsersFromGroupCallStatus.Successful)
                    {
                        Logger.Trace("Failed to remove users from group call: " + resp.Status);
                    }
                    else
                    {
                        Logger.Trace("Successfully removed users from active voice session.");
                    }

                    return resp.Status;
                }
                catch (Exception ex)
                {

                    Logger.Error(ex, "Exception occurred while trying to kick users from group call!");
                    return RemoveUsersFromGroupCallStatus.Error;
                }
            }
            return RemoveUsersFromGroupCallStatus.NotFound;
        }

        /// <summary>
        /// Creates a group with the details in CreateGroupRequest and adds users invited to group
        /// </summary>
        public CreateGroupResponse CreateGroup(CreateGroupRequest request)
        {
            try
            {
                if (!request.Validate())
                {
                    return new CreateGroupResponse()
                    {
                        Status = CreateGroupStatus.Error,
                        ErrorMessage = request.ValidationMessage
                    };
                }


                // See if this user is the owner of too many groups, or too recent of a group:            
                var me = GetCurrentUserAndRegion();

                if (FriendsServiceConfiguration.Mode == ConfigurationMode.Release)
                {
                    var memberships = GroupMember.GetAllByUserID(me.User.UserID);
                    if (memberships.Count(p => !p.IsDeleted && p.BestRoleRank == 0) >= Group.MaxServersOwned)
                    {
                        return new CreateGroupResponse { Status = CreateGroupStatus.Throttled };
                    }
                }


                // Ensure that every user being added to the group is a friend:
                var filteredParticipants = (request.RecipientsUserIDs != null)
                    ? Friendship.GetAllConfirmed(me.Region.RegionID, me.User.UserID)
                        .Where(p => request.RecipientsUserIDs.Contains(p.OtherUserID))
                        .ToArray()
                    : new Friendship[0];

                // If the user is trying to add any users to the group, ensure that they are valid
                if (request.RecipientsUserIDs != null && request.RecipientsUserIDs.Any() && !filteredParticipants.Any())
                {
                    return new CreateGroupResponse { Status = CreateGroupStatus.InvalidRecipients };
                }

                // Create the main group record
                var members = filteredParticipants.Select(p => new NewGroupMember(p)).ToArray();
                var creator = new NewGroupMember(me.Region, me.User);
                var newGroup = Group.CreateNormalGroup(creator, request.Title, members, "Owner", "Guest");

                FriendsStatsManager.Current.GroupsCreatedByType.Track((int)GroupType.Normal);

                return new CreateGroupResponse { Status = CreateGroupStatus.Successful, GroupID = newGroup.GroupID };

            }
            catch (GroupPermissionException)
            {
                return new CreateGroupResponse { Status = CreateGroupStatus.Forbidden };
            }
            catch (ArgumentException)
            {
                Logger.Warn("Failed to create group due to invalid request arguments.", request);
                return new CreateGroupResponse { Status = CreateGroupStatus.Error };
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to create group!", request);
                return new CreateGroupResponse { Status = CreateGroupStatus.Error };
            }
        }

        /// <summary>
        /// Removes the requested UserId from the groupID of LeaveGroupRequest
        /// </summary>
        public BasicServiceResponse LeaveGroup(LeaveGroupRequest request)
        {
            try
            {
                if (!request.Validate())
                {
                    Logger.Debug("[LeaveGroup] Failed Validation: " + request.ValidationMessage, request);
                    return new BasicServiceResponse() { Status = BasicServiceResponseStatus.Invalid };
                }

                var currentUser = GetCurrentUserAndRegion();

                var group = Group.GetLocal(request.GroupID);
                if (group == null || !group.IsRootGroup) // You can only leave root groups
                {
                    return new BasicServiceResponse() { Status = BasicServiceResponseStatus.NotFound };
                }

                group.Leave(currentUser.User.UserID, currentUser.Region.RegionID);

                var removeUsersResult = RemoveUsersFromGroupCall(group, currentUser.User.UserID, new[] { currentUser.User.UserID }, UserDisconnectReason.LeftGroup);
                if (removeUsersResult != RemoveUsersFromGroupCallStatus.Successful && removeUsersResult != RemoveUsersFromGroupCallStatus.NotFound)
                {
                    Logger.Warn("[RemoveUsersFromGroup] Failed to remove users from group call: " + removeUsersResult);
                }


                return new BasicServiceResponse() { Status = BasicServiceResponseStatus.Successful };
            }
            catch (GroupPermissionException)
            {
                return new BasicServiceResponse() { Status = BasicServiceResponseStatus.Forbidden };
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Error Leaving group ", request.GroupID.ToString());
                return new BasicServiceResponse() { Status = BasicServiceResponseStatus.Error };
            }
        }

        /// <summary>
        /// Marks the requested group as User's favorite and notifies all the connected Endpoints 
        /// about the change made
        /// </summary>
        /// <returns></returns>
        public BasicServiceResponse ToggleFavoriteGroup(FavoriteGroupRequest request)
        {
            try
            {
                if (!request.Validate())
                {
                    return new BasicServiceResponse() { Status = BasicServiceResponseStatus.Invalid };
                }

                var group = Group.GetLocal(request.GroupID);
                if (group == null)
                {
                    //If the requested group does not exist
                    return new BasicServiceResponse() { Status = BasicServiceResponseStatus.NotFound };
                }

                var authContext = AuthenticationContext.Current;
                var groupMembership = group.CheckPermission(GroupPermissions.Access, authContext.UserID);

                groupMembership.IsFavorite = request.IsFavorite;
                groupMembership.Update(p => p.IsFavorite);

                //fan out to all end points of user
                ClientEndpoint.DispatchNotification(authContext.UserID,
                    ep => GroupPreferenceChangedNotifier.Create(ep, groupMembership));
                return new BasicServiceResponse() { Status = BasicServiceResponseStatus.Successful };
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Error Making group favorite", request.GroupID.ToString());
                return new BasicServiceResponse() { Status = BasicServiceResponseStatus.Error };
            }
        }

        private static string GetNewGroupAvatarUrl(Guid groupID)
        {
#if CONFIG_STAGING
            return "https://avatars.curseapp.tech/groups/" + groupID;
#else
            return "https://avatars.curseapp.net/groups/" + groupID;
#endif
        }

        private static string GetNewFriendAvatarUrl(int userID)
        {
#if CONFIG_STAGING
            return "https://avatars.curseapp.tech/users/" + userID;
#else
            return "https://avatars.curseapp.net/users/" + userID;
#endif
        }

        /// <summary>
        /// Returns a summarical view of all of a users group memberships
        /// </summary>
        /// <returns></returns>
        public GetMyGroupsResponse GetMyGroups()
        {
            try
            {
                var authToken = AuthenticationContext.Current;

                // Group memberships
                var groupMemberships = GroupMember.GetDictionaryByUserID(authToken.UserID);


                // Group dictionary
                var groups = Group.GetDictionaryFromIDs(groupMemberships.Keys);


                var groupNotifications = groups.Values.Where(p => p.IsRootGroup)
                                               .Select(p => p.ToContactNotification(FriendsServiceConfiguration.Instance.GroupsRootUrl, groups, groupMemberships))
                                               .ToArray();
                
                return new GetMyGroupsResponse()
                {
                    Status = BasicServiceResponseStatus.Successful,
                    Groups = groupNotifications.Select(p => new GroupMembership
                    {
                        DateJoined = p.Membership.DateJoined,
                        UserID = authToken.UserID,
                        GroupID = p.GroupID,
                        RootGroupID = p.RootGroupID,
                        ParentGroupID = p.ParentGroupID,
                        IsFavorite = p.Membership.IsFavorite,
                        DateMessaged = p.Membership.DateMessaged.RoundToMillisecond(),
                        DateRead = p.Membership.DateRead.RoundToMillisecond().AddMilliseconds(100), // This is a hack...
                        GroupAvatarUrl = GetNewGroupAvatarUrl(p.GroupID),
                        GroupMemberCount = p.MemberCount,
                        GroupRole = p.Membership.BestRole == 1 ? LegacyGroupRole.Owner : LegacyGroupRole.Member,
                        GroupTitle = p.GroupTitle,
                        GroupType = GroupType.Normal,
                        IsCallActive = !string.IsNullOrEmpty(p.VoiceSessionCode),
                        NotificationPreference = p.Membership.NotificationPreference
                    }).ToArray()
                };
            }
            catch (UserNotFoundException)
            {
                return new GetMyGroupsResponse() { Status = BasicServiceResponseStatus.Forbidden };
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "[GetMyGroups] Unhandled exception");
                return new GetMyGroupsResponse() { Status = BasicServiceResponseStatus.Error };
            }
        }

        /// <summary>
        /// Returns a detailed group response, including the full member list
        /// </summary>
        public GetGroupDetailsResponse GetGroupDetails(GroupDetailsRequest request)
        {
            if (!request.Validate())
            {
                return new GetGroupDetailsResponse { Status = BasicServiceResponseStatus.Invalid };
            }

            try
            {
                var authContext = AuthenticationContext.Current;
                var group = Group.GetLocal(request.GroupID);
                if (group == null || group.Type == GroupType.Large)
                {
                    return new GetGroupDetailsResponse { Status = BasicServiceResponseStatus.NotFound };
                }

                if (!group.IsRootGroup)
                {
                    return new GetGroupDetailsResponse { Status = BasicServiceResponseStatus.Invalid };
                }

                // Ensure the user has access
                group.CheckPermission(GroupPermissions.Access, authContext.UserID);

                var resp = new GetGroupDetailsResponse { Status = BasicServiceResponseStatus.Successful };
                var members = group.GetAllMembers(true).Where(p => !p.IsDeleted).Select(p => p.ToNotification(null)).ToArray();
                resp.Groups = new[] { group.ToNotification("") };
                resp.Members = members.Select(p => new GroupMemberNotification
                {
                    UserID = p.UserID,
                    CurrentGroupID = Guid.Empty, 
                    Username = p.Username,
                    Role = p.BestRole == 1 ? LegacyGroupRole.Owner : LegacyGroupRole.Member
                }).ToArray();

                return resp;
            }
            catch (GroupPermissionException)
            {
                return new GetGroupDetailsResponse { Status = BasicServiceResponseStatus.Forbidden };
            }
            catch (UserNotFoundException)
            {
                return new GetGroupDetailsResponse { Status = BasicServiceResponseStatus.Forbidden };
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "[GetGroupDetails] Unhandled exception");
                return new GetGroupDetailsResponse { Status = BasicServiceResponseStatus.Error };
            }

        }

        /// <summary>
        /// Changes notification perference for a user to a group. Adds filters for notification.
        /// </summary>
        public BasicServiceResponse ChangeGroupNotificationPreferences(ChangeGroupNotificationPreferencesRequest request)
        {
            if (!request.Validate())
            {
                return new BasicServiceResponse { Status = BasicServiceResponseStatus.Invalid };
            }

            try
            {
                var authContext = AuthenticationContext.Current;

                var group = Group.GetLocal(request.GroupID);
                if (group == null)
                {
                    return new BasicServiceResponse { Status = BasicServiceResponseStatus.NotFound };
                }

                var myMembership = group.CheckPermission(GroupPermissions.Access, authContext.UserID);


                myMembership.UpdateUserPreference(request.Preference, request.FilterSet);

                //fan out to all endpoints of user
                ClientEndpoint.DispatchNotification(authContext.UserID,
                    ep => GroupPreferenceChangedNotifier.Create(ep, myMembership));

                return new BasicServiceResponse { Status = BasicServiceResponseStatus.Successful };
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "[ChangeUserNotificationPreference] Unhandled exception");
                return new BasicServiceResponse { Status = BasicServiceResponseStatus.Error };
            }
        }




        //public BasicServiceResponse KickGuestsFromGroupCall(KickGuestsFromGroupCallRequest request)
        //{
        //    try
        //    {
        //        if (!request.Validate())
        //        {
        //            return new BasicServiceResponse
        //            {
        //                Status = BasicServiceResponseStatus.Invalid,
        //                StatusMessage = request.ValidationMessage
        //            };
        //        }

        //        var group = Group.GetLocal(request.GroupID);
        //        if (group == null)
        //        {
        //            return new BasicServiceResponse { Status = BasicServiceResponseStatus.NotFound };
        //        }

        //        var requestorID = AuthenticationContext.Current.UserID;
        //        group.CheckRemoveGuestPermission(requestorID);

        //        var guestsToRemove = group.MemberList.Members.GetValues()
        //            .Where(m => m.Role == LegacyGroupRole.Guest && m.CurrentGroupID == request.GroupID)
        //            .Select(m => m.UserID)
        //            .ToArray();


        //        switch (RemoveUsersFromGroupCall(group, requestorID, guestsToRemove, UserDisconnectReason.Kicked))
        //        {
        //            case RemoveUsersFromGroupCallStatus.NotFound:
        //                return new BasicServiceResponse { Status = BasicServiceResponseStatus.NotFound };

        //            case RemoveUsersFromGroupCallStatus.Forbidden:
        //                return new BasicServiceResponse { Status = BasicServiceResponseStatus.Forbidden };
        //            case RemoveUsersFromGroupCallStatus.Error:
        //                return new BasicServiceResponse { Status = BasicServiceResponseStatus.Error };
        //        }

        //        if (request.RemoveKickedGuestsFromClan)
        //        {
        //            group.RootGroup.RemoveUsers(requestorID, new HashSet<int>(guestsToRemove));
        //        }

        //        return new BasicServiceResponse { Status = BasicServiceResponseStatus.Successful };
        //    }
        //    catch (GroupPermissionException)
        //    {
        //        return new BasicServiceResponse { Status = BasicServiceResponseStatus.Forbidden };
        //    }
        //    catch (Exception ex)
        //    {
        //        Logger.Error(ex, "Failed to kick guests from the group call.");
        //        return new BasicServiceResponse { Status = BasicServiceResponseStatus.Error };
        //    }
        //}

        #region Voice Calls

        /// <summary>
        /// Creates a group with the details in CreateGroupRequest and adds users invited to group
        /// </summary>
        public VoiceSessionResponse CallGroup(CallGroupRequest request)
        {
            try
            {
                if (!request.Validate())
                {
                    return new VoiceSessionResponse() { Status = VoiceSessionRequestStatus.Invalid };
                }

                var authContext = AuthenticationContext.Current;

                var group = Group.GetLocal(request.GroupID);

                if (group == null)
                {
                    // If the group does not exist return
                    return new VoiceSessionResponse { Status = VoiceSessionRequestStatus.Forbidden };
                }

                // Ensure the user has permission to do this
                group.CheckPermission(GroupPermissions.Access, authContext.UserID);

                var resp = CurseVoiceServiceClient.TryServiceCall("GroupVoiceSession", c => c.GroupVoiceSessionV2(FriendsServiceConfiguration.Instance.CentralServiceApiKey,
                        OperationContext.Current.GetClientIPAddress(false), Version.Parse(request.ClientVersion),
                        authContext.UserID, group.Title, group.GroupID, request.Force, group.RootGroup.VoiceRegionID));


                if (resp.Status != CreateVoiceSessionStatus.Successful || resp.VoiceInstance == null)
                {
                    Logger.Warn("Failed to call group!", new { resp });
                    return new VoiceSessionResponse { Status = VoiceSessionRequestStatus.Error };
                }

                if (group.VoiceSessionCode != resp.VoiceInstance.InviteCode)
                {
                    group.VoiceSessionCode = resp.VoiceInstance.InviteCode;
                    group.Update(p => p.VoiceSessionCode);
                }

                group.CheckPushToTalkThreshold();
                var callDetails = FromVoiceInstanceDetails(resp.VoiceInstance, group);

                if (request.SendInvitation && group.MemberCount < Group.MaxUsersForCallNotification)
                {
                    try
                    {
                        FriendsStatsManager.Current.GroupVoiceInvitationsSent.Track();
                    }
                    catch (Exception ex)
                    {
                        Logger.Error(ex, "[CallGroup] Failed to track stat!");
                    }

                    
                    GroupCallCoordinator.CallStarted(AuthenticationContext.Current.UserID, group, callDetails);
                }

                return new VoiceSessionResponse
                {
                    Status = VoiceSessionRequestStatus.Successful,
                    AccessToken = AccessTokenHelper.CreateAccessToken(resp.VoiceInstance.CallID, AuthenticationContext.Current.UserID),
                    InviteUrl = callDetails.InviteUrl
                };

            }
            catch (GroupPermissionException ex)
            {
                Logger.Trace(ex, "User attempted to call a group without permission.");
                return new VoiceSessionResponse { Status = VoiceSessionRequestStatus.Forbidden };
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to call group");
                return new VoiceSessionResponse { Status = VoiceSessionRequestStatus.Error };
            }
        }

        private VoiceSessionResponse DoCallFriend(Friendship friendship, CallFriendRequest request)
        {

            var userAndRegion = GetCurrentUserAndRegion();


            // If this call includes an invite, add the pending user to the call
            PendingUserRequest pendingUser = null;
            if (request.SendInvitation)
            {
                pendingUser = new PendingUserRequest
                {
                    UserID = friendship.OtherUserID,
                    DisplayName = friendship.OtherUsername,
                    AvatarUrl = string.Format(FriendsServiceConfiguration.Instance.AvatarUrlFormat, friendship.AvatarUrlSlug, friendship.AvatarUrlID)
                };
            }

            // Make an API call to the central voice server
            var resp = CurseVoiceServiceClient.TryServiceCall("FriendVoiceSession",
                c => c.FriendVoiceSessionV2(FriendsServiceConfiguration.Instance.CentralServiceApiKey,
                    OperationContext.Current.GetClientIPAddress(false),
                    Version.Parse(request.ClientVersion),
                    userAndRegion.User.UserID,
                    userAndRegion.User.Username,
                    friendship.OtherUserID,
                    request.Force,
                    pendingUser)
                );

            switch (resp.Status)
            {
                case CreateVoiceSessionStatus.IncompatibleClient:
                    return new VoiceSessionResponse { Status = VoiceSessionRequestStatus.IncompatibleClient };
                case CreateVoiceSessionStatus.Throttled:
                    return new VoiceSessionResponse { Status = VoiceSessionRequestStatus.Throttled };
                case CreateVoiceSessionStatus.Error:
                case CreateVoiceSessionStatus.NoHostsAvailable:
                    return new VoiceSessionResponse { Status = VoiceSessionRequestStatus.NoHostsAvailable };
            }

            var callDetails = FromVoiceInstanceDetails(resp.VoiceInstance, friendship);

            if (request.SendInvitation)
            {
                try
                {
                    FriendsStatsManager.Current.VoiceInvitationsSent.Track();
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "[CallFriend] Failed to track stat!");
                }

                CallResolver.Create(AuthenticationContext.Current.UserID, AuthenticationContext.Current.Username, friendship.OtherUserID, callDetails);
            }

            return new VoiceSessionResponse
            {
                AccessToken = AccessTokenHelper.CreateAccessToken(resp.VoiceInstance.CallID, AuthenticationContext.Current.UserID),
                InviteUrl = resp.VoiceInstance.InviteUrl,
                Status = VoiceSessionRequestStatus.Successful
            };
        }

        private static CallType FromVoiceInstanceType(VoiceInstanceType type)
        {
            switch (type)
            {
                case VoiceInstanceType.AdHoc:
                    return CallType.AdHoc;
                case VoiceInstanceType.AutoMatch:
                    return CallType.AutoMatch;
                case VoiceInstanceType.Friend:
                    return CallType.Friend;
                case VoiceInstanceType.Group:
                    return CallType.Group;
                case VoiceInstanceType.MultiFriend:
                    return CallType.MultiFriend;
                default:
                    Logger.Warn("Unknown voice instance type, assuming ad hoc.", new { VoiceInstanceType = (int)type });
                    return CallType.AdHoc;
            }
        }

        private static CallDetails FromVoiceInstanceDetails(VoiceInstanceDetailsResponse response, IConversationContainer conversation)
        {
            return new CallDetails
            {
                HostName = response.HostName,
                InviteUrl = response.InviteUrl,
                CallID = response.CallID,
                Type = FromVoiceInstanceType(response.Type),
                ConversationID = conversation != null ? conversation.ConversationID : null,
                Timestamp = DateTime.UtcNow,
                HostID = response.HostID,
                RegionName = response.RegionName,
                GameID = response.GameID,
                CreatorName = response.CreatorName,
                AutoMatchKey = response.AutoMatchKey,
                IpAddress = response.IpAddress,
                InviteCode = response.InviteCode,
                CreatorID = response.UserID
            };
        }

        public VoiceSessionResponse CallFriend(CallFriendRequest request)
        {
            try
            {
                if (!request.Validate())
                {
                    return new VoiceSessionResponse() { Status = VoiceSessionRequestStatus.Invalid };
                }

                var userAndRegion = GetCurrentUserAndRegion();

                var them = Friendship.Get(userAndRegion.Region.RegionID, userAndRegion.User.UserID, request.FriendID);

                // Confirm that the target recipient is a confirmed friend
                if (them == null || them.Status != FriendshipStatus.Confirmed)
                {
                    return new VoiceSessionResponse { Status = VoiceSessionRequestStatus.Forbidden };
                }

                return DoCallFriend(them, request);

            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to call friend");
                return new VoiceSessionResponse { Status = VoiceSessionRequestStatus.Error };
            }
        }

        private VoiceSessionResponse GetVoiceSessionResponse(CreateVoiceSessionResponse resp, AuthenticationToken authToken)
        {
            switch (resp.Status)
            {
                case CreateVoiceSessionStatus.AlreadyExists:
                // Fall-through - already exists is still a success
                case CreateVoiceSessionStatus.Successful:
                    return new VoiceSessionResponse
                    {
                        Status = VoiceSessionRequestStatus.Successful,
                        InviteUrl = resp.InviteUrl,
                        AccessToken = AccessTokenHelper.CreateAccessToken(resp.InstanceCode, authToken.UserID)
                    };
                case CreateVoiceSessionStatus.IncompatibleClient:
                    return new VoiceSessionResponse { Status = VoiceSessionRequestStatus.IncompatibleClient };
                case CreateVoiceSessionStatus.Throttled:
                    return new VoiceSessionResponse { Status = VoiceSessionRequestStatus.Throttled };
                case CreateVoiceSessionStatus.NoHostsAvailable:
                    return new VoiceSessionResponse { Status = VoiceSessionRequestStatus.NoHostsAvailable };
                default:
                    return new VoiceSessionResponse { Status = VoiceSessionRequestStatus.Error };
            }
        }

        public BasicServiceResponse AddFriendToCall(AddFriendToCallRequest request)
        {
            try
            {
                var me = GetUserAndRegion(AuthenticationContext.Current.UserID);
                var them = Friendship.Get(me.Region.RegionID, me.User.UserID, request.FriendID);

                // Confirm that the target recipient is a confirmed friend
                if (them == null || them.Status != FriendshipStatus.Confirmed)
                {
                    return new BasicServiceResponse
                    {
                        Status = BasicServiceResponseStatus.Forbidden,
                    };
                }

                // Unlock the session (may already be unlocked, but we don't care)
                var unlockResp = UnlockVoiceSession(request.CallCode);

                if (unlockResp.Status != UnlockVoiceSessionStatus.Successful)
                {
                    Logger.Warn("[AddFriendToCall] Failed to unlock voice session when adding a friend to a call.", new { request, unlockResp });
                    return new BasicServiceResponse
                    {
                        Status = BasicServiceResponseStatus.Error,
                    };
                }


                try
                {
                    FriendsStatsManager.Current.VoiceInvitationsSent.Track();
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "[AddFriendToCall] Failed to track stat!");
                }

                // Now let our voice host know that we have a pending user (if this fails, log it, but still send the invite)
                try
                {
                    var pendingUser = new PendingUserRequest
                    {
                        UserID = them.OtherUserID,
                        DisplayName = them.OtherUsername,
                        AvatarUrl = GetNewFriendAvatarUrl(them.OtherUserID)
                    };

                    // Add pending user to the call
                    var pendingResp = CurseVoiceServiceClient.TryServiceCall("AddPendingVoiceUsers", c => c.AddPendingVoiceUsersV2(FriendsServiceConfiguration.Instance.CentralServiceApiKey,
                                                                        OperationContext.Current.GetClientIPAddress(),
                                                                        request.CallCode,
                                                                        AuthenticationContext.Current.UserID,
                                                                        new[] { pendingUser }));

                    if (pendingResp.Status != BasicVoiceServiceStatus.Successful || pendingResp.VoiceDetails == null)
                    {
                        Logger.Warn("[AddFriendToCall] Unsuccessful attempt to add pending users to voice host: " + pendingResp.Status, new { pendingResp, request.CallCode });
                    }

                    CallResolver.Create(AuthenticationContext.Current.UserID, AuthenticationContext.Current.Username, request.FriendID, FromVoiceInstanceDetails(pendingResp.VoiceDetails, them));
                }
                catch (Exception ex)
                {

                    Logger.Error(ex, "[AddFriendToCall] Failed to add pending users to voice host.");
                }


                return new BasicServiceResponse
                {
                    Status = BasicServiceResponseStatus.Successful,
                };
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to add friend to call");
                return new BasicServiceResponse
                {
                    Status = BasicServiceResponseStatus.Error,
                };
            }
        }

        public RespondToCallResponse RespondToCall(RespondToCallRequest request)
        {
            var response = new RespondToCallResponse
            {
                Timestamp = DateTime.UtcNow,
                Status = RespondToCallStatus.Error
            };

            if (!request.Validate())
            {
                response.Status = RespondToCallStatus.Invalid;
                return response;
            }

            try
            {
                var me = GetCurrentUserAndRegion();

                if (request.Accepted)
                {
                    response.Status = AcceptCall(me, request.InviteUrl, request.GroupID, request.FriendID);
                }
                else
                {
                    response.Status = DeclineCall(me, request.InviteUrl, request.GroupID, request.FriendID);
                }

                return response;
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "An error occurred while declining a call.");
                return response;
            }
        }

        private class WrappedCallDetails
        {
            public CallDetails CallDetails { get; private set; }
            public IConversationContainer Conversation { get; private set; }

            public WrappedCallDetails(VoiceInstanceDetailsResponse response, IConversationContainer conversation)
            {
                CallDetails = FromVoiceInstanceDetails(response, conversation);
                Conversation = conversation;
            }
        }

        private WrappedCallDetails GetWrappedCallDetailsByInviteCode(string inviteCode)
        {

            if (inviteCode.StartsWith("http://"))
            {
                var parsedCode = inviteCode.Substring(inviteCode.LastIndexOf("/"));
                Logger.Info("parsing invote code from url", new { inviteCode, parsedCode });
                inviteCode = parsedCode;
            }

            // Get the call details
            var voiceInstance = CurseVoiceServiceClient.TryServiceCall("GetVoiceInstanceDetails",
                    c => c.FindVoiceSessionV2(FriendsServiceConfiguration.Instance.CentralServiceApiKey, inviteCode));

            if (voiceInstance == null || voiceInstance.Status == VoiceInstanceStatus.Shutdown)
            {
                return null;
            }

            var conversation = ConversationManager.GetConversationContainer(AuthenticationContext.Current.UserID, voiceInstance.ExternalID);

            if (conversation == null)
            {
                return null;
            }

            return new WrappedCallDetails(voiceInstance, conversation);
        }

        private void RespondToCall(WrappedCallDetails wrappedCall, CallResponseReason reason)
        {
            var callRespondedNotification = new CallRespondedNotification
            {
                CallID = wrappedCall.CallDetails.CallID,
                Reason = reason,
                Timestamp = DateTime.UtcNow,
                UserID = AuthenticationContext.Current.UserID,
                Username = AuthenticationContext.Current.Username
            };

            if (wrappedCall.Conversation is Group)
            {
                GroupCallCoordinator.CallResponded((Group)wrappedCall.Conversation, wrappedCall.CallDetails, callRespondedNotification);
            }
            else if (wrappedCall.Conversation is Friendship)
            {
                CallRespondedResolver.Create(AuthenticationContext.Current.UserID, callRespondedNotification);
            }
        }

        private RespondToCallStatus DeclineCall(UserRegionInfo me, string inviteUrl, Guid? groupID, int? friendID)
        {

            var wrappedDetails = GetWrappedCallDetailsByInviteCode(inviteUrl);
            if (wrappedDetails == null)
            {
                return RespondToCallStatus.NotFound;
            }

            if (!wrappedDetails.Conversation.CanCall(AuthenticationContext.Current.UserID))
            {
                return RespondToCallStatus.Forbidden;
            }

            // Track the stats
            if (wrappedDetails.Conversation is Group)
            {
                FriendsStatsManager.Current.GroupVoiceInvitationsDeclined.Track();
            }
            else if (wrappedDetails.Conversation is Friendship)
            {
                FriendsStatsManager.Current.VoiceInvitationsDeclined.Track();
            }

            RespondToCall(wrappedDetails, CallResponseReason.Declined);

            // Remove pending user from voice host
            try
            {
                var removeResponse = CurseVoiceServiceClient.TryServiceCall("RemovePendingVoiceUser",
                        c => c.RemovePendingVoiceUser(FriendsServiceConfiguration.Instance.CentralServiceApiKey,
                                                       OperationContext.Current.GetClientIPAddress(),
                                                      wrappedDetails.CallDetails.InviteCode,
                                                      AuthenticationContext.Current.UserID));

                if (removeResponse.Status == BasicVoiceServiceStatus.Error || removeResponse.Status == BasicVoiceServiceStatus.Failed || removeResponse.Status == BasicVoiceServiceStatus.Forbidden)
                {
                    Logger.Warn("[DeclineCall] Unsuccessful attempt to remove pending user from voice host: " + removeResponse.Status, new { removeResponse });
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "[DeclineCall] Failed to remove pending user from voice host.");
            }

            return RespondToCallStatus.Successful;
        }

        private RespondToCallStatus AcceptCall(UserRegionInfo me, string inviteUrl, Guid? groupID, int? friendID)
        {
            var callParent = GetWrappedCallDetailsByInviteCode(inviteUrl);
            if (callParent == null)
            {
                return RespondToCallStatus.NotFound;
            }

            if (!callParent.Conversation.CanCall(AuthenticationContext.Current.UserID))
            {
                return RespondToCallStatus.Forbidden;
            }

            RespondToCall(callParent, CallResponseReason.Accepted);

            return RespondToCallStatus.Successful;
        }

        private UnlockVoiceSessionResponse UnlockVoiceSession(string sessionGuid)
        {
            try
            {

                return CurseVoiceServiceClient.TryServiceCall("UnlockVoiceSession", c => c.UnlockVoiceSession(FriendsServiceConfiguration.Instance.CentralServiceApiKey,
                        OperationContext.Current.GetClientIPAddress(), sessionGuid,
                        AuthenticationContext.Current.UserID));

            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to unlock voice session.");
                return new UnlockVoiceSessionResponse { Status = UnlockVoiceSessionStatus.Error };
            }

        }

        public BasicServiceResponse UnlockCall(UnlockCallRequest request)
        {
            try
            {
                if (!request.Validate())
                {
                    return new BasicServiceResponse
                    {
                        Status = BasicServiceResponseStatus.Invalid,
                        StatusMessage = request.ValidationMessage
                    };
                }

                var resp = UnlockVoiceSession(request.CallCode);

                switch (resp.Status)
                {
                    case UnlockVoiceSessionStatus.Successful:
                        return new BasicServiceResponse { Status = BasicServiceResponseStatus.Successful };
                    case UnlockVoiceSessionStatus.NotFound:
                        return new BasicServiceResponse { Status = BasicServiceResponseStatus.NotFound };
                    case UnlockVoiceSessionStatus.Forbidden:
                        return new BasicServiceResponse { Status = BasicServiceResponseStatus.Forbidden };
                    default:
                        return new BasicServiceResponse { Status = BasicServiceResponseStatus.Error };
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to unlock a call.");
                return new BasicServiceResponse { Status = BasicServiceResponseStatus.Error };
            }
        }

        #endregion

#if DEBUG_JSON
        public string HelloWorld()
        {
            return "Hello World";
        }

        public string[] TestLists(int numberOfElements)
        {
            var list = new List<string>();
            for(var i = 0; i < numberOfElements; i++)
            {
                list.Add(i.ToString());
            }

            return list.ToArray();
        }

        public ComplexType TestObject(int someNumber, string someString)
        {
            return new ComplexType { SomeNumber = someNumber, SomeText = someString };
        }


        public ComplexType[] TestObjectList(int numberOfElements)
        {
            var list = new List<ComplexType>();
            for (var i = 0; i < numberOfElements; i++)
            {
                list.Add(new ComplexType { SomeNumber = i, SomeText = "My String " + i });
            }

            return list.ToArray();            
        }

#endif
        public string HealthCheck()
        {
            return "I am alive and well in " + Curse.Friends.Configuration.FriendsServiceConfiguration.Instance.LastKnownRegion + "!";
        }

    }
}