﻿using Curse.Aerospike;
using Curse.Friends.Data;
using Curse.Friends.Enums;
using Curse.Logging;
using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Linq;
using System.Threading;

namespace Curse.Friends.WebService
{
    /// <summary>
    /// updates all available hostIps constantly scanning Notificationhost
    /// </summary>
    public static class NotificationHostManager
    {
        private static readonly ConcurrentDictionary<string, NotificationHost> _hosts = new ConcurrentDictionary<string, NotificationHost>();
        private static string[] _allHostIPs;
        private static readonly HostEnvironment _hostEnvironment = HostEnvironment.Unknown; 

        static NotificationHostManager()
        {

#if CONFIG_DEBUG
            _hostEnvironment = HostEnvironment.Debug;
#elif CONFIG_STAGING
            _hostEnvironment = HostEnvironment.Staging;
#elif CONFIG_RELEASE
            _hostEnvironment = HostEnvironment.Release;
#endif
            try
            {
                UpdateHostList();

            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to setup initial notification host list.");
            }

            new Thread(UpdateHostListThread) { Name = "UpdateHostListThread", IsBackground = true }.Start();            
        }

        public static void Initialize() {  }

        /// <summary>
        ///updates the host list every 10 seconds in a thread
        /// </summary>
        private static void UpdateHostListThread()
        {
            while(true)
            {
                try
                {
                    Thread.Sleep(10000);
                    UpdateHostList();
                }
                catch (ThreadAbortException)
                {
                    return;
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to update notification host list.");
                }
            }
        }

        /// <summary>
        /// Adds new hosts to the list
        /// </summary>
        private static void UpdateHostList()
        {
            var allHosts = NotificationHost.GetAllLocal(p => p.IndexMode, IndexMode.Default).Where(p => p.Environment == HostEnvironment.Unknown || p.Environment == _hostEnvironment).ToArray();

            if (allHosts.Any(p => p.Version == p.CurrentVersion))
            {
                allHosts = allHosts.Where(p => p.Version == p.CurrentVersion).ToArray();
            }

#if DEBUGLOCAL
            allHosts = allHosts.Where(p => p.MachineName == Environment.MachineName).ToArray();
#endif

            var allHostIPs = new HashSet<string>(allHosts.Where(p => !string.IsNullOrEmpty(p.InternalIPAddress)).Select(p => p.InternalIPAddress));

            foreach (var host in allHosts)
            {

                if (string.IsNullOrEmpty(host.MachineName) || string.IsNullOrEmpty(host.PublicIPAddress) || string.IsNullOrEmpty(host.InternalIPAddress))
                {
#if !CONFIG_LOAD_TESTING
                    Logger.Warn("Skipping invalid notification host!", host);
#endif
                    continue;                    
                }

                NotificationHost found;
                if (!_hosts.TryGetValue(host.InternalIPAddress, out found))
                {
                    _hosts.TryAdd(host.InternalIPAddress, host);                    
                }
                else
                {
                    found.PublicIPAddress = host.PublicIPAddress;
                }
            }

            var missingKeys = _hosts.Keys.Where(p => !allHostIPs.Contains(p));
            foreach(var key in missingKeys)
            {
                NotificationHost removedHost = null;
                if(_hosts.TryRemove(key, out removedHost))
                {                    
                    Logger.Info("Removed missing notification host: " + removedHost.InternalIPAddress);
                }
                else 
                { 
                    Logger.Info("Unable to remove missing notification host: " + key);
                }

            }

            _allHostIPs = _hosts.Select(p => p.Value.PublicIPAddress).ToArray();
            
        }               

        public static string[] GetHostList()
        {
#if DEBUGLOCAL
            return new[] { "127.0.0.1" };
#else
            return _allHostIPs;
#endif            
            //lock (_syncRoot)
            //{


            //    if (_hostDistribution.Count == 0)
            //    {
            //        return new string[0];
            //    }

            //    ++CurrentHostIndex;

            //    if (CurrentHostIndex >= _hostDistribution.Count)
            //    {
            //        CurrentHostIndex = 0;                    
            //    }

            //    return _hostDistribution[CurrentHostIndex];            
            //}
        }
    }
}