﻿using Curse.Friends.Configuration;
using Curse.Friends.Statistics;
using Curse.Friends.Statistics.Models;
using Curse.Logging;
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Diagnostics;
using System.Linq;
using System.Net;
using System.ServiceProcess;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using Curse.CloudQueue;
using Curse.Friends.ServiceClients;
using Curse.Logging.Uploader;

namespace Curse.Friends.WorkerService
{
    public partial class CurseWorkerService : ServiceBase
    {
        public CurseWorkerService()
        {
            InitializeComponent();
        }

        public void OnDebugStart()
        {
            OnStart(null);

        }

        public void OnDebugStop()
        {
            OnStop();
        }
                
        protected override void OnStart(string[] args)
        {          
            var sw = Stopwatch.StartNew();
            EventLog.WriteEntry("Worker Service Starting", EventLogEntryType.Information);

            Logger.Init(new LoggerConfig(@"C:\Curse\Logs") {LogRetainCount = 100, MaxLogSize = Int32.MaxValue});

            LogUploader.Initialize((int)ServiceHostType.FriendsWorkerService, FriendsServiceConfiguration.Instance.CentralLogServiceUrl, FriendsServiceConfiguration.Instance.LoggingServiceApiKey);

            Logger.Info("Worker Service Starting");

             // Add the event handler for handling non-UI thread exceptions to the event. 
            AppDomain.CurrentDomain.UnhandledException += CurrentDomain_UnhandledException;

            // This is done async, so that a lengthy startup time doesn't pwn the Windows service layer
            Task.Factory.StartNew(() => ReallyStartService(sw));
        }

        IEnumerable<FriendsQueueStats> GetQueueStats()
        {
            return CloudQueueInstanceManager.GetQueueStats().Select(p => new FriendsQueueStats
            {
                QueueName = p.QueueName,
                TotalMessagesProcessed = p.TotalMessagesProcessed,
                CurrentlyProcessing = p.CurrentlyProcessing,
                AverageProcessingTimeMs = (int)p.AverageProcessTime.TotalMilliseconds,
                BestProcessingTimeMs = (int)p.BestProcessTime.TotalMilliseconds,
                WorstProcessingTimeMs = (int)p.WorstProcessTime.TotalMilliseconds,
                AverageQueuedTimeMs = (int)p.AverageQueuedTime.TotalMilliseconds,
                BestQueuedTimeMs = (int)p.BestQueuedTime.TotalMilliseconds,
                WorstQueuedTimeMs = (int)p.WorstQueuedTime.TotalMilliseconds
            }).ToArray();
        }

        void ReallyStartService(Stopwatch sw)
        {            
            try
            {
                EventLog.WriteEntry("Initializing Configuration", EventLogEntryType.Information);

                Logger.Info("Determining current region...");
                var currentRegion = StorageConfiguration.GetCurrentRegion();

                if (currentRegion == null)
                {
                    Logger.Error("Failed to determine current region!");
                }
                else
                {
                    Logger.Info("Current region detected: " + currentRegion.Key);
                }

                Logger.Info("Initializing service client...");
                FriendsServiceClients.Initialize(FriendsServiceConfiguration.Instance.EncryptionKey, FriendsServiceConfiguration.Instance.EncryptionIterations, FriendsServiceConfiguration.Instance.CentralServiceApiKey);

                FriendsStatsManager.Initialize((int)ServiceHostType.FriendsWorkerService, ServiceHostType.FriendsWorkerService.ToString(), currentRegion != null ? currentRegion.ID : 1, GetQueueStats);
                FriendsStatsManager.BeginStartup();                

                StorageConfiguration.Initialize("WorkerService");
                
                EventLog.WriteEntry("Worker Server Starting", EventLogEntryType.Information);
                WorkerServer.Start();
                FriendsStatsManager.Started();
            }
            catch (Exception ex)
            {
                EventLog.WriteEntry("Failed to start: " + ex.Message, EventLogEntryType.Error);
                Logger.Error(ex, "Notification has failed to start, due to an unhandled exception.");
                throw;
            }

            EventLog.WriteEntry("Worker Service Started", EventLogEntryType.Information);

            sw.Stop();
            Logger.Info("Worker Service Started in " + sw.Elapsed.TotalSeconds.ToString("###,##0.00") + " seconds");
            
        }

        void CurrentDomain_UnhandledException(object sender, UnhandledExceptionEventArgs e)
        {
            var exception = e.ExceptionObject as Exception;
            Logger.Error(exception, "Service has crashed!");
        }

        protected override void OnStop()
        {
            try
            {
                base.RequestAdditionalTime(30 * 1000);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to request additional time.");
            }

            try
            {
                FriendsStatsManager.BeginShutdown();
            }
            catch (Exception ex)
            {

                Logger.Error(ex);
            }

            try
            {
                WorkerServer.Stop();
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to cleanly stop service.");
            }

            try
            {
                FriendsStatsManager.Shutdown();
            }
            catch (Exception ex)
            {

                Logger.Error(ex);
            }
        }
    }
}
