﻿using Curse.Aerospike;
using Curse.Extensions;
using Curse.Friends.Configuration;
using Curse.Friends.Data;
using Curse.Friends.TwitchApi;
using Curse.Friends.WorkerService.Configuration;
using Curse.Logging;
using System;
using System.Collections.Generic;
using System.Linq;

namespace Curse.Friends.WorkerService.Partners
{
    public static class PartnerServerJoinedProcessor
    {
        private const string _fallbackLanguage = "en";

        private static Group _group;
        private static GroupRole _staffRole;
        private static Tuple<PartnerServerLanguage,GroupRole> _defaultRole;
        private static Dictionary<string, GroupRole> _languageRoleLookup;
        private static TwitchApiHelper _client;

        private static DateTime _lastRoleUpdateTime;

        public static void Initialize()
        {
            RefreshRoles();
        }

        private static void RefreshRoles()
        {
            var config = PartnerServerConfiguration.Instance;

            _group = Group.GetWritable(config.GroupID);
            if (_group == null)
            {
                Logger.Warn("Partner Server not found!");
                return;
            }

            _staffRole = GroupRole.Get(_group.RegionID, _group.GroupID, config.StaffRoleID);
            if (_staffRole == null)
            {
                Logger.Warn("Partner Server staff role not found!");
            }

            var lookup = config.Languages.ToDictionary(l => l.GroupRoleID);
            var roles = GroupRole.MultiGet(_group.RegionID, lookup.Keys.Select(id => new KeyInfo(_group.GroupID, id)));
            var languageRoleLookup = new Dictionary<string, GroupRole>();

            foreach (var role in roles)
            {
                if (role.IsDeleted)
                {
                    Logger.Warn($"Partner role {role.Name} is deleted");
                    continue;
                }

                PartnerServerLanguage language;
                if (!lookup.TryGetValue(role.RoleID, out language))
                {
                    Logger.Warn($"Unable to find language for role {role.Name}");
                    continue;
                }

                if (language.IsDefault)
                {
                    _defaultRole = Tuple.Create(language, role);
                }

                languageRoleLookup[language.LanguageCode] = role;
            }

            _languageRoleLookup = languageRoleLookup;

            if (roles.Count != lookup.Count || roles.Any(r => r.IsDeleted))
            {
                Logger.Warn("Some Partner Server roles are missing!", lookup.Values.Where(l => !languageRoleLookup.ContainsKey(l.LanguageCode)).ToArray());
            }

            _client = TwitchApiHelper.GetClient(FriendsServiceConfiguration.Instance.TwitchClientID);

            _lastRoleUpdateTime = DateTime.UtcNow;
        }

        public static void Process(int userID)
        {
            if(_lastRoleUpdateTime < DateTime.UtcNow.AddMinutes(-5))
            {
                try
                {
                    RefreshRoles();
                }
                catch(Exception ex)
                {
                    Logger.Debug("Failed to update roles from DB");
                }
            }

            // Short circuit without hitting Twitch or Aerospike
            if(_group == null)
            {
                Logger.Warn("Skipping Partner Server Role Assignment, Partner Server not found!", new { userID });
                return;
            }

            var user = Data.User.FindByUserID(userID);
            if (user == null || !user.IsMerged)
            {
                return;
            }

            var member = _group.GetMember(userID);
            if(member== null)
            {
                Logger.Warn("Skipping Partner Server Role Assignment, Member not found in Server!", new { userID });
                return;
            }

            AddLanguageRole(member, user.TwitchID);
            AddStaffRole(member, user.TwitchID);
        }

        private static void AddLanguageRole(GroupMember member, string twitchUserID)
        {
            try
            {
                var acct = ExternalAccount.GetByTwitchUserID(twitchUserID);
                var isPartner = acct?.IsPartnered ?? false;
                var language = _fallbackLanguage;
                try
                {
                    var channel = _client.GetChannel(twitchUserID);
                    if (channel.Status == TwitchResponseStatus.Success)
                    {
                        isPartner = channel.Value.Partner;
                        language = (channel.Value.Language ?? _defaultRole?.Item1.LanguageCode ?? _fallbackLanguage).Split(new[] { '-' }, 2)[0];
                    }
                    else
                    {
                        Logger.Warn("Could not get partner channel info, using fallbacks", new { member.UserID, twitchUserID, channel });
                    }

                }
                catch(Exception ex)
                {
                    Logger.Warn(ex, "Could not get partner channel info, using fallbacks", new { member.UserID, twitchUserID });
                }

                if (!isPartner)
                {
                    return;
                }

                GroupRole role;
                if(!_languageRoleLookup.TryGetValue(language, out role))
                {
                    role = _defaultRole?.Item2;
                }

                if (role != null && !role.IsDeleted)
                {
                    _group.SystemAddMemberRole(role, member);
                }
            }
            catch(Exception ex)
            {
                Logger.Warn(ex, "Unable to add Partner Server language role");
            }
        }

        private static void AddStaffRole(GroupMember member, string twitchUserID)
        {
            if (_staffRole?.IsDeleted ?? true)
            {
                Logger.Warn("Unable to add Partner Server staff role, staff role not found!");
                return;
            }

            try
            {
                var user = _client.GetPublicUser(twitchUserID);
                if (user.Status != TwitchResponseStatus.Success)
                {
                    Logger.Warn("Unable to add Partner Server staff role, couldn't retrieve user from Twitch", new { twitchUserID, user });
                    return;
                }

                if (!string.Equals(user.Value.Type, "staff", StringComparison.OrdinalIgnoreCase))
                {
                    // Not staff
                    return;
                }

                _group.SystemAddMemberRole(_staffRole, member);
            }
            catch(Exception ex)
            {
                Logger.Warn(ex, "Unable to add Partner Server staff role");
            }
        }
    }
}
