﻿using Newtonsoft.Json.Linq;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace Curse.Friends.WorkerService.Payloads
{
    /// <summary>
    /// Alert Portion of the Notification Payload
    /// </summary>
    public class AppleNotificationAlert
    {
        /// <summary>
        /// Constructor
        /// </summary>
        public AppleNotificationAlert()
        {
            Body = null;
            ActionLocalizedKey = null;
            LocalizedKey = null;
            LocalizedArgs = new List<object>();
            LaunchImage = null;
        }

        /// <summary>
        /// Body Text of the Notification's Alert
        /// </summary>
        public string Body
        {
            get;
            set;
        }

        /// <summary>
        /// Action Button's Localized Key
        /// </summary>
        public string ActionLocalizedKey
        {
            get;
            set;
        }

        /// <summary>
        /// Localized Key
        /// </summary>
        public string LocalizedKey
        {
            get;
            set;
        }

        /// <summary>
        /// Localized Argument List
        /// </summary>
        public List<object> LocalizedArgs
        {
            get;
            set;
        }

        public void AddLocalizedArgs(params object[] values)
        {
            this.LocalizedArgs.AddRange(values);
        }

        /// <summary>
        /// Filename of the Lauch Image (with ou without extension)
        /// >= iOS 4.0
        /// </summary>
        public string LaunchImage
        {
            get;
            set;
        }

        /// <summary>
        /// Determines if the Alert is empty and should be excluded from the Notification Payload
        /// </summary>
        public bool IsEmpty
        {
            get
            {
                if (!string.IsNullOrEmpty(Body)
                    || !string.IsNullOrEmpty(ActionLocalizedKey)
                    || !string.IsNullOrEmpty(LocalizedKey)
                    || (LocalizedArgs != null && LocalizedArgs.Count > 0)
                    || !string.IsNullOrEmpty(LaunchImage)
                    )
                    return false;
                else
                    return true;
            }
        }
    }

    public class AppleNotificationPayload
    {
        public AppleNotificationAlert Alert { get; set; }

        public int? ContentAvailable { get; set; }

        public int? Badge { get; set; }

        public string Sound { get; set; }

        public bool HideActionButton { get; set; }

        public string Category { get; set; }

        public Dictionary<string, object[]> CustomItems
        {
            get;
            private set;
        }

        public AppleNotificationPayload()
        {
            HideActionButton = false;
            Alert = new AppleNotificationAlert();
            CustomItems = new Dictionary<string, object[]>();
        }

        public AppleNotificationPayload(string alert)
        {
            HideActionButton = false;
            Alert = new AppleNotificationAlert() { Body = alert };
            CustomItems = new Dictionary<string, object[]>();
        }

        public AppleNotificationPayload(string alert, int badge)
        {
            HideActionButton = false;
            Alert = new AppleNotificationAlert() { Body = alert };
            Badge = badge;
            CustomItems = new Dictionary<string, object[]>();
        }

        public AppleNotificationPayload(string alert, int badge, string sound)
        {
            HideActionButton = false;
            Alert = new AppleNotificationAlert() { Body = alert };
            Badge = badge;
            Sound = sound;
            CustomItems = new Dictionary<string, object[]>();
        }

        public AppleNotificationPayload(string alert, int badge, string sound, string category)
            : this(alert, badge, sound)
        {
            Category = category;
        }

        public void AddCustom(string key, params object[] values)
        {
            if (values != null)
                this.CustomItems.Add(key, values);
        }

        public JObject ToJObject()
        {
            JObject json = new JObject();
            JObject aps = new JObject();

            if (!this.Alert.IsEmpty)
            {
                if (!string.IsNullOrEmpty(this.Alert.Body)
                    && string.IsNullOrEmpty(this.Alert.LocalizedKey)
                    && string.IsNullOrEmpty(this.Alert.ActionLocalizedKey)
                    && (this.Alert.LocalizedArgs == null || this.Alert.LocalizedArgs.Count <= 0)
                    && string.IsNullOrEmpty(this.Alert.LaunchImage)
                    && !this.HideActionButton)
                {
                    aps["alert"] = new JValue(this.Alert.Body);
                }
                else
                {
                    JObject jsonAlert = new JObject();

                    if (!string.IsNullOrEmpty(this.Alert.LocalizedKey))
                        jsonAlert["loc-key"] = new JValue(this.Alert.LocalizedKey);

                    if (this.Alert.LocalizedArgs != null && this.Alert.LocalizedArgs.Count > 0)
                        jsonAlert["loc-args"] = new JArray(this.Alert.LocalizedArgs.ToArray());

                    if (!string.IsNullOrEmpty(this.Alert.Body))
                        jsonAlert["body"] = new JValue(this.Alert.Body);

                    if (this.HideActionButton)
                        jsonAlert["action-loc-key"] = new JValue((string)null);
                    else if (!string.IsNullOrEmpty(this.Alert.ActionLocalizedKey))
                        jsonAlert["action-loc-key"] = new JValue(this.Alert.ActionLocalizedKey);

                    if (!string.IsNullOrEmpty(this.Alert.LaunchImage))
                        jsonAlert["launch-image"] = new JValue(this.Alert.LaunchImage);

                    aps["alert"] = jsonAlert;
                }
            }

            if (this.Badge.HasValue)
                aps["badge"] = new JValue(this.Badge.Value);

            if (!string.IsNullOrEmpty(this.Sound))
                aps["sound"] = new JValue(this.Sound);

            if (this.ContentAvailable.HasValue)
            {
                aps["content-available"] = new JValue(this.ContentAvailable.Value);
                if (string.IsNullOrEmpty(this.Sound))
                {
                    //You need to add an empty string for sound or the payload is not sent
                    aps["sound"] = new JValue("");
                }
            }

            if (!string.IsNullOrEmpty(this.Category))
            {
                // iOS8 Interactive Notifications
                aps["category"] = new JValue(this.Category);
            }

            if (aps.Count > 0)
                json["aps"] = aps;

            foreach (string key in this.CustomItems.Keys)
            {
                if (this.CustomItems[key].Length == 1)
                {
                    object custom = this.CustomItems[key][0];
                    json[key] = custom is JToken ? (JToken)custom : new JValue(custom);
                }
                else if (this.CustomItems[key].Length > 1)
                    json[key] = new JArray(this.CustomItems[key]);
            }
           
            return json;// encodedString.ToString();
        }        

        public override string ToString()
        {
            return ToJObject().ToString();
        }
    }
}
