﻿using System;
using System.CodeDom;
using System.Collections.Generic;
using System.Linq;
using Curse.Friends.Data;
using Curse.Friends.Enums;
using Curse.Friends.Tracing;

namespace Curse.Friends.WorkerService.Presence
{
    public class PresenceHelper
    {
        private static readonly FilteredUserLogger FilteredLogger = new FilteredUserLogger("PresenceHelper");



        public static UserPresence CalculatePresence(string description, User user, UserConnectionStatus? requestedOverride, IReadOnlyCollection<ClientEndpoint> endpoints, ClientEndpoint sourceEndpoint)
        {
            if (user == null)
            {
                throw new ArgumentNullException(nameof(user));
            }

            if (endpoints == null)
            {
                throw new ArgumentNullException(nameof(endpoints));
            }

            var availability = UserAvailability.Create(user, endpoints, requestedOverride);
            var capabilities = UserCapability.Create(endpoints);

            var presence = new UserPresence(user, sourceEndpoint, availability, capabilities);

            FilteredLogger.Log(user, "Calculated user presence for: " + description, new { user = user.GetLogData(), requestedOverride, presence }, () => new { endpoints });
            
            return presence;
        }

        #region Activity

        public static UserActivity CalculateActivity(string description, User user, IReadOnlyCollection<ClientEndpoint> endpoints, ClientEndpoint sourceEndpoint)
        {
            if (user == null)
            {
                throw new ArgumentNullException(nameof(user));
            }

            if (endpoints == null)
            {
                throw new ArgumentNullException(nameof(endpoints));
            }

            var activityEndpoints = endpoints.Where(p => p.IsConnected).OrderByDescending(p => p.ActivityTimestamp);
            var watching = WatchingActivity.Create(activityEndpoints);
            var playing = PlayingActivity.Create(activityEndpoints);
            var broadcasting = BroadcastingActivity.Create(activityEndpoints);

            return new UserActivity(user, sourceEndpoint, playing, watching, broadcasting);
        }

        #endregion

        #region Utility

        public static IReadOnlyCollection<ClientEndpoint> GetUserEndpoints(User user, UserRegion userRegion)
        {
            var allEndpoints = ClientEndpoint.GetAllForUser(user.UserID);

            if (allEndpoints.Count > ClientEndpoint.PruneThreshold)
            {
                var initialEndpointCount = allEndpoints.Count;
                allEndpoints = ClientEndpoint.UpdateUserMap(userRegion);
                FilteredLogger.Log(user, "User endpoints pruned", new { user = user.GetLogData(), initialEndpointCount, newEndointCount = allEndpoints.Count });
            }

            return allEndpoints;
        }

        #endregion
    }
}
