﻿using System;
using System.Collections.Generic;
using Curse.Friends.Data;
using System.Linq.Expressions;
using Curse.Friends.Data.Queues;
using Curse.Friends.UserEvents;
using Curse.Friends.Statistics;
using Curse.Logging;
using Curse.Friends.Enums;
using Curse.Friends.Tracing;

namespace Curse.Friends.WorkerService.Presence
{
    public class UserActivity
    {
        private static readonly FilteredUserLogger Logger = new FilteredUserLogger("UserActivity");

        public User User { get; }

        public ClientEndpoint SourceEndpoint { get; }

        public PlayingActivity Playing { get; }

        public WatchingActivity Watching { get; }

        public BroadcastingActivity Broadcasting { get; }

        public UserActivity(User user, ClientEndpoint sourceEndpoint, PlayingActivity playing, WatchingActivity watching, BroadcastingActivity broadcasting)
        {
            User = user;
            SourceEndpoint = sourceEndpoint;
            Playing = playing;
            Watching = watching;
            Broadcasting = broadcasting;
        }

        public void SaveGlobalStatus()
        {
            var userUpdates = new List<Expression<Func<User, object>>>();
            if (Watching.HasChanged(User.WatchingChannelID))
            {
                User.WatchingChannelID = Watching.ChannelID;
                userUpdates.Add(u => u.WatchingChannelID);
            }

            if (Playing.HasChanged(User.CurrentGameID, User.CurrentGameState, User.CurrentGameStatusMessage, User.CurrentGameTimestamp))
            {
                var lastPlaying = User.CurrentGameID;

                User.CurrentGameID = Playing.CurrentGameID;
                User.CurrentGameState = Playing.GameState;
                User.CurrentGameStatusMessage = Playing.GameStatusMessage;
                User.CurrentGameTimestamp = Playing.GameTimestamp;

                userUpdates.AddRange(new Expression<Func<User, object>>[]
                {
                    u=>u.CurrentGameID,
                    u=>u.CurrentGameState,
                    u=>u.CurrentGameStatusMessage,
                    u=>u.CurrentGameTimestamp
                });

                // Playing a new game now
                if(lastPlaying != User.CurrentGameID)
                {
                    FriendsStatsManager.Current.CurrentGames.Track(Playing.CurrentGameID);
                    FriendsStatsManager.Current.GameNotifications.Track(Playing.CurrentGameID);
                }

                // Decrement old game
                if (lastPlaying > 0)
                {
                    FriendsStatsManager.Current.CurrentGames.Decrement(lastPlaying);
                }
            }

            if (Broadcasting.HasChanged(User.IsBroadcasting, User.BroadcastingGameID))
            {
                User.IsBroadcasting = Broadcasting.IsBroadcasting;
                User.BroadcastingGameID = Broadcasting.BroadcastingGameID;
                userUpdates.Add(u => u.IsBroadcasting);
                userUpdates.Add(u => u.BroadcastingGameID);
            }

            if (userUpdates.Count == 0)
            {
                return;
            }

            try
            {
                User.Update(userUpdates.ToArray());
                User.UpdateStatistics();
                RegionalUserChangeResolver.Create(User.UserID);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to update global activity");
            }
        }

        public bool SaveLocalStatus(bool raiseStatusEvent = true)
        {
            var userUpdates = new List<Expression<Func<User, object>>>();
            if (Watching.HasChanged(User.WatchingChannelIDLocal))
            {
                User.WatchingChannelIDLocal = Watching.ChannelID;
                userUpdates.Add(u => u.WatchingChannelIDLocal);
            }

            if (Playing.HasChanged(User.CurrentGameIDLocal, User.CurrentGameStateLocal, User.CurrentGameStatusMessageLocal, User.CurrentGameTimestampLocal))
            {
                User.CurrentGameIDLocal = Playing.CurrentGameID;
                User.CurrentGameStateLocal = Playing.GameState;
                User.CurrentGameStatusMessageLocal = Playing.GameStatusMessage;
                User.CurrentGameTimestampLocal = Playing.GameTimestamp;

                userUpdates.AddRange(new Expression<Func<User, object>>[]
                {
                    u=>u.CurrentGameIDLocal,
                    u=>u.CurrentGameStateLocal,
                    u=>u.CurrentGameStatusMessageLocal,
                    u=>u.CurrentGameTimestampLocal
                });
            }

            if (Broadcasting.HasChanged(User.IsBroadcastingLocal, User.BroadcastingGameIDLocal))
            {
                User.IsBroadcastingLocal = Broadcasting.IsBroadcasting;
                User.BroadcastingGameIDLocal = Broadcasting.BroadcastingGameID;
                userUpdates.Add(u => u.IsBroadcastingLocal);
                userUpdates.Add(u => u.BroadcastingGameIDLocal);
            }

            if (userUpdates.Count == 0)
            {
                return false;
            }

            try
            {
                User.Update(userUpdates.ToArray());

                if (SourceEndpoint?.Platform != DevicePlatform.Twitch)
                {
                    if (raiseStatusEvent)
                    {
                        RaisePresenceStatusEvent();
                    }
                    return true;
                }
            }
            catch(Exception ex)
            {
                Logger.Error(ex, "Failed to update local activity");
            }

            return false;
        }

        private void RaisePresenceStatusEvent()
        {
            Logger.Log(User.UserID, "Raising presence status event", new { User = User.GetLogData() });
            new PresenceStatusEventV2
            {
                UserID = User.UserID,
                ConnectionStatus = User.ConnectionStatus,
                CustomStatusTimestamp = User.CustomStatusTimestamp,

                GameID = Playing.CurrentGameID,
                GameState = Playing.GameState,
                GameStatusMessage = Playing.GameStatusMessage,
                ChannelID = Watching.ChannelID
            }.Enqueue();
        }
    }
}
