﻿using Curse.Friends.Data;
using Curse.Friends.Enums;
using Newtonsoft.Json;
using System.Collections.Generic;
using System.Linq;

namespace Curse.Friends.WorkerService.Presence
{
    public class UserAvailability
    {
        [JsonIgnore]
        public User User { get; }

        public UserConnectionStatus Availability { get; }

        public UserConnectionStatus? Override { get; }

        private UserAvailability(User user, UserConnectionStatus availability, UserConnectionStatus? availabilityOverride)
        {
            User = user;
            Availability = availability;
            Override = availabilityOverride;
        }

        public UserConnectionStatus GetStatus()
        {
            if(Availability == UserConnectionStatus.Offline)
            {
                return UserConnectionStatus.Offline;
            }

            if (Override.HasValue)
            {
                return Override.Value;
            }

            if(User.ConnectionStatus == UserConnectionStatus.Away || User.ConnectionStatus == UserConnectionStatus.DoNotDisturb || User.ConnectionStatus == UserConnectionStatus.Invisible)
            {
                return User.ConnectionStatus;
            }

            if(User.ConnectionStatus == UserConnectionStatus.Offline && User.LastConnectionStatus != UserConnectionStatus.Offline && User.LastConnectionStatus != UserConnectionStatus.Idle)
            {
                return User.LastConnectionStatus;
            }

            return Availability;
        }

        public static UserAvailability Create(User user, IReadOnlyCollection<ClientEndpoint> allEndpoints, UserConnectionStatus? requestedOverride)
        {

            // If any other endpoint is connected, check if they are idle
            if (allEndpoints.Any(p => p.IsConnected))
            {
                // If all endpoints are idle, status is idle, regardless of requested status
                if (allEndpoints.Where(p => p.IsConnected).All(p => p.IsIdle))
                {
                    return new UserAvailability(user, UserConnectionStatus.Idle, requestedOverride);
                }

                return new UserAvailability(user, UserConnectionStatus.Online, requestedOverride);
            }

            // At this point, nothing is connected. Set the user's status to Idle if anything is deliverable
            if (allEndpoints.Any(p => p.IsDeliverable))
            {
                return new UserAvailability(user, UserConnectionStatus.Idle, requestedOverride);
            }

            // Nothing is deliverable, user is officially offline
            return new UserAvailability(user, UserConnectionStatus.Offline, requestedOverride);
        }

        public static UserConnectionStatus GetEffectiveAvailability(UserConnectionStatus combined)
        {
            switch (combined)
            {
                case UserConnectionStatus.Away:
                case UserConnectionStatus.DoNotDisturb:
                case UserConnectionStatus.Online:
                case UserConnectionStatus.Invisible:
                    return UserConnectionStatus.Online;
                case UserConnectionStatus.Idle:
                    return UserConnectionStatus.Idle;
                case UserConnectionStatus.Offline:
                default:
                    return UserConnectionStatus.Offline;
            }
        }

        public static UserConnectionStatus GetEffectiveOverride(UserConnectionStatus combined)
        {
            switch (combined)
            {
                case UserConnectionStatus.Away:
                    return UserConnectionStatus.Away;
                case UserConnectionStatus.DoNotDisturb:
                    return UserConnectionStatus.DoNotDisturb;
                case UserConnectionStatus.Invisible:
                    return UserConnectionStatus.Invisible;
                case UserConnectionStatus.Idle:
                case UserConnectionStatus.Offline:
                case UserConnectionStatus.Online:
                default:
                    return UserConnectionStatus.Online;
            }
        }
    }
}
