﻿using System;
using System.Collections.Generic;
using Curse.Friends.Data;
using Curse.Friends.Enums;
using Curse.Friends.Tracing;
using Curse.Friends.UserEvents;
using System.Linq.Expressions;
using Curse.Friends.Data.Queues;
using Curse.Logging;
using Newtonsoft.Json;

namespace Curse.Friends.WorkerService.Presence
{
    public class UserPresence
    {

        private static readonly FilteredUserLogger Logger = new FilteredUserLogger("UserPresence");

        [JsonIgnore]
        public User User { get; }

        public ClientEndpoint SourceEndpoint { get; set; }

        public UserAvailability Availability { get; }

        public UserCapability Capability { get; }

        public UserPresence(User user, ClientEndpoint sourceEndpoint, UserAvailability availability, UserCapability capability)
        {
            User = user;
            Availability = availability;
            Capability = capability;
        }

        public void SaveGlobalPresence()
        {
            var userUpdates = new List<Expression<Func<User, object>>>();

            if (User.Capabilities != Capability.Capability)
            {
                User.Capabilities = Capability.Capability;
                userUpdates.Add(u => u.Capabilities);
            }

            var status = Availability.GetStatus();
            if (status != User.ConnectionStatus)
            {
                User.ConnectionStatus = Availability.GetStatus();
                User.CustomStatusTimestamp = DateTime.UtcNow;
                userUpdates.Add(u => u.ConnectionStatus);
                userUpdates.Add(u => u.CustomStatusTimestamp);
            }

            var sendOverride = false;
            if (Availability.Override.HasValue && Availability.Override != User.LastConnectionStatus)
            {
                User.LastConnectionStatus = Availability.Override.Value;
                userUpdates.Add(u => u.LastConnectionStatus);
                sendOverride = true;
            }

            if (userUpdates.Count == 0)
            {
                return;
            }

            try
            {
                User.Update(userUpdates.ToArray());
                User.UpdateStatistics();
                RegionalUserChangeResolver.Create(User.UserID);

                if (sendOverride && SourceEndpoint?.Platform != DevicePlatform.Twitch)
                {
                    RaisePresenceSettingsEvent();
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to update the user's global presence!");
            }
        }

        private void RaisePresenceSettingsEvent()
        {
            Logger.Log(User.UserID, "Raising presence settings event", new { User = User.GetLogData() });
            new PresenceSettingEventV2
            {
                UserID = User.UserID,
                Availability = User.LastConnectionStatus,
                ShareActivity = User.GetPrivacy().ShareActivityPrivacy == ShareActivityPrivacy.Share,
            }.Enqueue();
        }

        public bool SaveLocalPresence(bool raiseStatusEvent = true)
        {
            var userUpdates = new List<Expression<Func<User, object>>>();

            var effectiveAvailability = UserAvailability.GetEffectiveAvailability(User.ConnectionStatusLocal);
            var effectiveOverride = UserAvailability.GetEffectiveOverride(User.ConnectionStatusLocal);

            if (Availability.Availability != effectiveAvailability)
            {
                if (Availability.Availability == UserConnectionStatus.Online)
                {
                    User.ConnectionStatusLocal = Availability.GetStatus();
                }
                else
                {
                    User.ConnectionStatusLocal = Availability.Availability;
                }

                userUpdates.Add(u => u.ConnectionStatusLocal);
            }

            if (userUpdates.Count == 0)
            {
                return false;
            }

            try
            {
                User.Update(userUpdates.ToArray());
                if (SourceEndpoint?.Platform != DevicePlatform.Twitch)
                {
                    if (raiseStatusEvent)
                    {
                        RaisePresenceStatusEvent();
                    }
                    return true;
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to update the user's local presence!");
            }

            return false;
        }

        private void RaisePresenceStatusEvent()
        {
            Logger.Log(User.UserID, "Raising Presence Status event", new { User = User.GetLogData() });
            new PresenceStatusEventV2
            {
                UserID = User.UserID,
                ChannelID = User.WatchingChannelIDLocal,
                ConnectionStatus = User.ConnectionStatusLocal,
                CustomStatusTimestamp = User.CustomStatusTimestamp,
                GameID = User.CurrentGameIDLocal,
                GameState = User.CurrentGameStateLocal,
                GameStatusMessage = User.CurrentGameStatusMessageLocal,
            }.Enqueue();
        }
    }
}
