﻿using System;
using System.Linq;
using Curse.Aerospike;
using Curse.Extensions;
using Curse.Friends.Data.Queues;
using Curse.Friends.Data.Messaging;
using Curse.Logging;
using Curse.Friends.Data;
using Curse.Friends.NotificationContracts;
using Curse.Friends.Tracing;
using Curse.Friends.UserEvents;

namespace Curse.Friends.WorkerService
{
    class ConversationReadProcessor
    {
        private static readonly FilteredUserLogger Logger = new FilteredUserLogger("ConversationReadProcessor");
        
        public static void Process(ConversationReadWorker e)
        {
            // Ensure time is UTC
            var dateRead = e.MessageTimestamp.NormalizeToUtc();

            // Prevent dates which should not be possible
            if (dateRead < ConversationConstants.DateOfInception)
            {
                Logger.Warn(e.UserID, "Received a mark as read request too far in the past. It will be discarded.", e);
                return;
            }

            // Prevent dates which should not be possible
            if (dateRead > DateTime.UtcNow.AddMinutes(5))
            {
                Logger.Warn(e.UserID, "Received a mark as read request too far in the future. It will be discarded.", e);
                return;
            }

            // Get the conversation container (ex: Group)
            var container = ConversationManager.GetConversationContainer(e.UserID, e.ConversationID);

            // No convo found. Try to fall back to the legacy conversation state (to be removed post-deploy)
            if (container == null)
            {
                if (!ProcessLegacyReadRequest(e))
                {
                    Logger.Warn(e.UserID, "Attempt to mark a non existent conversation as read.", e);
                }
                return;
            }

            // Ensure the user has access
            if (!container.CanAccess(e.UserID))
            {
                Logger.Warn(e.UserID, "Attempt to mark a conversation as read, without having access.", e);
                return;
            }

            // Get the conversation parent (ex: GroupMember)
            var conversationParent = container.GetConversationParent(e.UserID);

            if (conversationParent == null)
            {
                Logger.Warn(e.UserID, "Attempt to mark a non existent conversation parent as read.", e);
                return;
            }

            Logger.Log(e.UserID, "Marking conversation as read", e);

            // Mark as read (or in the case of a group, coordinate the read state)
            conversationParent.MarkAsRead(e.MessageTimestamp);

            NotifyReadState(e);

            // Relay this to Twitch
            if (container is PrivateConversation && !e.IsExternal)
            {
                RaiseEvent(e, container as PrivateConversation);
            }
        }

        private static void RaiseEvent(ConversationReadWorker e, PrivateConversation pm)
        {            
            new ReadConversationEvent()
            {
                MessageID = e.MessageID,
                MessageTimestamp = e.MessageTimestamp.ToEpochMilliseconds(),
                UserID = e.UserID,
                ConversationID = e.ConversationID
            }.Enqueue();
        }

        private static void NotifyReadState(ConversationReadWorker e)
        {
            // Notify all connected endpoints that the conversation has been read            
            var notification = new ConversationReadNotification
            {
                ConversationID = e.ConversationID,
                Timestamp = e.MessageTimestamp
            };

            var deliverableEndpoints = ClientEndpoint.GetAllDeliverable(e.UserID);

            if (!string.IsNullOrEmpty(e.MachineKey))
            {
                deliverableEndpoints = deliverableEndpoints.Where(p => p.MachineKey != e.MachineKey).ToArray();
            };


            ClientEndpoint.DispatchNotification(deliverableEndpoints, endpoint => ConversationReadNotifier.Create(endpoint, notification), null);
        }

        private static bool ProcessLegacyReadRequest(ConversationReadWorker e)
        {
            var friendID = ConversationManager.GetFriendID(e.UserID, e.ConversationID);

            if (friendID == 0)
            {
                return false;
            }

            var friendship = Friendship.GetByOtherUserID(e.UserID, friendID);

            if (friendship == null)
            {
                return false;
            }

            Logger.Log(e.UserID, "Using legacy conversation mode", e);
            friendship.ResetCounterAndSetValue(UpdateMode.Fast, p => p.UnreadCount, p => p.DateRead, e.MessageTimestamp);
            NotifyReadState(e);
            return true;
        }
    }
}
