﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Curse.Aerospike;
using Curse.Friends.Data;
using Curse.Friends.Enums;
using Curse.Friends.Tracing;
using Curse.Logging;

namespace Curse.Friends.WorkerService.Processors
{
    class ExternalAccountSyncProcessor
    {
        private static readonly FilteredUserLogger FilteredLogger = new FilteredUserLogger("ExternalAccountSync");

        public static void Process(ExternalAccountSyncResolver e)
        {
            var user = User.FindByUserID(e.UserID);
            if (user == null)
            {
                FilteredLogger.Warn(e.UserID, "Unable to retrieve user.");
                return;
            }
            
            user.LastAccountSync = DateTime.UtcNow;
            user.Update(p => p.LastAccountSync);

            FilteredLogger.Log(user, "Syncing account", user.GetLogData());

            var mappings = ExternalAccountMapping.GetAllLocal(m => m.UserID, e.UserID).Where(m => !m.IsDeleted).ToArray();
            var accounts = ExternalAccount.MultiGetLocal(mappings.Select(m => new KeyInfo(m.ExternalID, m.Type)));

            foreach (var mapping in mappings)
            {
                var account = accounts.FirstOrDefault(a => a.ExternalID == mapping.ExternalID && a.Type == mapping.Type);
                if (account == null)
                {
                    FilteredLogger.Warn(user, "Failed to find an external account that is mapped to the user.", mapping);
                    continue;
                }

                switch (mapping.Type)
                {
                    case AccountType.Twitch:
                        FilteredLogger.Log(user, "Syncing twitch account.", account.GetLogData());
                        TwitchAccountSyncWorker.Create(mapping.ExternalID);
                        break;
                    case AccountType.YouTube:
                        // Do nothing for now
                        break;
                    case AccountType.WorldOfWarcraft:
                        // Do nothing
                        break;
                    default:
                        Logger.Info("User has an unsupported external community type.", new { Resolver = e, mapping.Type });
                        break;
                }

                // If account has every been synced and it hasn't been synced in over a day, self-heal instead of syncing
                var performFullSync = mapping.LastMembershipSyncTime > DateTime.MinValue && DateTime.UtcNow - mapping.LastMembershipSyncTime >= TimeSpan.FromDays(1);
                
                if (performFullSync)
                {
                    FilteredLogger.Log(user, "Performing full sync.", account.GetLogData());

                    mapping.LastMembershipSyncTime = DateTime.UtcNow;
                    mapping.Update(a => a.LastMembershipSyncTime);

                    switch (mapping.Type)
                    {
                        case AccountType.Twitch:
                        case AccountType.YouTube:
                            ExternalUserSyncWorker.Create(e.UserID, mapping.ExternalID, mapping.Type);
                            break;
                        case AccountType.WorldOfWarcraft:
                            ExternalGuildUserSyncWorker.Create(e.UserID, mapping.Type);
                            break;
                    }
                }               
            }
        }
    }
}
