﻿using System;
using System.Collections.Concurrent;
using System.Linq;
using System.Threading;
using Curse.Friends.Data;
using Curse.Friends.Data.Search;
using Curse.Friends.Tracing;
using Curse.Logging;
using Curse.Extensions;

namespace Curse.Friends.WorkerService.Processors
{
    class ExternalCommunityIndexProcessor
    {
        private static readonly LogCategory Logger = new LogCategory("ExternalCommunityIndex") { Throttle = TimeSpan.FromMinutes(3) };
        private static ConcurrentBag<ExternalCommunity> _communitiesToIndex = new ConcurrentBag<ExternalCommunity>();
        private static readonly Thread workerThread = new Thread(Index);
        private static readonly CancellationTokenSource cancellationToken = new CancellationTokenSource();        
        private static readonly TimeSpan interval = TimeSpan.FromSeconds(5);

        public static void Initialize()
        {                        
            workerThread.Start();
        }

        public static void Stop()
        {
            try
            {
                cancellationToken.Cancel();
                workerThread.Join();
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Unexpected error stopping ExternalCommunityIndexProcessor");
            }            
        }

        public static void Process(ExternalCommunityIndexWorker worker)
        {
            _communitiesToIndex.Add(worker.Community);            
        }
        

        private static void Index(object state)
        {
            var lastComplete = DateTime.UtcNow; 
            while (!cancellationToken.IsCancellationRequested)
            {                
                var communities = _communitiesToIndex;
                _communitiesToIndex = new ConcurrentBag<ExternalCommunity>();

                if (!communities.IsEmpty)
                {
                    foreach (var batch in communities.InSetsOf(500))
                    {
                        var batchArray = batch.ToArray();
                        try
                        {
                            Logger.Debug($"Indexing external communities", new { count = batchArray.Length });
                            ExternalCommunitySearchManager.BulkUpsertCommunities(batchArray);
                        }
                        catch (Exception ex)
                        {
                            Logger.Error(ex, "Failed to index external communities", new { Communities = batchArray.Select(c => c.GetLogData()) });
                        }
                    }
                }

                var msToWait = (int)interval.TotalMilliseconds - (int)((DateTime.UtcNow - lastComplete).TotalMilliseconds);
                lastComplete = DateTime.UtcNow;
                if (msToWait > 0)
                {
                    cancellationToken.Token.WaitHandle.WaitOne(msToWait);
                }                
            }            
        }
    }
}
