﻿using System;
using System.Collections.Generic;
using System.Linq;
using Curse.Aerospike;
using Curse.Extensions;
using Curse.Friends.Data;
using Curse.Friends.Enums;
using Curse.Friends.Tracing;

namespace Curse.Friends.WorkerService.Processors
{
    class ExternalCommunityMemberSyncProcessor
    {
        private static readonly FilteredUserLogger Logger = new FilteredUserLogger("ExternalCommunityMemberSync");

        public static void Process(ExternalCommunityMemberSyncWorker worker)
        {
            try
            {
                var community = ExternalCommunity.GetLocal(worker.ExternalCommunityID, worker.AccountType);
                if (community == null)
                {
                    Logger.Warn(null, null, "Couldn't find external community", worker);
                    return;
                }

                if (community.MappedGroups.Count == 0)
                {
                    Logger.Log(community.ExternalID, "Members will not be synced, as the community is not mapped to any groups.", community);
                    return;
                }

                Logger.Log(community, "Processing community memberships", new { worker, community });

                // Front-load all DB calls
                var memberships = worker.RoleFilter.HasValue
                    ? ExternalCommunityMembership.GetAllLocal(m => m.CommunityTypeAndRoleIndex, community.ExternalID + community.Type + worker.RoleFilter.Value)
                    : ExternalCommunityMembership.GetAllLocal(m => m.CommunityAndTypeIndex, community.ExternalID + community.Type);
                var accounts = GetExternalAccounts(memberships.GroupBy(g => new Tuple<string, AccountType>(g.ExternalUserID, g.Type)).Select(g => g.Key));
                var links = GetExternalAccountMappings(accounts);
               
                // Make convenience collections
                var linksByExternalUser = links.ToLookup(g => g.ExternalID);
                var linksByUser = links.ToLookup(g => g.UserID);
                var uniqueUsers = new HashSet<int>(accounts.SelectMany(a => a.MappedUsers ?? new HashSet<int>()));

                foreach (var groupID in community.MappedGroups)
                {
                    var group = Group.GetWritableByID(groupID);
                    if (group == null)
                    {
                        Logger.Log(community, "Skipping mapped group. It could not be retrieved from the database.", new { groupID, worker, community });
                        continue;
                    }

                    var allGroupRoles = group.GetRoles(true);
                    var groupRoles = allGroupRoles.Where(r => r.IsSynced && r.SyncID == worker.ExternalCommunityID).ToLookup(r => r.Tag);

                    var members = group.GetMembers(uniqueUsers).ToDictionary(m => m.UserID);
                    var membershipsByExternalUser = memberships.ToLookup(m => m.ExternalUserID);

                    foreach (var membership in memberships)
                    {
                        var membersWithMapping = new List<GroupMember>();
                        foreach (var link in linksByExternalUser[membership.ExternalUserID])
                        {
                            GroupMember member;
                            if (members.TryGetValue(link.UserID, out member))
                            {
                                membersWithMapping.Add(member);
                            }
                        }

                        if (membersWithMapping.Count == 0)
                        {
                            continue;
                        }

                        var affectedRoles = groupRoles[membership.RoleTag].ToList();
                        if(affectedRoles.Count == 0)
                        {
                            var externalRole = ExternalCommunityRole.GetLocal(community.ExternalID, community.Type, membership.RoleTag);
                            if(externalRole == null)
                            {
                                continue;
                            }

                            GroupRole role;
                            try
                            {
                                // Try to create role
                                role = Group.CreateSyncedRole(group, externalRole, allGroupRoles);
                                if (role != null)
                                {
                                    allGroupRoles = allGroupRoles.Concat(new[] { role }).ToArray();
                                }
                            }
                            catch (Exception ex)
                            {
                                // Someone else probably created the role at the same time, get all roles again
                                allGroupRoles = group.GetRoles(true);
                                role = allGroupRoles.FirstOrDefault(r => r.IsSynced & r.SyncID == community.ExternalID && r.Tag == membership.RoleTag);
                            }

                            if (role != null)
                            {
                                affectedRoles.Add(role);
                            }
                        }

                        foreach (var groupRole in affectedRoles)
                        {
                            if (membership.Status == ExternalCommunityMembershipStatus.Active && !groupRole.IsDeleted)
                            {
                                foreach (var member in membersWithMapping.Where(m => !m.Roles.Contains(groupRole.RoleID)))
                                {
                                    group.SystemAddMemberRole(groupRole, member, allGroupRoles);
                                }
                            }
                            else if (membership.Status == ExternalCommunityMembershipStatus.Deleted)
                            {
                                foreach (var member in membersWithMapping.Where(m => m.Roles.Contains(groupRole.RoleID)))
                                {
                                    if (linksByUser[member.UserID].All(
                                            l => l.IsDeleted || membershipsByExternalUser[l.ExternalID].All(m => m.Status == ExternalCommunityMembershipStatus.Deleted)))
                                    {
                                        group.SystemRemoveMemberRole(groupRole, member, allGroupRoles);
                                    }
                                }
                            }
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Error occurred while processing community memberships!", worker);
            }
        }

        private static List<ExternalAccount> GetExternalAccounts(IEnumerable<Tuple<string, AccountType>> memberships)
        {
            var accounts = new List<ExternalAccount>();
            foreach (var grouping in memberships.InSetsOf(2000))
            {
                accounts.AddRange(ExternalAccount.MultiGetLocal(grouping.Select(g => new KeyInfo(g.Item1, g.Item2))));
            }
            return accounts;
        }

        private static List<ExternalAccountMapping> GetExternalAccountMappings(IEnumerable<ExternalAccount> accounts)
        {
            var mappings = new List<ExternalAccountMapping>();
            foreach (var grouping in accounts.InSetsOf(1000))
            {
                mappings.AddRange(ExternalAccountMapping.MultiGetLocal(grouping.SelectMany(g => g.MappedUsers.Select(u => new KeyInfo(u, g.Type, g.ExternalID)))));
            }
            return mappings;
        }

        private static Dictionary<Guid, List<ExternalCommunityMembership>> GetMembershipMappings(IEnumerable<ExternalCommunityMembership> memberships, ExternalCommunity community)
        {
            if (!community.MappedGroups.Any())
            {
                return null;
            }
            
            var externalCommunityMappings = community.GetAllMappings();
            var communitiesToGroups = externalCommunityMappings.ToDictionary(x => x.ExternalID, x => x.GroupID);
            var membershipMappings = new Dictionary<Guid, List<ExternalCommunityMembership>>();
            foreach (var membershipBatch in memberships.InSetsOf(2000 / community.MappedGroups.Count))
            {
                var grouping = membershipBatch.GroupBy(x => x.ExternalCommunityID);          
                foreach(var group in grouping)
                {
                    var groupID = communitiesToGroups.GetValueOrDefault(group.Key);                    
                    if(groupID == default(Guid))
                    {
                        continue;                        
                    }
                    if (!membershipMappings.ContainsKey(groupID))
                    {
                        membershipMappings.Add(groupID, new List<ExternalCommunityMembership>());
                    }

                    membershipMappings[groupID].AddRange(group);
                }                              
            }

            return membershipMappings;
        }
    }
}
