﻿using System;
using System.Collections.Generic;
using System.Linq;
using Curse.Aerospike;
using Curse.Friends.Data;
using Curse.Friends.Enums;
using Curse.Friends.Tracing;
using Curse.Logging;

namespace Curse.Friends.WorkerService.Processors
{
    class ExternalUserSyncProcessor
    {
        private static readonly FilteredUserLogger FilteredLogger = new FilteredUserLogger("ExternalUserSync");

        public static void Process(ExternalUserSyncWorker worker)
        {
            try
            {
                var links = ExternalAccountMapping.GetAllLocal(l => l.UserID, worker.CurseUserID).Where(l => l.Type == worker.AccountType).ToArray();

                IGrouping<string, ExternalCommunityMembership>[] allMemberships;
                if (worker.RoleFilter.HasValue)
                {
                    allMemberships = links.SelectMany(l => ExternalCommunityMembership.GetAllLocal(m => m.UserTypeAndRoleIndex, l.ExternalID + l.Type + worker.RoleFilter.Value))
                        .GroupBy(m => m.ExternalCommunityID).ToArray();
                }
                else
                {
                    // Mark account mappings as having a full sync performed since no filtering is used
                    foreach (var link in links)
                    {
                        try
                        {
                            link.LastMembershipSyncTime = DateTime.UtcNow;
                            link.Update(l => l.LastMembershipSyncTime);
                        }
                        catch (Exception ex)
                        {
                            Logger.Error(ex, "Failed to update account link's sync time", new { link, KeyHashes = link.GetKeyHashes() });
                        }
                    }
                    allMemberships = links.SelectMany(l => ExternalCommunityMembership.GetAllLocal(m => m.UserAndTypeIndex, l.ExternalID + l.Type)).GroupBy(m => m.ExternalCommunityID).ToArray();
                }

                FilteredLogger.Log(worker.CurseUserID, "Syncing external community memberships", new { worker, membershipCount = allMemberships.Length });

                var allCommunities = ExternalCommunity.MultiGetLocal(allMemberships.Select(s => new KeyInfo(s.Key, worker.AccountType))).ToDictionary(m => m.ExternalID);
                var allMappings =
                    ExternalCommunityMapping.MultiGetLocal(allCommunities.SelectMany(c => c.Value.MappedGroups.Select(id => new KeyInfo(id, c.Key, c.Value.Type))));
                var allGroups = Group.MultiGetLocal(allMappings.GroupBy(m => m.GroupID).Select(g => new KeyInfo(g.Key))).ToDictionary(g => g.GroupID);

                var allGroupMemberships = new Dictionary<Guid, GroupMember>();
                foreach (var region in allGroups.GroupBy(g => g.Value.RegionID))
                {
                    var groupMembers = GroupMember.MultiGet(region.Key, region.Select(r => new KeyInfo(r.Key, worker.CurseUserID)));
                    foreach (var groupMember in groupMembers)
                    {
                        allGroupMemberships[groupMember.GroupID] = groupMember;
                    }
                }

                var mappingsLookup = allMappings.ToLookup(c => c.ExternalID);

                foreach (var membershipGrouping in allMemberships)
                {
                    ExternalCommunity community;
                    if (!allCommunities.TryGetValue(membershipGrouping.Key, out community))
                    {
                        // No community to sync
                        continue;
                    }

                    if (community.MappedGroups.Count == 0)
                    {
                        // No groups
                        continue;
                    }


                    foreach (var mapping in mappingsLookup[membershipGrouping.Key])
                    {
                        Group group;
                        if (!allGroups.TryGetValue(mapping.GroupID, out group))
                        {
                            continue;
                        }

                        GroupMember member;
                        if (!allGroupMemberships.TryGetValue(mapping.GroupID, out member))
                        {
                            continue;
                        }

                        var groupRoles = GroupRole.GetAll(group.RegionID, g => g.GroupID, mapping.GroupID).ToArray();

                        foreach (var roleGrouping in membershipGrouping.GroupBy(m => m.RoleTag))
                        {
                            var role = groupRoles.FirstOrDefault(r => r.IsSynced && r.SyncID == community.ExternalID && r.Source == community.Type && r.Tag == roleGrouping.Key);
                            if (role == null)
                            {
                                continue;
                            }

                            if (roleGrouping.All(m => m.Status == ExternalCommunityMembershipStatus.Deleted || links.First(l => l.ExternalID == m.ExternalUserID).IsDeleted))
                            {
                                // Delete the member's role if it exists
                                if (member != null && member.Roles.Contains(role.RoleID))
                                {
                                    FilteredLogger.Log(worker.CurseUserID, "Removing group role", new { member.UserID, roleID = role.RoleID, groupID = role.GroupID, roleName = role.Name });
                                    group.SystemRemoveMemberRole(role, member, groupRoles);
                                }
                            }
                            else
                            {
                                // Add the role to the member
                                if (member != null && !member.Roles.Contains(role.RoleID) && !role.IsDeleted)
                                {
                                    FilteredLogger.Log(worker.CurseUserID, "Adding group role", new { member.UserID, roleID = role.RoleID, groupID = role.GroupID, roleName = role.Name });
                                    group.SystemAddMemberRole(role, member, groupRoles);
                                }
                            }
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to process worker.", worker);
            }
        }
    }
}
