﻿using Curse.Aerospike;
using Curse.Friends.Data;
using Curse.Friends.Data.Queues;
using Curse.Logging;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;

namespace Curse.Friends.WorkerService.Processors
{
    class GroupIntegrityWorkerProcessor
    {
        public static void Process(GroupIntegrityWorker e)
        {
            Group group = null;

            // Try to get the group
            try
            {
                group = Group.GetLocal(e.RootGroupID);

                var sw = Stopwatch.StartNew();

                if (group == null)
                {
                    Logger.Warn("Failed to get group from database! Integrity check will be skipped.", e);
                    return;
                }

                group.DateChecked = DateTime.UtcNow;
                group.Update(p => p.DateChecked);

                if (group.MemberCount > 100)
                {
                    Logger.Debug("Starting integrity check: " + group.Title, new { group.GroupID, group.Title, group.MemberCount });
                }

                RunIntegrityCheck(group);

                if (sw.Elapsed.TotalSeconds > 10)
                {
                    Logger.Warn("Integrity check took a while to complete (" + sw.Elapsed.TotalSeconds.ToString("F2") + " seconds)!", new { group.GroupID, group.Title, group.MemberCount });
                }

            }
            catch (Exception ex)
            {
                if (group != null)
                {
                    Logger.Error(ex, "Integrity check failed for group " + group.Title + " in region " + group.GetRegionKey() + ". It will be retried as needed.", new { group.GroupID, group.Title, group.MemberCount });
                }
                else
                {
                    Logger.Error(ex, "Integrity check failed! It will be retried as needed.");
                }
            }
        }

        private static void RunIntegrityCheck(Group group)
        {

            // Get the full non-deleted members list from the root
            var allMembers = group.TryGetAllMembers();
            if (allMembers == null)
            {
                Logger.Warn("Unable to complete integrity check. Failed to retrieve all group members from database.");
                return;
            }

            var membersDictionary = allMembers.ToDictionary(p => p.UserID);

            var notifyGroupChanged = false;

            // Get the full list of channels
            var allChannels = group.GetAllChildren();

            // Fix child group hashset
            var childGroupsUpdated = false;
            if(group.ChildGroupIDs == null)
            {
                group.ChildGroupIDs = new System.Collections.Generic.HashSet<Guid>();
                childGroupsUpdated = true;
            }

            foreach (var channel in allChannels)
            {
                childGroupsUpdated |= group.ChildGroupIDs.Add(channel.GroupID);
            }

            if (childGroupsUpdated)
            {
                notifyGroupChanged = true;
                Logger.Info("Found mismatch between child group HashSet and secondary index for group '" + group.Title + "' (" + group.GroupID + ")!");
                group.Update(g => g.ChildGroupIDs);
            }

            // Fix memberships
            foreach (var channel in allChannels)
            {
                // Get the member list for the channel
                var channelMembers = channel.TryGetAllMembers().ToDictionary(p => p.UserID);

                // Look for any members missing from this channel
                var missingMembers = membersDictionary.Where(serverMember => !channelMembers.ContainsKey(serverMember.Key)).ToArray();

                if (missingMembers.Length > 0)
                {
                    Logger.Info("Found " + missingMembers.Length + " missing members for group '" + group.Title + "' and channel '" + channel.Title + "'");
                }


                foreach (var missing in missingMembers)
                {
                    var newMember = new GroupMember(channel, missing.Value.UserID, missing.Value.Username, missing.Value.DisplayName, missing.Value.RegionID, null, missing.Value.NotificationPreference);
                    try
                    {
                        newMember.InsertLocal();
                    }
                    catch (Exception ex)
                    {
                        Logger.Warn(ex, "Failed to insert new group member.");
                    }
                }

                foreach (var channelMembership in channelMembers)
                {
                    GroupMember parent;
                    if (!membersDictionary.TryGetValue(channelMembership.Key, out parent))
                    {
                        Logger.Trace("Unable to to get parent group membership for channel member.", new { ServerID = group.GroupID, ServerTitle = group.Title, ChannelID = channel.GroupID, UserID = channelMembership.Key });
                        continue;
                    }

                    if (channelMembership.Value.IsDeleted != parent.IsDeleted)
                    {
                        Logger.Trace("Found an inconsistent member delete state", new { ChannelDeleted = channelMembership.Value.IsDeleted, UserID = channelMembership.Value.UserID, ChannelID = channel.GroupID });
                        channelMembership.Value.IsDeleted = parent.IsDeleted;
                        channelMembership.Value.Update(p => p.IsDeleted);
                    }

                }

            }

            var communities = group.GetMappedCommunities();
            var isStreaming = communities.Count > 0 && communities.Any(m => m.IsLive);
            if ((group.IsStreaming && !isStreaming) || (!group.IsStreaming && isStreaming))
            {
                Logger.Info("Group '" + group.Title + "' has the wrong IsStreaming value: " + group.IsStreaming);
                group.IsStreaming = !group.IsStreaming;
                group.Update(g => g.IsStreaming);
                GroupSearchIndexWorker.CreateGroupStreamingStatus(group, group.GetServerSearchSettingsOrDefault());
                notifyGroupChanged = true;
            }

            if (notifyGroupChanged)
            {
                GroupChangeCoordinator.ChangeInfo(group, 0);
            }
        }

    }
}
