﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using Curse.Aerospike;
using Curse.Extensions;
using Curse.Friends.Data;
using Curse.Friends.Data.Queues;
using Curse.Friends.Enums;
using Curse.Friends.Tracing;
using Curse.Friends.WorkerService.Configuration;
using Curse.Friends.WorkerService.Partners;

namespace Curse.Friends.WorkerService.Processors
{
    class GroupMemberWorkerProcessor
    {
        private static readonly FilteredUserLogger Logger = new FilteredUserLogger("GroupMemberWorkerProcessor");

        /// <summary>
        /// Gets the groupmembership from database and updates with new values.
        /// </summary>
        /// <param name="value"></param>
        public static void Process(GroupMemberWorker value)
        {
            try
            {
                // Ensure the group exists locally (if not, race condition?)
                var rootGroup = Group.GetLocal(value.GroupID);

                if (rootGroup == null)
                {
                    Logger.Warn(value.RequestingUserID, "Unable to process group worker. The root group could not be retrieved from the database!", new { value.GroupID });
                    return;
                }

                if (!rootGroup.IsRootGroup)
                {
                    Logger.Warn(value.RequestingUserID, "Unable to process group worker. It is not a root group!", new { value.GroupID });
                    return;
                }

                if (rootGroup.RegionID != Group.LocalConfigID)
                {
                    Logger.Warn(value.RequestingUserID, "Processing a group member worker in the non-home region. This can cause issues.");
                }

                if (value.Type == GroupMemberWorkerType.NewGroup)
                {
                    // Get all of the local members for the root group
                    var rootMemberships = rootGroup.TryGetAllMembers();

                    var newGroup = Group.GetLocal(value.ChildGroupID);
                    var allRoles = newGroup.GetRoles().ToDictionary(p => p.RoleID);

                    foreach (var membership in rootMemberships)
                    {
                        GroupRole[] roles = null;

                        if (newGroup.IsRootGroup)
                        {
                            roles = allRoles.Where(p => membership.Roles.Contains(p.Key)).Select(p => p.Value).ToArray();
                        }

                        var groupMembership = new GroupMember(newGroup, membership.UserID, membership.Username, membership.DisplayName, membership.RegionID, null, NotificationPreference.Enabled, roles);
                        groupMembership.InsertLocal();
                    }
                }
                else if (value.Type == GroupMemberWorkerType.Info)
                {
                    // Deprecated!

                }
                else if (value.Type == GroupMemberWorkerType.DeleteGroup)
                {
                    //The group might be actually deleted from Aerospike by the time it is processed by this quueue.
                    var allMemberships = rootGroup.GetAllRootMembers();

                    foreach (var groupMember in allMemberships)
                    {
                        groupMember.SetDeleted();
                    }
                }
                else if (value.Type == GroupMemberWorkerType.DeleteChannel)
                {
                    Logger.Log(value.RequestingUserID, "Processing channel deletion...", value);

                    // Soft delete all memberships                
                    foreach (var groupID in value.AffectedGroupIDs)
                    {
                        var childGroup = rootGroup.GetChildGroup(groupID, true);

                        if (childGroup == null)
                        {
                            Logger.Warn(value.RequestingUserID, "Unable to process channel deletion. The group could not be retrieved from the database: " + groupID, new { RootGroup = rootGroup, Value = value });
                            continue;
                        }

                        var allMemberships = childGroup.TryGetAllMembers();

                        Logger.Log(value.RequestingUserID, "Soft deleting " + allMemberships.Length + " memberships...");

                        foreach (var membership in allMemberships)
                        {
                            membership.SetDeleted();
                        }
                    }
                }
                else if (value.Type == GroupMemberWorkerType.RestoreChannel)
                {
                    // Undelte all memberships
                    var allGroupIDs = value.AffectedGroupIDs.Concat(new[] { value.GroupID });

                    foreach (var groupID in allGroupIDs)
                    {
                        var childGroup = rootGroup.GetChildGroup(groupID, true);

                        if (childGroup == null)
                        {
                            Logger.Warn(value.RequestingUserID, "Unable to process channel restore. The group could not be retrieved from the database: " + groupID, new { RootGroup = rootGroup, Value = value });
                            continue;
                        }

                        var allMemberships = childGroup.TryGetAllMembers();

                        foreach (var membership in allMemberships)
                        {
                            membership.IsDeleted = false;
                            membership.Update(p => p.IsDeleted);
                        }
                    }
                }
                // We need to copy the members 
                else if (value.Type == GroupMemberWorkerType.Restructured)
                {
                    var allChildren = rootGroup.GetAllChildren(true, true);

                    foreach (var group in allChildren)
                    {
                        var members = group.TryGetAllMembers();

                        foreach (var member in members)
                        {
                            member.ParentGroupID = group.ParentGroupID;
                            member.Update(p => p.ParentGroupID);
                        }
                    }
                }
                // We need to update any affected users best role rank
                else if (value.Type == GroupMemberWorkerType.ChangeRoleRank)
                {
                    Logger.Log(value.RequestingUserID, "Starting ChangeRoleRank worker...");
                    var sw = Stopwatch.StartNew();
                    var rankDictionary = new Dictionary<int, Tuple<int, int>>();
                    var members = rootGroup.TryGetAllMembers();
                    var ranksByRole = rootGroup.GetRoles().ToDictionary(p => p.RoleID);
                    foreach (var member in members)
                    {

                        var bestRole = member.Roles.Select(p => ranksByRole.GetValueOrDefault(p))
                            .Where(p => p != null)
                            .OrderBy(p => p.Rank)
                            .FirstOrDefault();
                        if (bestRole == null)
                        {
                            continue;
                        }

                        if (member.BestRoleRank != bestRole.Rank || member.BestRole != bestRole.RoleID)
                        {

                            member.BestRole = bestRole.RoleID;
                            member.BestRoleRank = bestRole.Rank;
                            member.Update(p => p.BestRoleRank, p => p.BestRole);
                            rankDictionary.Add(member.UserID, new Tuple<int, int>(member.BestRole, member.BestRoleRank));
                        }
                    }

                    if (rankDictionary.Any())
                    {
                        GroupMemberIndexWorker.RoleRanks(rootGroup, rankDictionary);
                        GroupChangeCoordinator.UpdateUsers(rootGroup, 0, new HashSet<int>(rankDictionary.Keys));
                    }

                    sw.Stop();
                    Logger.Log(value.RequestingUserID, "ChangeRoleRank worker complete in " + sw.Elapsed.TotalSeconds.ToString("F2") + " seconds.");
                }
                else if (value.Type == GroupMemberWorkerType.MemberAcitivity)
                {
                    rootGroup.DateMemberActivitySaved = DateTime.UtcNow;
                    rootGroup.Update(p => p.DateMemberActivitySaved);

                    var timestamp = DateTime.UtcNow;

                    foreach (var set in value.AffectedUserIDs.InSetsOf(100))
                    {
                        var setMembers = GroupMember.MultiGetLocal(set.Select(p => new KeyInfo(rootGroup.GroupID, p)));
                        foreach (var member in setMembers)
                        {
                            member.DateLastActive = timestamp;
                            member.Update(UpdateMode.Fast, p => p.DateLastActive);
                        }
                    }

                    // Reflect these changes immediately in elastic
                    GroupMemberIndexWorker.CreateMembersActivity(rootGroup, value.AffectedUserIDs.ToArray(), timestamp);

                }

                else if (value.Type == GroupMemberWorkerType.MembersRemoved)
                {
                    foreach (var userID in value.AffectedUserIDs)
                    {
                        // reconcile owner
                        if (rootGroup.OwnerID == userID)
                        {
                            rootGroup.ReconcileOwnership(userID);
                        }

                        rootGroup.ReconcileExternalCommunity(userID);
                        rootGroup.ReconcileExternalGuild(userID);
                    }
                }

                else if (value.Type == GroupMemberWorkerType.RoleDeleted)
                {
                    rootGroup.ReconcileRemovedRole(value.RequestingUserID, value.AffectedRoleID);
                }

                else if (value.Type == GroupMemberWorkerType.MembersAdded)
                {
                    GroupMemberIndexWorker.CreateMembersAddedWorker(rootGroup, value.AffectedUserIDs);

                    if (rootGroup.GroupID == PartnerServerConfiguration.Instance.GroupID)
                    {
                        foreach (var userID in value.AffectedUserIDs)
                        {
                            PartnerServerJoinedProcessor.Process(userID);
                        }
                    }

                    var linkedCommunities = ExternalCommunityMapping.GetAllLocal(m => m.GroupID, value.GroupID).Where(c => !c.IsDeleted).ToArray();
                    var linkedGuilds = ExternalGuildMapping.GetAllLocal(m => m.GroupID, value.GroupID).Where(g => !g.IsDeleted).GroupBy(g => g.Type).ToArray();

                    if (linkedCommunities.Length == 0 && linkedGuilds.Length == 0)
                    {
                        // No links, don't try to sync the user
                        return;
                    }

                    foreach (var userID in value.AffectedUserIDs)
                    {
                        // Only sync if the user has a linked account with the same account type as a linked community
                        var linkedAccounts = ExternalAccountMapping.GetAllLocal(a => a.UserID, userID).Where(a => !a.IsDeleted).ToArray();

                        var twitchCommunities = linkedCommunities.Where(c => c.Type == AccountType.Twitch).ToArray();
                        if (twitchCommunities.Any() && linkedAccounts.Any(a=>a.Type == AccountType.Twitch))
                        {
                            Logger.Log(userID, "Kicking off TwitchServerRoleWorker", new {value});
                            TwitchServerRoleWorker.Create(userID, value.GroupID);
                        }

                        var otherCommunities = linkedCommunities.Where(c => c.Type != AccountType.Twitch).ToArray();
                        if (otherCommunities.Any())
                        {
                            foreach (var linkedAccount in linkedAccounts.Where(a => a.Type != AccountType.Twitch && otherCommunities.Any(c => a.Type == c.Type)))
                            {
                                ExternalUserSyncWorker.Create(userID, linkedAccount.ExternalID, linkedAccount.Type);
                            }
                        }

                        foreach (var linkedGuildGrouping in linkedGuilds)
                        {
                            if (linkedAccounts.Any(a => a.Type == linkedGuildGrouping.Key))
                            {
                                ExternalGuildUserSyncWorker.Create(userID, linkedGuildGrouping.Key);
                            }
                        }
                    }
                }

                else if (value.Type == GroupMemberWorkerType.AccountUnlinked)
                {
                    rootGroup.ReconcileExternalCommunity(value.RequestingUserID);
                    rootGroup.ReconcileExternalGuild(value.RequestingUserID);
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Error processing a group member worker.", value);

            }
        }
    }
}
