﻿using System;
using System.Diagnostics;
using Curse.Extensions;
using Curse.Friends.Data;
using Curse.Friends.Data.Messaging;
using Curse.Friends.Data.Models;
using Curse.Friends.Data.Queues;
using Curse.Friends.Enums;
using Curse.Friends.NotificationContracts;
using Curse.Friends.Statistics;
using Curse.Friends.Tracing;
using Curse.Friends.TwitchApi;
using Curse.Friends.UserEvents;
using User = Curse.Friends.Data.User;
using Curse.Friends.TwitchApi.CurseShim;

namespace Curse.Friends.WorkerService
{
    class PrivateMessageProcessor
    {
        private static CurseShimClient _twitchApi;
        private static int _maxWhispersPerSecond;
        private static int _maxWhispersPerMinute;

        private static readonly FilteredUserLogger Logger = new FilteredUserLogger("PrivateMessageProcessor");
        
        public static void Initialize(string twitchShimApiKey, string twitchShimUrl, int maxWhispersPerSecond, int maxWhispersPerMinute)
        {
            Logger.Info("Initializing Twitch Shim API client...");
            _twitchApi = new CurseShimClient(twitchShimApiKey, twitchShimUrl);
            _maxWhispersPerSecond = maxWhispersPerSecond;
            _maxWhispersPerMinute = maxWhispersPerMinute;
        }

        public static void Process(PrivateMessageWorker e)
        {
            try
            {
                TrySendMessage(e);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to process private message.");
                SendResponse(e.ConversationID, e.SenderID, e.SenderMachineKey, e.ClientMessageID, DeliveryStatus.Error);
            }
        }

        public static void ProcessExternal(ExternalPrivateMessageWorker e)
        {
            try
            {
                TrySendExternalMessage(e);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to process external private message.", new { e.SenderID, e.ConversationID });
            }
        }

        static void TrySendExternalMessage(ExternalPrivateMessageWorker e)
        {
            var sw = Stopwatch.StartNew();

            // Get sender and recipient data
            var senderRegion = UserRegion.GetByUserID(e.SenderID);
            if (senderRegion == null)
            {
                Logger.Warn(e.SenderID, "Sender UserRegion not found! Message will be discarded", new { e.SenderID, e.ConversationID });
                return;
            }

            var recipientUserID = ConversationManager.GetFriendID(e.SenderID, e.ConversationID);
            var recipientRegion = UserRegion.GetByUserID(recipientUserID);
            if(recipientRegion == null)
            {
                Logger.Warn(recipientUserID, "Recipient UserRegion not found! Message will be discarded", new { e.SenderID, RecipientID = recipientUserID, e.ConversationID });
                return;
            }

            // Get the sender user
            var senderUser = senderRegion.GetUser();

            if (senderUser == null)
            {
                Logger.Warn(e.SenderID, "Sender user record could not be retrieved. Message will be discared.", new { e.SenderID, RecipientID = recipientUserID });
                return;
            }

            // Get or create the PrivateConversation records
            var senderConversation = PrivateConversation.GetOrCreateByUserIDAndOtherUserID(e.MessageTimestamp, e.SenderID, recipientUserID);
            var recipientConversation = PrivateConversation.GetOrCreateByUserIDAndOtherUserID(e.MessageTimestamp, recipientUserID, e.SenderID);
            var recipientPrivacySettings = UserPrivacySettings.GetByUserOrDefault(recipientUserID);
            var recipientUser = recipientRegion.GetUser();

            // Check the user's privacy preferences and smartly update local familiarity state
            var deliverability = GetDeliverability(senderUser, e.SenderIpAddress, recipientUser, e.Body, senderConversation, recipientConversation, recipientPrivacySettings);
            if (!deliverability.IsDeliverable)
            {
                return;
            }

            var envelope = new MessageEnvelope(senderUser)
            {
                Body = e.Body,
                ConversationID = e.ConversationID,
                MessageID = e.MessageID,
                RecipientConversation = recipientConversation,
                SenderConversation = senderConversation,
                RecipientRegion = recipientRegion,                
                SpamConfidence = deliverability.SpamConfidence,
                Timestamp = e.MessageTimestamp,
                IsFamiliar = deliverability.IsFamiliar,
                EmoteOverrides = e.EmoteOverrides,
            };

            // We are now committed to sending this off!
            var message = DeliverMessage(envelope);

            Logger.Log(senderUser, "External message finished processing and dispatching", new { Elapsed = sw.ElapsedMilliseconds.ToString("N2") + " ms", Envelope = envelope.GetLogData(), Message = message.GetLogData() });
        }

        static void SendResponse(string conversationID, int userID, string machineKey, Guid clientMessageID, DeliveryStatus status, long? retryAfter = null, MessageForbiddenReason forbiddenReason = MessageForbiddenReason.Unknown)
        {

            var clientEndpoint = ClientEndpoint.GetLocal(userID, machineKey);
            if (clientEndpoint == null || !clientEndpoint.IsRoutable)
            {
                return;
            }

            var response = new ConversationMessageResponse
            {
                ClientID = clientMessageID,
                Status = status,
                ConversationID = conversationID,
                RetryAfter = retryAfter,
                ForbiddenReason = forbiddenReason
            };

            ChatMessageResponseNotifier.Create(clientEndpoint, response);

        }

        static void TrySendMessage(PrivateMessageWorker e)
        {
            var sw = Stopwatch.StartNew();
            var timestamp = DateTime.UtcNow.RoundToMillisecond();
            var epoch = timestamp.ToEpochMilliseconds();

            // Get the sender's endpoint, for the purpose of notifying them of the response. 
            // Note: This will be null for system messages
            var senderEndpoint = ClientEndpoint.GetLocal(e.SenderID, e.SenderMachineKey);
            var senderIpAddress = senderEndpoint != null ? senderEndpoint.IPAddress : "127.0.0.1";
            var senderRegion = UserRegion.GetByUserID(e.SenderID);

            // Get recipient user data
            var recipientUserID = ConversationManager.GetFriendID(e.SenderID, e.ConversationID);
            var recipientRegion = UserRegion.GetByUserID(recipientUserID);

            // Ensure both users have a user region record
            if (senderRegion == null || recipientRegion == null)
            {
                Logger.Warn(e.SenderID, "Sender or recipient region could not be retrieved. Message will be discared.", new { e.SenderID, RecipientID = recipientUserID });
                SendResponse(e.ConversationID, e.SenderID, e.SenderMachineKey, e.ClientMessageID, DeliveryStatus.UnknownUser);
                return;
            }

            // Ensure the recipient has not blocked the sender
            if (UserBlock.IsBlocked(senderRegion.UserID, recipientRegion.UserID))
            {
                Logger.Warn(e.SenderID, "Private message blocked by recipient.", new { e.SenderID, RecipientID = recipientUserID });
                SendResponse(e.ConversationID, e.SenderID, e.SenderMachineKey, e.ClientMessageID, DeliveryStatus.Forbidden, null, MessageForbiddenReason.PrivateMessagingRecipientBlockedSender);
                return;
            }

            // prevent sending a message to someone that the sender has blocked since they won't be able to respond.
            if (UserBlock.IsBlocked(recipientRegion.UserID, senderRegion.UserID))
            {
                Logger.Warn(e.SenderID, "Private message blocked: sender has blocked recipient.", new { e.SenderID, RecipientID = recipientUserID });
                SendResponse(e.ConversationID, e.SenderID, e.SenderMachineKey, e.ClientMessageID, DeliveryStatus.Forbidden, null, MessageForbiddenReason.PrivateMessagingSenderBlockedRecipient);
                return;
            }

            // Get the sender user
            var senderUser = senderRegion.GetUser();

            if (senderUser == null)
            {
                Logger.Warn(e.SenderID, "Sender user record could not be retrieved. Message will be discared.", new { e.SenderID, RecipientID = recipientUserID });
                SendResponse(e.ConversationID, e.SenderID, e.SenderMachineKey, e.ClientMessageID, DeliveryStatus.UnknownUser);
                return;
            }

            if (UserBan.IsBannedFrom(e.SenderID, UserBanType.Whisper))
            {
                Logger.Log(senderUser, "Attempt to send a private message while sender is Whisper Banned");
                SendResponse(e.ConversationID, e.SenderID, e.SenderMachineKey, e.ClientMessageID, DeliveryStatus.Forbidden, null, MessageForbiddenReason.PrivateMessagingBanned);
                return;
            }

            if (UserBan.IsBannedFrom(recipientUserID, UserBanType.Whisper))
            {
                Logger.Log(senderUser, "Attempt to send a private message while recipient is Whisper Banned");
                SendResponse(e.ConversationID, e.SenderID, e.SenderMachineKey, e.ClientMessageID, DeliveryStatus.Forbidden, null, MessageForbiddenReason.PrivateMessagingRecipientBanned);
                return;
            }

            // Determine if they are friends
            var isFriend = Friendship.IsConfirmed(senderRegion.RegionID, e.SenderID, recipientUserID);

            // Get or create the PrivateConversation records
            var senderConversation = PrivateConversation.GetOrCreateByUserIDAndOtherUserID(timestamp, e.SenderID, recipientUserID);
            var recipientConversation = PrivateConversation.GetOrCreateByUserIDAndOtherUserID(timestamp, recipientUserID, e.SenderID);

            // If they are not locally familiar, need to check with Twitch            
            var shouldThrottleStrangerWhispers = !isFriend;

            var throttle = UserPrivateConversationThrottle.GetOrCreateForUser(e.SenderID);

            // Time throttling
            long? retryAfter;
            if (throttle.ShouldThrottleForFrequency(_maxWhispersPerSecond, _maxWhispersPerMinute, epoch, out retryAfter))
            {
                Logger.Log(senderUser, "User is throttled for sending too many messages too quickly. Message will be discarded", new {e.SenderID, RecipientID = recipientUserID});
                SendResponse(e.ConversationID, e.SenderID, e.SenderMachineKey, e.ClientMessageID, DeliveryStatus.Throttled, retryAfter);
                return;
            }

            if (shouldThrottleStrangerWhispers)
            {
                // Get the throttle record
                // Check if the user is banned from stranger whispers
                if (throttle.BannedUntil > epoch)
                {
                    Logger.Log(senderUser, "User is banned from private messages. Message will be discared.", new { e.SenderID, RecipientID = recipientUserID });
                    SendResponse(e.ConversationID, e.SenderID, e.SenderMachineKey, e.ClientMessageID, DeliveryStatus.Forbidden, null, MessageForbiddenReason.PrivateMessagingBanned);
                    return;
                }

                Logger.Log(senderUser, "Processed and passed user throttle.", new { e.SenderID, RecipientID = recipientUserID });
            }

            // Determine if this recipient is considered to be locally familiar
            var recipientPrivacySettings = UserPrivacySettings.GetByUserOrDefault(recipientUserID);
            var recipientUser = recipientRegion.GetUser();

            // Check if the user's privacy preferences and smartly update local familiarity state
            var deliverability = GetDeliverability(senderUser, senderIpAddress, recipientUser, e.Body, senderConversation, recipientConversation, recipientPrivacySettings);
            if (!deliverability.IsDeliverable)
            {
                SendResponse(e.ConversationID, senderUser.UserID, e.SenderMachineKey, e.ClientMessageID, DeliveryStatus.Forbidden, null, MessageForbiddenReason.NotFamiliar);
                return;
            }

            // Add or update this in the throttle
            if (shouldThrottleStrangerWhispers)
            {
                // Determine if this is a verified bot
                var bot = Bot.GetByUserID(e.SenderID);
                var limit = bot != null && bot.IsVerified ? 1000 : 40;
                throttle.Track(e.ConversationID, epoch, limit);
            }

            // Stats Tracking
            FriendsStatsManager.Current.MessagesSent.Track(e.SenderID);

            if (senderEndpoint != null)
            {
                FriendsStatsManager.Current.MessagesSentByPlatform.Track((int)senderEndpoint.Platform);
            }

            // Send the response
            SendResponse(e.ConversationID, e.SenderID, e.SenderMachineKey, e.ClientMessageID, DeliveryStatus.Successful);

            var envelope = new MessageEnvelope(senderUser)
            {
                Attachment = e.Attachment,
                Body = e.Body,
                ConversationID = e.ConversationID,
                ClientMessageID = e.ClientMessageID,
                MessageID = Guid.NewGuid(),
                RecipientConversation = recipientConversation,
                SenderConversation = senderConversation,
                RecipientRegion = recipientRegion,                
                SpamConfidence = deliverability.SpamConfidence,
                Timestamp = timestamp,
                IsFamiliar = deliverability.IsFamiliar
            };

            // We are now committed to sending this off!
            var message = DeliverMessage(envelope);

            sw.Stop();

            RaiseUserEvent(message, envelope.MessageID, timestamp, e.Attachment, senderIpAddress);

            Logger.Log(senderUser, "Private message finished processing and dispatching", new { Elapsed = sw.ElapsedMilliseconds.ToString("N2") + " ms", Sender = envelope.Sender.GetLogData(), recipientUserID, isFriend, deliverability, shouldThrottle = shouldThrottleStrangerWhispers, blockStrangers = recipientPrivacySettings });
        }

        private class MessageDeliverability
        {
            public MessageDeliverability(bool isDeliverable, SpamConfidence confidence, bool isFamiliar)
            {
                IsDeliverable = isDeliverable;
                IsFamiliar = isFamiliar;
                SpamConfidence = confidence;
            }

            public SpamConfidence SpamConfidence { get; }
            public bool IsDeliverable { get; }
            public bool IsFamiliar { get; }
        }

        private static void RaiseUserEvent(ConversationMessage message, Guid messageID, DateTime timestamp, Attachment attachment, string senderIpAddress)
        {
            new ConversationMessageEvent
            {
                UserID = message.SenderID,
                ConversationID = message.ConversationID,
                MessageBody = message.Body,
                MessageID = messageID,
                SenderIpAddress = senderIpAddress,
                MessageTimestamp = timestamp,
                AttachmentUrl = attachment != null ? attachment.Url : null
            }.Enqueue();
        }


        private class MessageEnvelope
        {
            public MessageEnvelope(IUserIdentity sender)
            {
                Sender = sender;
            }

            public string ConversationID { get; set; }
            public Guid ClientMessageID { get; set; }
            public Guid MessageID { get; set; }
            public DateTime Timestamp { get; set; }
            public string Body { get; set; }
            public IUserIdentity Sender { get; private set; }            
            public PrivateConversation SenderConversation { get; set; }
            public PrivateConversation RecipientConversation { get; set; }
            public UserRegion RecipientRegion { get; set; }
            public SpamConfidence SpamConfidence { get; set; }
            public Attachment Attachment { get; set; }
            public bool IsFamiliar { get; set; }
            public ConversationMessageEmoteSubstitution[] EmoteOverrides { get; set; }

            public object GetLogData()
            {
                try
                {
                    return new { SenderID = SenderConversation.UserID, SenderUsername = RecipientConversation.Title, RecipientID = RecipientRegion.UserID, RecipientUsername = SenderConversation.Title };
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to produce log data from envelope.");                    
                    return new { Error = "Failed to produce log data." };
                }
                
            }
        }


        private static ConversationMessage DeliverMessage(MessageEnvelope envelope)
        {

            // Update the message date for the sender
            envelope.SenderConversation.MarkAsSent(envelope.Timestamp);

            // Increment the unread count for recipient
            envelope.RecipientConversation.MarkAsUnread(envelope.Timestamp, 1);

            // Construct the message
            var message = ConversationMessage.Create(envelope.ConversationID, ConversationType.Friendship, envelope.MessageID, envelope.Body, envelope.Timestamp, envelope.Sender, null, 0, 0, envelope.RecipientConversation.UserID, null, envelope.Attachment, envelope.EmoteOverrides);

            // Create the notification
            var senderNotification = message.ToNotification(envelope.SenderConversation.UserID, envelope.ClientMessageID, ConversationType.Friendship, ConversationNotificationType.Normal);

            // Queue it off to be persisted to Elasticsearch
            ConversationMessageWorker.CreateNewMessage(message);
            
            // Notify the Sender
            ClientEndpoint.DispatchNotification(envelope.SenderConversation.UserID, ep => FriendMessageNotifier.Create(ep, senderNotification));

            // Create the recipient notification
            var recipientNotification = message.ToNotification(envelope.RecipientConversation.UserID, envelope.ClientMessageID, ConversationType.Friendship, ConversationNotificationType.Normal, envelope.SpamConfidence);

            // Notify the Recipient (includes push notifications)
            ClientEndpoint.DispatchNotification(envelope.RecipientConversation.UserID,
                ep => FriendMessageNotifier.Create(ep, recipientNotification),
                ep =>
                {
                    if (!envelope.IsFamiliar) // Do not send push notifications for non-friend messages
                    {
                        return;
                    }

                    PushNotificationWorker.ConversationMessage(envelope.RecipientRegion.RegionID, ep, envelope.Sender.UserID, envelope.Sender.GetTitleName(), envelope.RecipientConversation.UserID, envelope.Body, envelope.ConversationID, envelope.Timestamp);
                });

            return message;
        }

        private static MessageDeliverability GetDeliverability(User senderUser, string ipAddress, User recipientUser, string body, PrivateConversation senderConversation, PrivateConversation recipientConversation, UserPrivacySettings recipientPrivacySettings)
        {
            var isFamiliar = recipientConversation.IsFriend || recipientConversation.HasSent || senderConversation.IsFamiliar;
            if (recipientConversation.IsFriend || (isFamiliar && !senderConversation.HasFamiliarityExpired))
            {
                // Don't check familiarity remotely if they are friends
                Logger.Log(senderUser, "Skipping spam classifier, participants are familiar", new { Sender = senderUser.GetLogData(), Recipient = recipientUser.GetLogData() });
                return new MessageDeliverability(true, SpamConfidence.Unknown, true);
            }

            if (!senderUser.IsMerged || !recipientUser.IsMerged)
            {
                // If at least one user isn't merged, default to local values
                Logger.Log(senderUser, "Skipping spam classifier, at least one user is not merged", new { Sender = senderUser.GetLogData(), Recipient = recipientUser.GetLogData() });
                return new MessageDeliverability(!recipientPrivacySettings.BlockStrangerPMs || isFamiliar, SpamConfidence.Unknown, isFamiliar);
            }

            // Classify spam and get familiarity
            TwitchResponse<ClassifySpamResponse> response = null;
            var sw = new Stopwatch();
            sw.Start();
            try
            {
                response = _twitchApi.ClassifySpam(senderUser.TwitchID, senderUser.Username, ipAddress, recipientUser.TwitchID, recipientUser.Username, body);
                sw.Stop();
                Logger.Log(senderUser, $"Spam classified in {sw.ElapsedMilliseconds} ms", response);
                if(response.Status != TwitchResponseStatus.Success)
                {
                    Logger.Warn(senderUser, null, "Bad response from the classify spam endpoint, defaulting to allow delivery", new { response });
                    return new MessageDeliverability(true, SpamConfidence.Unknown, isFamiliar);
                }

                if(response.Value.IsFamiliar != senderConversation.IsFamiliar)
                {
                    senderConversation.IsFamiliar = response.Value.IsFamiliar;
                    senderConversation.FamiliarityTimestamp = DateTime.UtcNow.ToEpochMilliseconds();
                    senderConversation.Update(p => p.IsFamiliar, p => p.FamiliarityTimestamp);
                }

                var allow = response.Value.AllowMessage && (!recipientPrivacySettings.BlockStrangerPMs || response.Value.IsFamiliar);
                return new MessageDeliverability(allow, ConvertConfidenceType(senderUser, response.Value.ConfidenceType), response.Value.IsFamiliar);
            }
            catch (Exception ex)
            {
                Logger.Warn(ex, senderUser, null, "Exception calling the classify spam endpoint, defaulting to allow delivery", new { response });
                return new MessageDeliverability(true, SpamConfidence.Unknown,  isFamiliar);
            }
        }

        private static SpamConfidence ConvertConfidenceType(User senderUser, string confidenceType)
        {
            switch (confidenceType)
            {
                case "low":
                    return SpamConfidence.Low;
                case "medium":
                    return SpamConfidence.Medium;
                case "high":
                    return SpamConfidence.High;
                default:
                    Logger.Warn(senderUser, "Unknown spam confidence type: " + confidenceType);
                    return SpamConfidence.Unknown;
            }
        }
    }
}
