﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text.RegularExpressions;
using Curse.Aerospike;
using Curse.Extensions;
using Curse.Friends.Configuration;
using Curse.Friends.Data;
using Curse.Friends.Data.Messaging;
using Curse.Friends.Enums;
using Curse.Logging;
using PushSharp.Core;
using PushSharp.Google;
using PushSharp.Apple;
using Curse.Friends.WorkerService.Payloads;
using Newtonsoft.Json.Linq;

namespace Curse.Friends.WorkerService
{

    public class PushNotificationConfig
    {
        public byte[] AppleGeneralPushCert { get; set; }

        public string AppleGeneralPushCertPassword { get; set; }

        public byte[] AppleVoicePushCert { get; set; }

        public string AppleVoicePushCertPassword { get; set; }

        public string GooglePushApiKey { get; set; }
    }

    public static class PushNotificationProcessor
    {
        private static ApnsServiceBroker ApplePushBroker;
        private static ApnsServiceBroker ApplePushKitBroker;
        private static GcmServiceBroker GoogleServiceBroker;

        private static readonly LogCategory Logger = new LogCategory("PushNotifications") { AlphaLevel = Logging.LogLevel.Trace, ReleaseLevel = Logging.LogLevel.Info };

        public static void Initialize(PushNotificationConfig config)
        {

            var appleServerEnvironment = ApnsConfiguration.ApnsServerEnvironment.Production;
            
            if (config.AppleGeneralPushCert != null && !string.IsNullOrEmpty(config.AppleGeneralPushCertPassword))
            {
                try
                {

                    ApplePushBroker = new ApnsServiceBroker(new ApnsConfiguration(appleServerEnvironment, config.AppleGeneralPushCert, config.AppleGeneralPushCertPassword));
                    ApplePushBroker.OnNotificationFailed += ApplePushBrokerOnNotificationFailed;
                    ApplePushBroker.Start();
                    Logger.Info("Registered with Apple push notification service");
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to register with Apple push notification service");
                }
            }

            if (config.AppleVoicePushCert != null && !string.IsNullOrEmpty(config.AppleVoicePushCertPassword))
            {
                try
                {
                    ApplePushKitBroker = new ApnsServiceBroker(new ApnsConfiguration(appleServerEnvironment, config.AppleVoicePushCert, config.AppleVoicePushCertPassword, false));
                    ApplePushKitBroker.OnNotificationFailed += ApplePushKitBrokerOnNotificationFailed;                    
                    ApplePushKitBroker.Start();
                    Logger.Info("Registered with Apple pushkit notification service");
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to register with Apple push notification service");
                }
            }

            if (!string.IsNullOrEmpty(config.GooglePushApiKey))
            {
                try
                {
                    GoogleServiceBroker = new GcmServiceBroker(new GcmConfiguration(config.GooglePushApiKey));
                    GoogleServiceBroker.OnNotificationFailed += GoogleServiceBrokerOnNotificationFailed;
                    GoogleServiceBroker.Start();
                    Logger.Info("Registered with Google push notification service");
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to register with Google push notification service");
                }
            }
        }

        public static void Stop()
        {
            try
            {
                if (ApplePushBroker != null)
                {
                    Logger.Info("Stopping Apple push broker...");
                    ApplePushBroker.Stop();
                }

                if (ApplePushKitBroker != null)
                {
                    Logger.Info("Stopping Apple push kit broker...");
                    ApplePushKitBroker.Stop();
                }

                if (GoogleServiceBroker != null)
                {
                    Logger.Info("Stopping Google service broker...");
                    GoogleServiceBroker.Stop();
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to stop push notification processor.");
            }
        }

        public static void ProcessItem(PushNotificationWorker value)
        {            
            switch (value.Type)
            {
                case PushNotificationType.Unknown:
                case PushNotificationType.ConversationMessage:                
                    MessageNotification(value);
                    break;
                case PushNotificationType.FriendConfirmation:
                case PushNotificationType.FriendRequest:
                    FriendshipNotification(value);
                    break;
                case PushNotificationType.IncomingCall:
                    CallNotification(value);
                    break;
                default:
                    Logger.Trace("Skipping unsupport worker type: " + value.Type);
                    break;                    
            }
        }

        private static void FriendshipNotification(PushNotificationWorker value)
        {
            
            var user = User.GetLocal(value.RecipientID);
            if (user == null)
            {
                Logger.Warn("Unable to process FriendshipNotification. User is null!");
                return;
            }

            if (!user.FriendRequestPushEnabled)
            {
                return;
            }

            var friendship = Friendship.GetLocal(value.RecipientID, value.SenderID);

            if (friendship == null)
            {
                Logger.Warn("Unable to process FriendshipNotification. Friendship is null!");
                return;
            }

            var avatarUrl = string.Format(FriendsServiceConfiguration.Instance.AvatarUrlFormat, User.AvatarUrlPath, friendship.OtherUserID);
    
            switch (value.Platform)
            {
                case DevicePlatform.iOS:

                    var displayMessage = value.Type == PushNotificationType.FriendConfirmation ?
                        friendship.FormattedDisplayName + " has confirmed your friend request!"
                        : friendship.FormattedDisplayName + " has requested to be your friend!";

                    var applePayload = new AppleNotificationPayload()
                    {
                        Badge = 1, 
                        Sound = "default",                        
                        Alert = new AppleNotificationAlert
                        {
                            Body = displayMessage,
                            LaunchImage = avatarUrl
                        }
                    };                    
                    applePayload.AddCustom("Type", (int)value.Type);
                    applePayload.AddCustom("SenderID", friendship.OtherUserID);
                    applePayload.AddCustom("SenderName", friendship.FormattedDisplayName);
                    applePayload.AddCustom("Timestamp", value.Timestamp);
                    
                    ApplePushBroker.QueueNotification(new ApnsNotification()
                    {
                        DeviceToken = value.DeviceID,
                        Payload = applePayload.ToJObject()
                    });
                    
                    break;

                case DevicePlatform.Android:

                    var payload = new Dictionary<string, string>
                        {
                            { "type", ((int)value.Type).ToString() },
                            { "friendId", friendship.OtherUserID.ToString() },
                            { "username", friendship.FormattedDisplayName },                            
                            { "avatarUrl", avatarUrl },                            
                            { "timestamp", value.Timestamp.ToEpochMilliseconds().ToString() }
                        };

                    var googlePayload = JObject.FromObject(payload);
                    

                    GoogleServiceBroker.QueueNotification(new GcmNotification { 
                        RegistrationIds = new List<string>(new[] { value.DeviceID }),
                        Data = googlePayload
                        }
                    );
                    
                    break;
            }
        }

        public static void MessageNotification(PushNotificationWorker value)
        {
            try
            {

                if (string.IsNullOrEmpty(value.ConversationID))
                {
                    Logger.Warn("Unable to process message notification. ConversationID is null or empty.");
                    return;
                }

                if (value.Message.Length > 1280)
                {
                    value.Message = value.Message.Substring(0, 128) + "...";
                }

                var user = User.GetLocal(value.RecipientID);

                var conversation = ConversationManager.GetConversationContainer(value.RecipientID, value.ConversationID);

                if (conversation == null)
                {
                    TrySendLegacyFriendPushNotification(user, value);                    
                    return;
                }

                var recipient = conversation.GetConversationParent(value.RecipientID);

                if (recipient == null)
                {
                    Logger.Warn("Unable to get conversation parent from ID: " + value.ConversationID);
                    return;
                }

                if (!recipient.ShouldSendPushNotification(user, value.Message, new HashSet<int>(value.MentionedUserIDs ?? new int[0])))
                {
                    Logger.Trace("Skipping message push notification.", value);
                    return;
                }

                var pushType = conversation.ConversationType == ConversationType.Group ? PushNotificationType.GroupMessage : PushNotificationType.InstantMessage;
                var senderName = recipient.GetSenderName(value.SenderID, value.SenderName);
                var avatarUrl = string.Format(FriendsServiceConfiguration.Instance.AvatarUrlFormat, conversation.AvatarUrlSlug, conversation.AvatarUrlID);
                SendMessagePushNotification(user, senderName, pushType, value, avatarUrl, conversation.Title);
            }
            catch (Exception ex)
            {
                Logger.Error("Exception raised processing notification", ex);
            }
        }

        private static void TrySendLegacyFriendPushNotification(User user, PushNotificationWorker value)
        {
            var friendID = ConversationManager.GetFriendID(user.UserID, value.ConversationID);

            if (friendID == 0)
            {
                Logger.Warn("Unable to get conversation from ID: " + value.ConversationID);
                return;
            }

            var friendship = Friendship.GetByOtherUserID(user.UserID, friendID);

            if (friendship == null)
            {
                Logger.Warn("Unable to get friendship from ID: " + value.ConversationID);
                return;
            }

            var senderName = friendship.FormattedDisplayName;
            var avatarUrl = string.Format(FriendsServiceConfiguration.Instance.AvatarUrlFormat, friendship.AvatarUrlSlug, friendship.AvatarUrlID);
            
            SendMessagePushNotification(user, senderName, PushNotificationType.InstantMessage, value, avatarUrl, friendship.FormattedDisplayName);
        }

        private static void SendMessagePushNotification(User user, string senderName, PushNotificationType pushType, PushNotificationWorker value, string avatarUrl,string conversationTitle)
        {
            var resolvedMessage = ResolveMentions(user, value.Message, pushType, value.ConversationID);            
            var displayMessage = pushType == PushNotificationType.GroupMessage ? senderName + " (" + conversationTitle + "): " + resolvedMessage
                                                                                         : senderName + ": " + resolvedMessage;

            Logger.Trace("Sending message push notification.", new { value, conversationTitle, pushType, senderName, avatarUrl, displayMessage });

            switch (value.Platform)
            {
                case DevicePlatform.iOS:

                    var applePayload = new AppleNotificationPayload()
                    {
                        Badge = 1,
                        Sound = "message_received.wav",
                        Alert = new AppleNotificationAlert
                        {
                            Body = displayMessage,
                            LaunchImage = avatarUrl
                        }
                    };

                    applePayload.AddCustom("SenderID", value.SenderID);
                    applePayload.AddCustom("SenderName", senderName);
                    applePayload.AddCustom("Timestamp", value.Timestamp);
                    applePayload.AddCustom("Type", (int)pushType);

                    if (pushType == PushNotificationType.GroupMessage)
                    {
                        applePayload.AddCustom("GroupID", value.ConversationID);
                        applePayload.AddCustom("GroupTitle", conversationTitle);
                    }

                    ApplePushBroker.QueueNotification(new ApnsNotification()
                    {
                        DeviceToken = value.DeviceID,
                        Payload = applePayload.ToJObject()
                    });

                    break;

                case DevicePlatform.Android:

                    var payload = new Dictionary<string, string>
                        {
                            { "friendId", value.SenderID.ToString() },
                            { "username", senderName },
                            { "message", resolvedMessage },
                            { "avatarUrl", avatarUrl },                            
                            { "timestamp", value.Timestamp.ToEpochMilliseconds().ToString() },
                            { "type", ((int)pushType).ToString() }
                        };

                    if (pushType == PushNotificationType.GroupMessage)
                    {
                        payload.Add("groupID", value.ConversationID);
                        payload.Add("groupTitle", conversationTitle);
                    }


                    var googlePayload = JObject.FromObject(payload);

                    GoogleServiceBroker.QueueNotification(new GcmNotification
                    {
                        RegistrationIds = new List<string>(new[] { value.DeviceID }),
                        Data = googlePayload
                    }
                    );

                    break;
            }
        }

        private static readonly Regex MentionsRegex = new Regex("(?<mentionStart>\\s@|^@)(?<userID>\\d+):(?<username>\\w+)(?:\\b|$)", RegexOptions.Compiled);
        
        private static string ResolveMentions(User user, string message, PushNotificationType type, string conversationID)
        {            
            var resolved = MentionsRegex.Replace(message, match =>
            {
                int mentionedUserID;
                if (!int.TryParse(match.Groups["userID"].Value, out mentionedUserID))
                {
                    return match.Value;
                }

                var displayName = match.Groups["username"].Value;

                if (type == PushNotificationType.GroupMessage)
                {
                    var rootGroup = Data.Group.GetLocal(new Guid(conversationID))?.RootGroup;
                    var member = rootGroup?.GetMember(mentionedUserID);
                    displayName = ResolveGroupMemberName(member, displayName);
                }
                else
                {
                    if (mentionedUserID == user.UserID)
                    {
                        // Self - user own username
                        displayName = user.Username;
                    }
                    else // Use your display name for them if they are a friend
                    {
                        var myFriendship = Friendship.GetConfirmedLocal(user.UserID, mentionedUserID);
                        if (myFriendship != null)
                        {
                            displayName = myFriendship.FormattedDisplayName;
                        }
                    }
                }
                return match.Result(string.Format("${{mentionStart}}{0}", displayName));
            });
            
            return resolved;
        }

        private static string ResolveGroupMemberName(GroupMember member, string fallback)
        {
            var name = member?.Nickname;
            if (!string.IsNullOrEmpty(name))
            {
                return name;
            }
            name = member?.DisplayName;
            if (!string.IsNullOrEmpty(name))
            {
                return name;
            }
            name = member?.Username;
            return string.IsNullOrEmpty(name) ? fallback : name;
        }

        public static void CallNotification(PushNotificationWorker value)
        {
            try
            {                

                if (value.CallDetails == null)
                {
                    return;
                }                

                var conversation = ConversationManager.GetConversationContainer(value.RecipientID, value.CallDetails.ConversationID) ??  
                                    new AdHocConversation(value.RecipientID, Friendship.GetByOtherUserID(value.RecipientID, value.SenderID));
                               

                var recipient = conversation.GetConversationParent(value.RecipientID);

                if (recipient == null)
                {
                    Logger.Warn("Unable to get conversation recipient for call", value);
                    return;
                }
                
                var callerAvatarUrl = string.Format(FriendsServiceConfiguration.Instance.AvatarUrlFormat, User.AvatarUrlPath, value.SenderID);
                var senderName = recipient.GetSenderName(value.SenderID, value.SenderName);
                
                Logger.Trace("Sending message push notification.", new { value, conversation.Title, senderName, callerAvatarUrl });
               
                switch (value.Platform)
                {
                    case DevicePlatform.iOS:


                        var applePayload = new AppleNotificationPayload()
                        {
                            Badge = 1,
                            Sound = "incoming_call.mp3",                        
                            Alert = new AppleNotificationAlert
                            {
                                Body = string.Format("Incoming call from {0}.", senderName),
                                LaunchImage = callerAvatarUrl
                            }
                        };                    
                        
                        applePayload.AddCustom("SenderID", value.SenderID);
                        applePayload.AddCustom("SenderName", senderName);
                        applePayload.AddCustom("Timestamp", value.Timestamp);
                        applePayload.AddCustom("Type", PushNotificationType.IncomingCall);
                        applePayload.AddCustom("InviteCode", value.InviteCode);
                        applePayload.AddCustom("CallType", GetCallType(conversation));

                        if (conversation.ConversationType == ConversationType.Group)
                        {
                            applePayload.AddCustom("GroupID", conversation.ConversationID);
                            applePayload.AddCustom("GroupTitle", conversation.Title);
                        }

                        if (value.AccessToken.HasValue)
                        {
                            applePayload.AddCustom("AccessToken", value.AccessToken.Value);
                        }


                        if (string.IsNullOrEmpty(value.PushKitToken))
                        {
                            ApplePushBroker.QueueNotification(new ApnsNotification()
                            {
                                DeviceToken = value.DeviceID,
                                Payload = applePayload.ToJObject(),
                                Expiration = DateTime.MinValue
                            });
                        }
                        else
                        {
                            ApplePushKitBroker.QueueNotification(new ApnsNotification()
                            {
                                DeviceToken = value.PushKitToken,
                                Payload = applePayload.ToJObject(),
                                Expiration = DateTime.MinValue
                            });
                            
                        }
                        
                        break;

                    case DevicePlatform.Android:
                        var payload = new Dictionary<string, string>
                        {
                            {"friendId", value.SenderID.ToString()},
                            {"username", senderName},
                            {"message", string.Format("Incoming call from {0}.", senderName)},
                            {"avatarUrl", callerAvatarUrl},
                            {"timestamp", value.Timestamp.ToEpochMilliseconds().ToString()},
                            {"type", ((int) PushNotificationType.IncomingCall).ToString()},
                            {"inviteCode", value.InviteCode},
                            {"callType", ((int) GetCallType(conversation)).ToString()}
                        };

                        if (conversation.ConversationType == ConversationType.Group)
                        {
                            payload.Add("groupID", conversation.ConversationID);
                            payload.Add("groupTitle", conversation.Title);
                        }

                        if (value.AccessToken.HasValue)
                        {
                            payload.Add("accessToken", value.AccessToken.Value.ToString());
                        }

                        var googlePayload = JObject.FromObject(payload);
                    
                        GoogleServiceBroker.QueueNotification(new GcmNotification 
                            { 
                                RegistrationIds = new List<string>(new[] { value.DeviceID }),
                                Data = googlePayload
                            }
                        );
                        
                        break;
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Exception raised processing notification");
            }
        }

        private static CallType GetCallType(IConversationContainer conversation)
        {
            switch (conversation.ConversationType)
            {
                case ConversationType.Group:
                    return CallType.Group;
                case ConversationType.Friendship:
                    return CallType.Friend;
                default:
                    return CallType.AdHoc;
            }
        }

        static void UpdateDeviceTokenEndpoints(string deviceID, string newDeviceID = null)
        {
            if (deviceID == null)
            {
                return;
            }

            if (string.IsNullOrWhiteSpace(newDeviceID))
            {
                newDeviceID = null;
            }

            ClientEndpoint[] endpoints = null;

            // Try to update it
            try
            {
                Logger.Trace("Updating endpoint(s) device ID");

                endpoints = ClientEndpoint.GetAllLocal(p => p.DeviceID, deviceID);
                foreach (var ep in endpoints)
                {
                    Logger.Trace(newDeviceID == null ? "Removing device ID from endpoint" : "Updating device ID on endpoint", new { ep, newDeviceID });
                    ep.UpdateDeviceID(newDeviceID);
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to update device ID for endpoints!", new { DeviceID = deviceID, NewDeviceID = newDeviceID, Endpoints = endpoints });
            }
        }

        static void UpdatePushKitTokenEndpoints(string pushKitToken, string newPushKitToken = null)
        {
            if (pushKitToken == null)
            {
                return;
            }

            if (string.IsNullOrWhiteSpace(newPushKitToken))
            {
                newPushKitToken = null;
            }

            ClientEndpoint[] endpoints = null;

            // Try to update it
            try
            {
                Logger.Trace("Updating endpoint(s) device ID");

                endpoints = ClientEndpoint.GetAllLocal(p => p.PushKitToken, pushKitToken);
                foreach (var ep in endpoints)
                {
                    Logger.Trace(newPushKitToken == null ? "Removing PushKit token from endpoint" : "Updating PushKit token on endpoint", new { ep, newPushKitToken });
                    ep.UpdatePushKitToken(newPushKitToken);
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to update PushKit token for endpoints!", new { pushKitToken, newPushKitToken, Endpoints = endpoints });
            }
        }

        // Regular Apple Push Failures
        private static void ApplePushBrokerOnNotificationFailed(ApnsNotification notification, Exception exception)
        {
            if (exception is DeviceSubscriptonExpiredException)
            {
                var expiredException = exception as DeviceSubscriptonExpiredException;
                UpdateDeviceTokenEndpoints(expiredException.OldSubscriptionId, expiredException.NewSubscriptionId);
            }
            else
            {
                Logger.Warn(exception, "Apple push notification failed: " + exception.Message, notification);
            }
        }

        // PushKit Notification Failures
        private static void ApplePushKitBrokerOnNotificationFailed(ApnsNotification notification, Exception exception)
        {
            if (exception is AggregateException && exception.InnerException is DeviceSubscriptonExpiredException)
            {
                var expiredException = exception.InnerException as DeviceSubscriptonExpiredException;
                UpdatePushKitTokenEndpoints(expiredException.OldSubscriptionId, expiredException.NewSubscriptionId);
                return;
            }
            
            if(exception is AggregateException)
            {
                exception = exception.InnerException;
            }

            Logger.Warn(exception, "Apple push kit notification failed: " + exception.Message, notification);
        }

        private static void GoogleServiceBrokerOnNotificationFailed(GcmNotification notification, Exception exception)
        {
            if (exception is AggregateException && exception.InnerException is DeviceSubscriptonExpiredException)
            {

                var expiredException = exception.InnerException as DeviceSubscriptonExpiredException;
                UpdateDeviceTokenEndpoints(expiredException.OldSubscriptionId, expiredException.NewSubscriptionId);
                return;
            }

            if (exception is AggregateException)
            {
                exception = exception.InnerException;
            }
            Logger.Warn(exception, "Google push notification failed: " + exception.Message, notification);
        }
    }
}
