﻿using Curse.Friends.Configuration;
using Curse.Friends.Data;
using Curse.Friends.Enums;
using Curse.Friends.TwitchApi;
using Curse.Logging;
using System;
using System.Linq;
using System.Net.Http;

namespace Curse.Friends.WorkerService.Processors
{
    class SyncAvatarToTwitchProcessor
    {
        public static void Process(SyncAvatarToTwitchWorker worker)
        {
            try
            {
                var account = ExternalAccount.GetByTwitchUserID(worker.UserTwitchID);
                if (account == null)
                {
                    Logger.Debug("Unable to find external account", worker);
                    return;
                }

                if (!string.IsNullOrEmpty(account.AvatarUrl))
                {
                    // They have a Twitch avatar, prefer it at this point
                    Avatar.CreateOrUpdate(AvatarType.SyncedAccount, account.GetAvatarEntityID(), account.AvatarUrl);
                }
                else
                {
                    // They don't have a Twitch avatar, prefer Desktop and upload to Twitch as necessary
                    var avatar = Avatar.GetByTypeAndID(AvatarType.User, worker.UserID.ToString());
                    if (avatar == null || avatar.IsInappropriate || string.IsNullOrEmpty(avatar.Url) || string.IsNullOrEmpty(avatar.StorageKey))
                    {
                        Logger.Debug("Avatar cannot be migrated", worker);
                        return;
                    }

                    // Download Image
                    using (var client = new HttpClient())
                    {
                        var imageResponse = client.GetAsync(avatar.Url).Result;
                        if (!imageResponse.IsSuccessStatusCode)
                        {
                            Logger.Warn("Failed to download avatar", worker);
                            return;
                        }
                        var imageStream = imageResponse.Content.ReadAsStreamAsync().Result;

                        // Upload to Twitch and update synced account avatar
                        var clientID = account.ClientID ?? FriendsServiceConfiguration.Instance.TwitchLoginClientMap.First().Key;
                        TwitchApiHelper.GetClient(clientID).UploadAvatar(account.ExternalID, account.AuthToken, imageStream);
                        Avatar.CreateOrUpdate(AvatarType.SyncedAccount, account.GetAvatarEntityID(), avatar.Url);
                    }
                }
            }
            catch (Exception ex)
            {
                Logger.Warn(ex, "Unable to sync Curse avatar to Twitch", worker);
            }
        }
    }
}
