﻿using Curse.Friends.Data;
using Curse.Friends.Enums;
using Curse.Friends.Tracing;
using Curse.Friends.WorkerService.Presence;
using System;
using System.Linq;

namespace Curse.Friends.WorkerService.Processors
{
    class UserActivityResolverProcessor
    {
        private static readonly FilteredUserLogger FilteredLogger = new FilteredUserLogger("UserActivityProcessor");

        public static void Process(UserActivityResolver resolver)
        {
            try
            {
                var userRegion = UserRegion.GetLocal(resolver.UserID);
                if (userRegion == null)
                {
                    FilteredLogger.Warn(resolver.UserID, "Could not find user region in DB", new { resolver });
                    return;
                }

                var user = userRegion.GetUser();
                if (user == null)
                {
                    FilteredLogger.Warn(resolver.UserID, "Could not find user in DB", new { resolver });
                    return;
                }

                FilteredLogger.Log(user, "Resolving activity", new { user = user.GetLogData(), resolver });

                var endpoints = PresenceHelper.GetUserEndpoints(user, userRegion);

                var sourceEndpoint = endpoints.FirstOrDefault(e => e.MachineKey == resolver.MachineKey);
                var globalActivity = PresenceHelper.CalculateActivity("Global", user, endpoints, sourceEndpoint);
                globalActivity.SaveGlobalStatus();

                if (sourceEndpoint?.Platform != DevicePlatform.Twitch)
                {
                    var localActivity = PresenceHelper.CalculateActivity("Local", user, endpoints.Where(e => e.Platform != DevicePlatform.Twitch).ToArray(), sourceEndpoint);
                    localActivity.SaveLocalStatus();
                }

                // Sync external accounts as needed
                if (user.ShouldSyncExternalAccounts())
                {
                    FilteredLogger.Log(user, "Queuing external account sync resolver", new { user = user.GetLogData() });
                    ExternalAccountSyncResolver.Create(user.UserID);
                }

                FilteredLogger.Log(user, "Finished resolving activity", new { user = user.GetLogData(), resolver });
            }
            catch (Exception ex)
            {
                FilteredLogger.Error(ex, "Failed to process user activity", new { resolver });
            }
        }
    }
}
