﻿using Curse.CloudQueue;
using Curse.Friends.Data;
using Curse.Friends.Enums;
using Curse.Friends.Tracing;
using Curse.Friends.WorkerService.Presence;
using System;
using System.Linq;

namespace Curse.Friends.WorkerService.Processors
{
    class UserPresenceResolverProcessor
    {
        private static readonly FilteredUserLogger FilteredLogger = new FilteredUserLogger("UserPresenceProcessor");

        public static void Process(UserPresenceResolver resolver)
        {
            try
            {
                var userRegion = UserRegion.GetLocal(resolver.UserID);
                if (userRegion == null)
                {
                    FilteredLogger.Warn(resolver.UserID, "Could not find user region in DB", new { resolver });
                    return;
                }

                var user = userRegion.GetUser();
                if (user == null)
                {
                    FilteredLogger.Warn(resolver.UserID, "Could not find user in DB", new { resolver });
                    return;
                }

                FilteredLogger.Log(user, "Resolving presence", new { user = user.GetLogData(), resolver });

                var allEndpoints = PresenceHelper.GetUserEndpoints(user, userRegion);
                var sourceEndpoint = allEndpoints.FirstOrDefault(e => e.MachineKey == resolver.MachineKey);

                if (resolver.DisconnectEndpoint)
                {
                    if (sourceEndpoint == null)
                    {
                        FilteredLogger.Warn(user, "Unable to disconnect endpoint. Could not be retrieved from database.", new { user = user.GetLogData(), resolver });
                        return;
                    }

                    string reason;
                    if (!sourceEndpoint.Disconnect(resolver.SessionID, resolver.ConnectionID, userRegion.RegionID, out reason))
                    {
                        FilteredLogger.Warn(user, "Unable to disconnect endpoint. This Offline status change will not be processed.", new { reason, sourceEndpoint, user = user.GetLogData(), resolver });
                        return;
                    }
                }

                var globalPresence = PresenceHelper.CalculatePresence("Global", user, resolver.RequestedOverride, allEndpoints, sourceEndpoint);
                globalPresence.SaveGlobalPresence();

                if (sourceEndpoint?.Platform != DevicePlatform.Twitch)
                {
                    var localPresence = PresenceHelper.CalculatePresence("Local", user, resolver.RequestedOverride, allEndpoints.Where(e => e.Platform != DevicePlatform.Twitch).ToArray(), sourceEndpoint);
                    localPresence.SaveLocalPresence();
                }

                // Sync external accounts as needed
                if (user.ShouldSyncExternalAccounts())
                {
                    FilteredLogger.Log(user, "Queuing external account sync resolver", new { user = user.GetLogData() });
                    ExternalAccountSyncResolver.Create(user.UserID);
                }

                UserActivityResolver.Create(QueueConfiguration.LocalRegionID, resolver.UserID, resolver.MachineKey);

                FilteredLogger.Log(user, "Finished resolving presence", new { user = user.GetLogData(), resolver });
            }
            catch(Exception ex)
            {
                FilteredLogger.Error(ex, "Failed to process User Presence", new { resolver });
            }
        }
    }
}
