﻿using System;
using System.Linq;
using Curse.Extensions;
using Curse.Friends.Data;
using Curse.Friends.Data.Queues;
using Curse.Friends.Tracing;

namespace Curse.Friends.WorkerService.Processors
{
    class UserRegionChangeProcessor
    {
        private static readonly FilteredUserLogger FilteredLogger = new FilteredUserLogger("UserRegionChange");

        public static void Process(UserRegionChangeWorker worker)
        {
            if (!User.IsValidRegion(worker.NewRegionID))
            {
                FilteredLogger.Warn(worker.UserID, "Unable to process user region change. The supplied new region is not valid for this model.", worker);
                return;
            }

            var userRegion = UserRegion.GetLocal(worker.UserID);

            if (userRegion == null)
            {
                FilteredLogger.Warn(worker.UserID, "Unable to process user region change. User region could not be retrieved from the database.", worker);
                return;
            }

            var user = userRegion.GetUser();

            if (user == null)
            {
                FilteredLogger.Warn(worker.UserID, "Unable to process user region change. User could not be retrieved from the database.", worker);
                return;
            }

            FilteredLogger.Log(user, "Processing user region change! May the force be with us.", new { user = user.GetLogData(), userRegion, worker });

            if (userRegion.IsRelocating)
            {
                FilteredLogger.Log(user, "Detected a previously unfinished relocation attempt! Let's hope it goes better this time.", new { user = user.GetLogData(), userRegion, worker });
            }

            // 1. Mark the user region record as relocating, so it can be repaired if it does not complete
            userRegion.IsRelocating = true;
            userRegion.RelocationTimestamp = DateTime.UtcNow.ToEpochMilliseconds();
            userRegion.Update(p => p.IsRelocating, p => p.RelocationTimestamp);

            // 2. Copy the user over to the other region
            user.CopyToRegion(worker.NewRegionID);

            // 3. Copy the user's friends (from their side)
            var friends = Friendship.GetAll(userRegion.RegionID, p => p.UserID, user.UserID);

            FilteredLogger.Log(user, "Copying friends from old region to new region...", new { friendCount = friends.Length, worker});
            foreach (var friend in friends)
            {
                friend.CopyToRegion(worker.NewRegionID);
            }

            FilteredLogger.Log(user, "Updating friend regions...", new { friendCount = friends.Length, worker });
            ChangeFriendRegions(user, worker.NewRegionID);

            // 4. Update the denormalized region ID pointer in group memberships
            var groupMemberships = GroupMember.GetAllByUserID(user.UserID).Where(p => p.IsRootGroup).ToArray();

            FilteredLogger.Log(user, "Updating group memberships to new region...", new { groupCount = groupMemberships.Length, worker });
            foreach (var membership in groupMemberships)
            {
                membership.RegionID = worker.NewRegionID;
                membership.Update(p => p.RegionID);
            }

            // 5. Copy the user's suggestions (This is a soft fail, if it does fail)
            try
            {
                var suggestions = FriendSuggestion.GetAll(userRegion.RegionID, p => p.UserID, user.UserID);

                FilteredLogger.Log(user, "Copying suggestions from old region to new region...", new { suggestionCount = suggestions.Length, worker });
                foreach (var suggestion in suggestions)
                {
                    suggestion.CopyToRegion(worker.NewRegionID);
                }
            }
            catch (Exception ex)
            {
                FilteredLogger.Error(user, ex, "Failed to copy a friend suggestion between regions.");
            }
            
            // 6. Update the user region record (this is the big one!)
            userRegion.RegionID = worker.NewRegionID;
            userRegion.IsRelocating = false;
            userRegion.Update(p => p.IsRelocating, p => p.RegionID);
            
            FilteredLogger.Log(user, "Completed copying user to other region!");
        }

        private static void ChangeFriendRegions(User user, int newRegionID)
        {
            foreach (var region in Friendship.AllConfigurations)
            {
                var friends = Friendship.GetAll(region, p => p.OtherUserID, user.UserID);

                FilteredLogger.Log(user, "Processing friends memberships in region: " + region.RegionKey, new { user.UserID, friendCount = friends.Length });
                foreach (var friend in friends)
                {
                    friend.OtherUserRegionID = newRegionID;
                    friend.Update(p => p.OtherUserRegionID);
                }
            }
        }
    }
}
