﻿using System.Linq;
using Curse.Friends.Data;
using Curse.Friends.Enums;
using Curse.Logging;
using Curse.Friends.Tracing;
using Curse.Friends.WorkerService.Presence;
using Curse.Friends.UserEvents;

namespace Curse.Friends.WorkerService.Processors
{
    class UserStatusProcessor
    {
        private static readonly FilteredUserLogger FilteredLogger = new FilteredUserLogger("UserStatusProcessor");
        private static readonly LogCategory Logger = new LogCategory("UserStatusProcessor");

        public static void ProcessMessage(UserStatusResolver value)
        {
            switch (value.Type)
            {
                case UserStatusResolver.UserStatusResolverType.Active:
                    ProcessActive(value);
                    break;
                case UserStatusResolver.UserStatusResolverType.Idle:
                    ProcessIdle(value);
                    break;
                case UserStatusResolver.UserStatusResolverType.Presence:
                    ProcessPresence(value);
                    break;
                case UserStatusResolver.UserStatusResolverType.TwitchStatus:
                    ProcessTwitchStatus(value);
                    break;
                default:
                    ProcessConnectionStatus(value);
                    break;
            }
        }

        public static void ProcessRegionalMessage(RegionalUserStatusResolver resolver)
        {
            ProcessMessage(new UserStatusResolver
            {
                ConnectionID = resolver.ConnectionID,
                CustomStatusMessage = resolver.CustomStatusMessage,
                DisconnectEndpoint = resolver.DisconnectEndpoint,
                ExternalCommunityID = resolver.ExternalCommunityID,
                GameID = resolver.GameID,
                GameState = resolver.GameState,
                GameStatusMessage = resolver.GameStatusMessage,
                IsGameRunning = resolver.IsGameRunning,
                IsWatching = resolver.IsWatching,
                MachineKey = resolver.MachineKey,
                ResolverType = resolver.ResolverType,
                SessionID = resolver.SessionID,
                SourceServerName = resolver.SourceServerName,
                Status = resolver.Status,
                Type = resolver.Type,
                UserID = resolver.UserID,
                Completed = resolver.Completed,
                EnqueuedTimestamp = resolver.EnqueuedTimestamp,
                Retries = resolver.Retries
            });
        }

        private static void ProcessActive(UserStatusResolver value)
        {
            var region = UserRegion.GetByUserID(value.UserID);
            if (region == null)
            {
                Logger.Warn("Failed to retrieve user region!", new { value.ResolverType, value.UserID });
                return;
            }

            var user = region.GetUser();
            if (user == null)
            {
                Logger.Warn("Failed to retrieve user!", new { value.ResolverType, value.UserID });
                return;
            }

            // If their connection status was not idle, do nothing further.
            if(user.ConnectionStatus != UserConnectionStatus.Idle)
            {
                return;
            }

            // New status should be Online, or their last set status
            FilteredLogger.Log(user, "Processing active status", new { user = user.GetLogData() });

            RecalculatePresence(region, user, null, ClientEndpoint.GetLocal(value.UserID, value.MachineKey));
        }

        private static void ProcessIdle(UserStatusResolver value)
        {
            var region = UserRegion.GetByUserID(value.UserID);
            if (region == null)
            {
                Logger.Warn("Failed to retrieve user region!", new { value.ResolverType, value.UserID });
                return;
            }

            var user = region.GetUser();
            if (user == null)
            {
                Logger.Warn("Failed to retrieve user!", new { value.ResolverType, value.UserID });
                return;
            }

            // User has returned from being idle. We must mark the endpoint as no longer idle.
            var endpoint = ClientEndpoint.GetLocal(value.UserID, value.MachineKey);
            endpoint.ToggleIdle(true, region.RegionID);

            // If their connection status wasn't online (meaning they set it manually), we must examine their endpoints and see if we should mark them as idle
            if(user.ConnectionStatus != UserConnectionStatus.Online)
            {
                return;
            }
            FilteredLogger.Log(user, "Processing idle status", new { user = user.GetLogData() });
            RecalculatePresence(region, user, null, ClientEndpoint.GetLocal(value.UserID, value.MachineKey));
        }

        private static void ProcessConnectionStatus(UserStatusResolver value)
        {
            var region = UserRegion.GetByUserID(value.UserID);
            if (region == null)
            {
                FilteredLogger.Warn(value.UserID, "Failed to retrieve user region!", new { value.ResolverType, value.UserID });
                return;
            }

            var user = region.GetUser();
            if (user == null)
            {
                FilteredLogger.Warn(value.UserID, "Failed to retrieve user!", new { value.ResolverType, value.UserID, region.RegionID });
                return;
            }

            ClientEndpoint endpoint = null;
            if (!string.IsNullOrEmpty(value.MachineKey))
            {
                endpoint = ClientEndpoint.GetLocal(value.UserID, value.MachineKey);
            }

            if (value.DisconnectEndpoint)
            {
                if (endpoint == null)
                {
                    FilteredLogger.Warn(user, "Unable to disconnect endpoint. Could not be retrieved from database.", new { user = user.GetLogData(), value });
                    return;
                }

                string reason;
                if (!endpoint.Disconnect(value.SessionID, value.ConnectionID, region.RegionID, out reason))
                {
                    FilteredLogger.Warn(user, "Unable to disconnect endpoint. This Offline status change will not be processed.", new { reason, endpoint, user = user.GetLogData(), value });
                    return;
                }
            }

            RecalculatePresence(region, user, value.Status, endpoint);
        }

        private static void ProcessPresence(UserStatusResolver value)
        {
            var region = UserRegion.GetByUserID(value.UserID);
            if (region == null)
            {
                FilteredLogger.Warn(value.UserID, "Failed to retrieve user region!", new { value.ResolverType, value.UserID });
                return;
            }

            var user = region.GetUser();
            if (user == null)
            {
                FilteredLogger.Warn(value.UserID, "Failed to retrieve user!", new { value.ResolverType, value.UserID, region.RegionID });
                return;
            }

            RecalculatePresence(region, user, null, ClientEndpoint.GetLocal(value.UserID, value.MachineKey));
        }

        private static void ProcessTwitchStatus(UserStatusResolver value)
        {
            var region = UserRegion.GetByUserID(value.UserID);
            if (region == null)
            {
                Logger.Warn("Failed to retrieve user region!", new { value.ResolverType, value.UserID });
                return;
            }

            var user = region.GetUser();
            if (user == null)
            {
                Logger.Warn("Failed to retrieve user!", new { value.ResolverType, value.UserID });
                return;
            }

            FilteredLogger.Log(user, "Processing twitch availability", new { user = user.GetLogData() });
            RecalculatePresence(region, user, null, ClientEndpoint.GetLocal(value.UserID, value.MachineKey));
        }

        private static void RecalculatePresence(UserRegion userRegion, User user, UserConnectionStatus? requestedStatus, ClientEndpoint endpoint)
        {
            // Get all endpoints
            var allEndpoints = PresenceHelper.GetUserEndpoints(user, userRegion);

            // Coerce status to a valid saved status
            if (requestedStatus.HasValue)
            {
                switch (requestedStatus.Value)
                {
                    case UserConnectionStatus.Idle:
                    case UserConnectionStatus.Offline:
                        requestedStatus = null;
                        break;
                }
            }

            // Calculate the user's new presence
            var newPresence = PresenceHelper.CalculatePresence("Global", user, requestedStatus, allEndpoints, endpoint);
            newPresence.SaveGlobalPresence();

            var newActivity = PresenceHelper.CalculateActivity("Global", user, allEndpoints, endpoint);
            newActivity.SaveGlobalStatus();

            // Raise user events for anything that was not initiated by Twitch
            if (endpoint?.Platform != DevicePlatform.Twitch)
            {
                // Calculate the local presence (exclude Twitch endpoints)
                var localEndpoints = allEndpoints.Where(p => p.Platform != DevicePlatform.Twitch).ToArray();
                var localPresence = PresenceHelper.CalculatePresence("Interop", user, requestedStatus, localEndpoints, endpoint);
                var raiseStatus = localPresence.SaveLocalPresence(false);
                var localActivity = PresenceHelper.CalculateActivity("Interop", user, localEndpoints, endpoint);
                raiseStatus |= localActivity.SaveLocalStatus(false);

                if (raiseStatus)
                {
                    FilteredLogger.Log(user.UserID, "Raising Presence Status event", new { User = user.GetLogData() });
                    new PresenceStatusEventV2
                    {
                        UserID = user.UserID,
                        ChannelID = user.WatchingChannelIDLocal,
                        ConnectionStatus = user.ConnectionStatusLocal,
                        CustomStatusTimestamp = user.CustomStatusTimestamp,
                        GameID = user.CurrentGameIDLocal,
                        GameState = user.CurrentGameStateLocal,
                        GameStatusMessage = user.CurrentGameStatusMessageLocal,
                    }.Enqueue();
                }
            }

            // Sync external accounts as needed
            if (user.ShouldSyncExternalAccounts())
            {
                FilteredLogger.Log(user, "Queuing external account sync resolver", new { user = user.GetLogData() });
                ExternalAccountSyncResolver.Create(user.UserID);
            }

            FilteredLogger.Log(user, "Completed user presence update", new { user = user.GetLogData(), newPresence });
        }


    }
}
