﻿using System;
using System.Collections.Generic;
using System.Linq;
using Curse.Extensions;
using Curse.Friends.Configuration;
using Curse.Friends.Configuration.CurseVoiceService;
using Curse.Friends.Data;
using Curse.Friends.Data.Queues;
using Curse.Friends.Enums;
using Curse.Logging;

namespace Curse.Friends.WorkerService
{
    class VoicePermissionProcessor
    {
        private static readonly LogCategory Logger = new LogCategory("VoicePermissionProcessor") { AlphaLevel = LogLevel.Trace, ReleaseLevel = LogLevel.Debug};

        public static void Process(VoicePermissionsWorker worker)
        {
            
            try
            {
                if (worker.RootGroup.Type != GroupType.Large)
                {
                    return;
                }

                Logger.Trace("Processing " + worker.Type + " for " + worker.RootGroup.Title + (worker.Channel != null ? worker.Channel.Title : "Root"));

                switch (worker.Type)
                {
                    case VoicePermissionsWorkerType.ChannelPermissionsChanged:
                    {
                        if (worker.Channel == null)
                        {
                            Logger.Warn("Unable to process channel permissions changed. Channel is null.");
                            return;
                        }

                        // Get the list of users in calls
                        var allCallMembers = worker.RootGroup.GetServerCallMembers(0);
                        
                        // Get the list of channels potential affected by this
                        var voiceChannelMemberList = worker.Channel.GetAllChildren(false, true).ToDictionary(g => g.GroupID);

                        // Get all of these channels calls
                        var affectedCalls = allCallMembers.Where(c => voiceChannelMemberList.ContainsKey(c.Key) && c.Value.Members.Count > 0).ToArray();

                        Logger.Trace("Sending updated permissions for " + affectedCalls.Length + " channels.");

                        foreach (var groupCall in affectedCalls)
                        {
                            SendUpdatedPermissions(worker.RootGroup.GroupID, groupCall.Value);
                        }

                        break;
                    }
                    case VoicePermissionsWorkerType.RolePermissionsChanged:
                    {
                        // Get the list of users in calls
                        var voiceChannelMemberList = worker.RootGroup.GetServerCallMembers(0);

                        Logger.Trace("Sending updated permissions for " + voiceChannelMemberList.Count + " channels.");

                        // Update all calls
                        foreach (var channel in voiceChannelMemberList)
                        {
                            SendUpdatedPermissions(worker.RootGroup.GroupID, channel.Value);
                        }

                        break;
                    }
                    case VoicePermissionsWorkerType.MemberRoleChanged:
                    {
                        // Single user in single channel
                        var rootMember = worker.RootGroup.GetMember(worker.UserID);
                        if (rootMember == null)
                        {
                            Logger.Warn("Unable to process a member role change. The member was not found.");
                            return;
                        }

                        // Get the list of users in calls
                        var voiceChannelMemberList = worker.RootGroup.GetServerCallMembers(0);                        
                        var filteredVoiceChannelMemberList = voiceChannelMemberList.Where(c => c.Value.Members.Contains(worker.UserID)).ToArray();

                        Logger.Trace("Sending updated permissions for " + filteredVoiceChannelMemberList.Length + " channels.");

                        foreach (var channel in filteredVoiceChannelMemberList)
                        {
                            SendUpdatedPermissions(worker.RootGroup.GroupID, channel.Value);
                        }
                                                
                        break;
                    }
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to process voice permission worker", new {worker});
            }
        }

        private static void SendUpdatedPermissions(Guid serverID, GroupCallMemberList callMemberList)
        {
            if (callMemberList.Members.Count == 0)
            {
                Logger.Trace("Skipping update. No users connected: " + callMemberList.GroupID);
                return;    
            }

            var server = Group.GetByID(serverID);
            var channel = server.GetChildGroup(callMemberList.GroupID, true);
            if (channel == null)
            {
                Logger.Debug(string.Format("SendUpdatedPermissions - Channel {0} for server {1} was not found, skipping", callMemberList.GroupID, serverID));
                return;
            }
            
            if (channel.IsDeleted)
            {
                Logger.Debug(string.Format("SendUpdatedPermissions - Channel {0} for server {1} is deleted, skipping", callMemberList.GroupID, serverID));
                return;
            }
            
            var permissions = GetEffectivePermissions(server, channel, callMemberList.Members);
            CurseVoiceServiceClient.TryServiceCall("UpdatePermissions", svc => svc.UpdatePermissionsV2(FriendsServiceConfiguration.Instance.CentralServiceApiKey, callMemberList.GroupID, permissions));
            Logger.Trace("Updated permissions for " + callMemberList.Members.Count + " members in voice channel: " + channel.Title, permissions);
        }

        private static VoiceUserPermissions GetFallbackPermissions(int userID)
        {
            return new VoiceUserPermissions
            {
                UserID = userID,
                CanSpeak = true,
                CanModMute = false,
                CanModDeafen = false,
                CanModKick = false,
                BestRoleRank = GroupRole.DefaultRoleRank
            };
        }

        private static VoiceUserPermissions[] GetEffectivePermissions(Group server, Group channel, HashSet<int> userIDs)
        {            
            // Get all of the group member records
            if (server == null)
            {
                Logger.Debug("GetEffectivePermissions - Root Group is null, allowing everyone to speak but not mod", new {channel, userIDs});
                return userIDs.Select(GetFallbackPermissions).ToArray();
            }
            var rootMembers = server.GetMembers(userIDs);

            if (channel == null)
            {
                Logger.Debug("GetEffectivePermissions - Channel is null, allowing root members to speak but not mod", new { server, userIDs });
                return rootMembers.Select(rm => GetFallbackPermissions(rm.UserID)).ToArray();
            }
            var channelMemberships = channel.GetMembers(userIDs).ToDictionary(p => p.UserID);
            
            var permissionsList = new List<VoiceUserPermissions>();
            var channelRolePermissions = channel.GetAllRolePermissionsForGroup().ToDictionary(p => p.RoleID);

            Logger.Trace("GetEffectivePermissions - Role Permissions", channelRolePermissions);

            foreach(var rootMember in rootMembers)
            {
                var channelMember = channelMemberships.GetValueOrDefault(rootMember.UserID);
                var channelMemberRolePermissions = channelRolePermissions.Where(p => p.Value.RoleID == server.DefaultRoleID || (rootMember.Roles != null && rootMember.Roles.Contains(p.Key)))
                    .Select(p => p.Value).ToArray();

                var permissions = new VoiceUserPermissions
                {
                    UserID = rootMember.UserID,
                    BestRoleRank = rootMember.BestRoleRank,
                    CanModDeafen = channel.HasPermission(GroupPermissions.VoiceDeafenUser, channelMember, channelMemberRolePermissions),
                    CanModMute = channel.HasPermission(GroupPermissions.VoiceMuteUser, channelMember, channelMemberRolePermissions),
                    CanModKick = channel.HasPermission(GroupPermissions.VoiceKickUser, channelMember, channelMemberRolePermissions),
                    CanSpeak = channel.HasPermission(GroupPermissions.VoiceSpeak, channelMember, channelMemberRolePermissions),
                };

                permissionsList.Add(permissions);                
            }

            Logger.Trace("GetEffectivePermissions - Permissions List", permissionsList);

            return permissionsList.ToArray();
        }
    }
}
