using System;
using System.IO;
using System.Linq;
using System.Reflection;
using Curse.Friends.Data;
using Curse.Friends.Data.Queues;
using Curse.Logging;
using Curse.Friends.Configuration;
using Curse.Friends.Data.Models;
using Curse.Friends.Enums;
using Curse.Friends.TwitchApi;
using Curse.Friends.WorkerService.Processors;
using Curse.Logging.Uploader;
using Curse.Voice.Helpers;
using Curse.CloudServices.Jobs;
using Curse.Friends.WorkerService.Partners;

namespace Curse.Friends.WorkerService
{
    public static class WorkerServer
    {
        public static readonly LogCategory Logger = new LogCategory("WorkerServer");
        public static void Start()
        {            
            CheckTwitchApiAccess();
            InitializePushNotifications();
            StartQueueProcessors();
            JobScheduler.Initialize(StorageConfiguration.CurrentRegion.IsDefault);
        }

        private static void InitializePushNotifications()
        {
            PushNotificationConfig config;

            try
            {
                var binExe = new FileInfo(Assembly.GetExecutingAssembly().Location);

                if (!binExe.Exists || binExe.Directory == null)
                {
                    Logger.Warn("Unable to determine application folder path for push notification!", binExe.FullName);
                    return;
                }

                var binPath = binExe.Directory.FullName;

                var generalPath = Path.Combine(binPath, "Certs", "Apple.p12");
                var generalCertBytes = File.ReadAllBytes(generalPath);
                var voicePath = Path.Combine(binPath, "Certs", "AppleVoice.p12");
                var voiceCertBytes = File.ReadAllBytes(voicePath);
                
                // Read in the appropriate cert
                config = new PushNotificationConfig
                {
                    AppleGeneralPushCert = generalCertBytes,
                    AppleGeneralPushCertPassword = "Production-CurseAndAppleLovePushNotifications!",
                    AppleVoicePushCert = voiceCertBytes,
                    AppleVoicePushCertPassword = "L*VkeVJ;Bsr'dms4rePY)QxJJUaV#44",
                    GooglePushApiKey = "AAAAi3Jy3H8:APA91bFiHSr-Tb1fvuQrgSVwT-wJKaZoXnmwZChVBIZwUKI8rW_9v1w1nk-8MnwoRBqZSM0X2XsW8zRo5HV4E5bhcIj0vRVVEqUDkFJaNljwXjfnN6e8CQCMcnuu3S3NKbQqL94Jth-A"
                };
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to initialize push configuration!");
                return;
            }

            try
            {
                PushNotificationProcessor.Initialize(config);
            }
            catch (Exception ex)
            {

                Logger.Error(ex, "Failed to initialize push services!"); ;
            }

        }

        private static void StartQueueProcessors()
        {
            // Presence
            UserStatusResolver.StartProcessor(UserStatusProcessor.ProcessMessage);
            RegionalUserStatusResolver.StartProcessor(UserStatusProcessor.ProcessRegionalMessage);
            UserActivityResolver.StartProcessor(UserActivityResolverProcessor.Process);
            UserPresenceResolver.StartProcessor(UserPresenceResolverProcessor.Process);

            AccountRenameWorker.StartProcessor();
            FriendHintSearchIndexer.StartProcessor();
            FriendshipRemovedResolver.StartProcessor();
            FriendshipChangeResolver.StartProcessor();
            RegionalUserChangeResolver.StartProcessor();
            FriendshipSuggestionWorker.StartProcessor();
            FriendListSearchWorker.StartProcessor();
            ClientEndpointResolver.StartProcessor();
            PartnerServerJoinedProcessor.Initialize();
            GroupMemberWorker.StartProcessor(GroupMemberWorkerProcessor.Process);
            GroupMemberIndexWorker.StartProcessor();
            GroupSearchIndexWorker.StartProcessor();
            GroupBannedUserIndexWorker.StartProcessor();
            GroupEventWorker.StartProcessor();
            VoicePermissionsWorker.StartProcessor(VoicePermissionProcessor.Process);
            UserDissolveWorker.StartProcessor();

            ExternalCommunityIndexProcessor.Initialize();
            ExternalCommunityIndexWorker.StartProcessor(ExternalCommunityIndexProcessor.Process);
            
            PushNotificationWorker.StartProcessor(PushNotificationProcessor.ProcessItem);        
            ExternalAccountSyncResolver.StartProcessor(ExternalAccountSyncProcessor.Process);

            // Private Message Processor
            PrivateMessageProcessor.Initialize(FriendsServiceConfiguration.Instance.TwitchShimApiKey,
                FriendsServiceConfiguration.Instance.TwitchShimUrl,
                FriendsServiceConfiguration.Instance.MaxWhispersPerSecond,
                FriendsServiceConfiguration.Instance.MaxWhispersPerMinute);
            PrivateMessageWorker.StartProcessor(PrivateMessageProcessor.Process);
            ExternalPrivateMessageWorker.StartProcessor(PrivateMessageProcessor.ProcessExternal);

            ConversationMessageWorker.StartProcessor();
            ConversationMessageBulkWorker.StartProcessor();
            ConversationReadWorker.StartProcessor(ConversationReadProcessor.Process);
            CallResolver.StartProcessor(CallResolver_ProcessMessage);
            CallRespondedResolver.StartProcessor(CallRespondedResolver_ProcessMessage);

            GroupIntegrityWorker.StartProcessor(GroupIntegrityWorkerProcessor.Process);
            
            // External Communities
            ExternalCommunitySyncWorker.StartProcessor();
            ExternalCommunityMemberIndexWorker.StartProcessor();
            ExternalCommunityMemberSyncWorker.StartProcessor(ExternalCommunityMemberSyncProcessor.Process);
            ExternalUserSyncWorker.StartProcessor(ExternalUserSyncProcessor.Process);
            TwitchAccountResolver.StartProcessor();

            // External Guilds
            ExternalGuildSyncWorker.StartProcessor();
            ExternalGuildMemberSyncWorker.StartProcessor();
            ExternalGuildUserSyncWorker.StartProcessor();

            // Region Changer
            UserRegionChangeWorker.StartProcessor(UserRegionChangeProcessor.Process);

            SyncAvatarToTwitchWorker.StartProcessor(SyncAvatarToTwitchProcessor.Process);
        }

        static void CheckTwitchApiAccess()
        {
            try
            {
                using (var client = new CurseShimClient(FriendsServiceConfiguration.Instance.TwitchShimApiKey, FriendsServiceConfiguration.Instance.TwitchShimUrl))
                {
                    var resp = client.CheckAccess();
                    if (resp.Status != TwitchResponseStatus.Success)
                    {
                        Logger.Fatal("Unable to successfully communicate with Twitch Shim API", resp);
                    }
                    else
                    {
                        Logger.Info("Confirmed access to the Twitch Shim API. ");
                    }
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to check Twitch API access.");
            }
            
        }

        static void CallResolver_ProcessMessage(CallResolver e)
        {
            var callDetails = e.CallDetails;
            var accessToken = AccessTokenHelper.CreateAccessToken(callDetails.CallID, e.RecipientID);

            Logger.Debug("Processing call resolver...", e);

            // Get this user's connected endpoints
            ClientEndpoint.DispatchNotification(e.RecipientID, endpoint =>
            {                
                var notification = callDetails.ToNotification(accessToken, e.SenderID, e.SenderName);
                Logger.Debug("Notifying user of a call", notification);
                CallNotifier.Create(endpoint, notification);
            }, endpoint =>
            {                
                PushNotificationWorker.VoiceInvite(StorageConfiguration.CurrentRegion.ID, endpoint, e.SenderID, e.SenderName, e.RecipientID, callDetails, accessToken);
            });

            if (callDetails.Type == CallType.Friend)
            {
                var region = UserRegion.GetLocal(e.RecipientID);
                if (region == null)
                {
                    Logger.Debug("Recipient user region does not exist", e);
                    return;
                }
                var friendship = Friendship.Get(region.RegionID, e.RecipientID, e.SenderID);
                if (friendship == null)
                {
                    Logger.Debug("Recipient has no friendship with sender", e);
                    return;
                }
                if (friendship.DateMessaged < callDetails.Timestamp)
                {
                    friendship.DateMessaged = callDetails.Timestamp;
                    friendship.Update(f => f.DateMessaged);
                }
            }
        }

        static void CallRespondedResolver_ProcessMessage(CallRespondedResolver e)
        {            

            // Get this user's connected endpoints
            ClientEndpoint.DispatchNotification(e.UserID, endpoint =>
            {
                CallRespondedNotifier.Create(endpoint, e.Notification);
            });
        }
       
        public static void Stop()
        {
            try
            {
                PushNotificationProcessor.Stop();
            }
            catch (Exception ex)
            {

                Logger.Error(ex);
            }
            try
            {
                ExternalCommunityIndexProcessor.Stop();
            }
            catch (Exception ex)
            {
                Logger.Error(ex);
            }
            try
            {
                StorageConfiguration.Shutdown();
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to close all queue connections!");
            }

            try
            {
                LogUploader.Shutdown();
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to finish log upload!");
            }
        }
    }
}
