﻿using System.Security.Authentication;
using Curse.Friends.NotificationContracts;
using Curse.Logging;
using Curse.SocketInterface;
using Curse.SocketMessages;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Net;
using System.Net.NetworkInformation;
using System.Security.Cryptography;
using System.Text;
using System.Threading;
using Curse.Friends.Enums;
using WebSocketSharp;
using Newtonsoft.Json;
using Newtonsoft.Json.Linq;
using Logger = Curse.Logging.Logger;

namespace Curse.Friends.Client
{
    public class NotificationMessageWrapper<T> where T : Contract<T>, new()
    {
        public int TypeID { get; set; }
        public T Body { get; set; }

        public NotificationMessageWrapper(T body)
        {
            TypeID = Contract<T>.MessageType;
            Body = body;
        }

        public override string ToString()
        {
            var output = JsonConvert.SerializeObject(this);
            if (output == null)
            {
                Logger.Warn("Unable to serialize the notification message.", this);
            }

            return output;
        }
    }

    public enum ConnectionState
    {
        Disconnected = 1,
        Connecting,
        Joining,
        Handshaking,
        Connected
    }

    public class FriendsNotificationClient : IDisposable
    {
        private static readonly LogCategory Logger = new LogCategory("FriendsNotificationClient") { AlphaLevel = Logging.LogLevel.Trace };

        public ConnectionState ConnectionState = ConnectionState.Disconnected;
        private Thread _handshakeThread;
        public WebSocket WebSocket { get; set; }
        public bool IsConnected { get; set; }

        #region Connection

        private FriendsNotificationClient(string connectionUrl)
        {
            WebSocket = new WebSocket(connectionUrl);
            WebSocket.OnMessage += OnMessageReceived;
            WebSocket.OnClose += Client_Disconnected;
            WebSocket.OnOpen += WebSocket_OnOpen;
            WebSocket.OnError += WebSocket_OnError;

            AddContractDispatcher<ConversationReadNotification>(OnConversationReadNotification);

            AddContractDispatcher<FriendshipChangeNotification>(OnFriendshipChangeNotification);
            AddContractDispatcher<FriendshipRemovedNotification>(OnFriendshipRemovedNotification);
            AddContractDispatcher<FriendSuggestionNotification>(OnSuggestedFriendsNotification);
            AddContractDispatcher<FriendshipStatusNotification>(OnFriendshipStatusNotification);

            AddContractDispatcher<InstantMessageNotification>(OnInstantMessageNotification);
            AddContractDispatcher<GroupMessageNotification>(OnGroupMessageNotification);
            AddContractDispatcher<ConversationMessageNotification>(OnConversationMessageNotification);
            AddContractDispatcher<ConversationMessageResponse>(OnConversationMessageResponse);
            AddContractDispatcher<InstantMessageResponse>(OnInstantMessageResponse);
            AddContractDispatcher<JoinResponse>(OnJoin);

            AddContractDispatcher<GroupChangeNotification>(OnGroupChangeNotification);
            AddContractDispatcher<GroupMessageResponse>(OnGroupMessageResponse);
            AddContractDispatcher<GroupPreferenceNotification>(OnGroupPreferenceNotification);
            AddContractDispatcher<GroupVoiceInvitationResponse>(OnGroupVoiceInvitationResponse);
            AddContractDispatcher<GroupVoiceInvitationNotification>(OnGroupVoiceInvitationNotification);
            AddContractDispatcher<GroupVoiceDeclineNotification>(OnGroupVoiceDeclineNotification);

            AddContractDispatcher<UserChangeNotification>(OnUserChangeNotification);
            AddContractDispatcher<VoiceAcceptNotification>(OnVoiceAcceptNotification);
            AddContractDispatcher<VoiceDeclineNotification>(OnVoiceDeclineNotification);
            AddContractDispatcher<VoiceInvitationResponse>(OnVoiceInvitationResponse);
            AddContractDispatcher<VoiceInvitationNotification>(OnVoiceInvitationNotification);
            AddContractDispatcher<OfflineMessageNotification>(OnOfflineMessageNotification);
            AddContractDispatcher<OfflineMessageResponse>(OnOfflineMessageResponse);
        }

        static void RunPingTest(string ipAddress)
        {
            // Try to ping the server:
            try
            {
                using (var pingTest = new Ping())
                {
                    var options = new PingOptions
                    {
                        DontFragment = true
                    };

                    // Create a buffer of 32 bytes of data to be transmitted.                 
                    var buffer = Encoding.ASCII.GetBytes("CurseVoicePingTest");
                    var reply = pingTest.Send(ipAddress, 500, buffer, options);

                    if (reply != null)
                    {
                        Logger.Info("Ping test analysis of connectivity to " + ipAddress, new { reply.Status, reply.RoundtripTime });
                    }
                    else
                    {
                        Logger.Warn("Ping test failed to return a reply.");
                    }

                }
            }
            catch (Exception ex)
            {
                Logger.Warn(ex, "Ping test failed with an unhandled exception!");
            }
        }

        public static FriendsNotificationClient Connect(NotificationWebSocketConnectionInfo connectionInfo, out JoinResponse joinResponse)
        {
            foreach (var host in connectionInfo.NotificationHosts)
            {
                connectionInfo.LastConnectAttempt = DateTime.UtcNow;

                // Try each port
                var client = DoConnect(host, connectionInfo, out joinResponse);

                // We have a usable client!
                if (client != null)
                {
                    return client;
                }

                // We received a join response that indicates that future attempts will not work, so don't even try another
                //if (joinResponse != null &&
                //    (joinResponse.Status != JoinStatus.FailedUnhandledException &&
                //        joinResponse.Status != JoinStatus.Timeout))
                //{
                //    continue;
                //}

                // host failed, so run ping test
                
        }

            foreach (var host in connectionInfo.NotificationHosts)
        {
                RunPingTest(host);
            }

            joinResponse = new JoinResponse { Status = JoinStatus.Timeout };
            return null;
        }

        private static FriendsNotificationClient DoConnect(string host, NotificationWebSocketConnectionInfo connectionInfo, out JoinResponse joinResponse)
        {
            var sw = Stopwatch.StartNew();
            Logger.Info("Attempting notification server connection.", host);

            var client = new FriendsNotificationClient(host);
            joinResponse = client.TryConnect(connectionInfo);

            if (joinResponse.Status == JoinStatus.Successful)
            {
                Logger.Info("Notification server connection successful! Completed in " + sw.Elapsed.TotalSeconds.ToString("###,##0.00") + " seconds");
                return client;
            }

            Logger.Warn("Notification server connection failed!", joinResponse);
            return null;
        }

        protected JoinResponse TryConnect(NotificationWebSocketConnectionInfo connectionInfo)
        {
            var response = new JoinResponse { Status = JoinStatus.Timeout };

            try
            {
                if (Connect())
                {
                    response = Join(connectionInfo);

                    if (response.Status == JoinStatus.Successful)
                    {
                        _handshakeThread = new Thread(HandshakeThread) { IsBackground = true };
                        _handshakeThread.Start();
                    }
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Unhandled exception occurred while connecting to a voice server.");

            }
            finally
            {
                if (response.Status != JoinStatus.Successful)
                {
                    Dispose();
                }
            }

            return response;

        }

        protected bool Connect()
        {
            //Connect is synchronous
            WebSocket.Connect();
            return WebSocket.IsAlive;
        }

        public void Disconnect()
        {
            try
            {
                if (WebSocket != null && WebSocket.IsAlive)
                {
                    WebSocket.Close();
                }

                ConnectionState = ConnectionState.Disconnected;
                IsConnected = false;
            }
            catch (Exception ex)
            {
                Logger.Error(ex);
            }
        }

        protected void WebSocket_OnError(object sender, ErrorEventArgs e)
        {
            try
            {
                Logger.Error(e.Exception, e.Message);
                //TODO: try to rejoin if DCed
                if (WebSocket != null && WebSocket.IsAlive)
                {
                    WebSocket.Close();
                }
                ConnectionState = ConnectionState.Disconnected;
                IsConnected = false;
            }
            catch (Exception ex)
            {
                Logger.Error(ex);
            }
        }

        protected void WebSocket_OnOpen(object sender, EventArgs e)
        {
            try
            {
                IsConnected = true;
            }
            catch (Exception ex)
            {
                Logger.Error(ex);
            }
        }

        private ManualResetEvent _joinResetEvent;
        private readonly object _joinSync = new object();
        private JoinResponse _joinResponse;

        private JoinResponse Join(NotificationWebSocketConnectionInfo connectionInfo)
        {
            ConnectionState = ConnectionState.Joining;
            _joinResponse = null;

            try
            {
                var joinRequest = new JoinRequest
                {
                    SessionID = connectionInfo.SessionID,
                    MachineKey = connectionInfo.MachineKey,
                    UserID = connectionInfo.UserID,
                    Status = connectionInfo.DesiredStatus,
                    ClientVersion = connectionInfo.ClientVersion,
                    PublicKey = null,
                    CipherAlgorithm = 0,
                    CipherStrength = 0,
                };

                var requestMessage = new NotificationMessageWrapper<JoinRequest>(joinRequest);

                _joinResetEvent = new ManualResetEvent(false);
                ConnectionState = ConnectionState.Connecting;

                Logger.Info("Sending join session request...", joinRequest);
                
                WebSocket.Send(requestMessage.ToString());

                _joinResetEvent.WaitOne(10000); // Wait up to 10 seconds for the join call to complete                                
                _joinResetEvent.Dispose();
                _joinResetEvent = null;

                var resp = _joinResponse;
                if (resp == null)
                {
                    Logger.Warn("Join session request timed out.");
                    return new JoinResponse { Status = JoinStatus.Timeout };
                }

                if (resp.Status != JoinStatus.Successful)
                {
                    Logger.Warn("Join session request failed.", new { resp.Status });
                    return resp;
                }

                ConnectionState = ConnectionState.Connected;

                return resp;
            }
            catch (Exception ex)
            {
                Logger.Error(ex);
            }
            finally
            {
                lock (_joinSync)
                {
                    _joinResetEvent = null;
                }
            }

            return _joinResponse ?? new JoinResponse { Status = JoinStatus.Timeout };
        }

        void Client_Disconnected(object sender, CloseEventArgs e)
        {
            Logger.Info("Friends notification server disconnected. Reason: " + e.Reason);

            lock (_joinSync)
            {
                if (_joinResetEvent != null)
                {
                    _joinResetEvent.Set();
                }
            }

            ConnectionState = ConnectionState.Disconnected;
            IsConnected = false;

            var handler = ClientDisconnected;
            if (handler != null)
            {
                handler(this, e);
            }
        }

        #endregion

        #region Events

        public static event EventHandler<CloseEventArgs> ClientDisconnected;
        public static event EventHandler<EventArgs<FriendshipStatusNotification>> FriendshipStatusNotification;
        public static event EventHandler<EventArgs<VoiceInvitationResponse>> VoiceInvitationResponse;
        public static event EventHandler<EventArgs<VoiceInvitationNotification>> VoiceInvitationNotification;
        public static event EventHandler<EventArgs<VoiceDeclineNotification>> VoiceDeclineNotification;
        public static event EventHandler<EventArgs<InstantMessageNotification>> InstantMessageNotification;
        public static event EventHandler<EventArgs<InstantMessageResponse>> InstantMessageResponse;
        public static event EventHandler<EventArgs<FriendshipChangeNotification>> FriendshipChangeNotification;
        public static event EventHandler<EventArgs<FriendshipRemovedNotification>> FriendshipRemovedNotification;
        public static event EventHandler<EventArgs<UserChangeNotification>> UserChangeNotification;
        public static event EventHandler<EventArgs<ConversationReadNotification>> ConversationReadNotification;
        public static event EventHandler<EventArgs<FriendSuggestionNotification>> SuggestedFriendsNotification;
        public static event EventHandler<EventArgs<GroupChangeNotification>> GroupChangeNotification;
        public static event EventHandler<EventArgs<GroupMessageNotification>> GroupMessageNotification;
        public static event EventHandler<EventArgs<GroupMessageResponse>> GroupMessageResponse;
        public static event EventHandler<EventArgs<GroupVoiceInvitationResponse>> GroupVoiceInvitationResponse;
        public static event EventHandler<EventArgs<GroupVoiceInvitationNotification>> GroupVoiceInvitationNotification;
        public static event EventHandler<EventArgs<GroupVoiceDeclineNotification>> GroupVoiceDeclineNotification;
        public static event EventHandler<EventArgs<OfflineMessageNotification>> OfflineMessageNotification;
        public static event EventHandler<EventArgs<OfflineMessageResponse>> OfflineMessageResponse;
        public static event EventHandler<EventArgs<VoiceAcceptNotification>> VoiceAcceptNotification;
        public static event EventHandler<EventArgs<GroupPreferenceNotification>> GroupPreferenceNotification;

        #endregion

        #region Responses

        private void OnJoin(JoinResponse response)
        {
            _joinResponse = response;

            lock (_joinSync)
            {
                if (_joinResetEvent != null)
                {
                    _joinResetEvent.Set();
                }
            }
        }

        private void OnFriendshipStatusNotification(FriendshipStatusNotification response)
        {
            var handler = FriendshipStatusNotification;
            if (handler != null)
            {
                handler(this, new EventArgs<FriendshipStatusNotification>(response));
            }
        }


        /// <summary>
        /// Received after sending a voice invitation to a user
        /// </summary>
        /// <param name="response"></param>
        private void OnVoiceInvitationResponse(VoiceInvitationResponse response)
        {
            var handler = VoiceInvitationResponse;
            if (handler != null)
            {
                handler(this, new EventArgs<VoiceInvitationResponse>(response));
            }
        }

        /// <summary>
        /// Notification when a user is invited to a voice session.
        /// </summary>
        /// <param name="notification"></param>
        private void OnVoiceInvitationNotification(VoiceInvitationNotification notification)
        {
            var handler = VoiceInvitationNotification;
            if (handler != null)
            {
                handler(this, new EventArgs<VoiceInvitationNotification>(notification));
            }
        }

        /// <summary>
        /// Notification when a user is invited to a voice session.
        /// </summary>
        /// <param name="notification"></param>
        private void OnVoiceDeclineNotification(VoiceDeclineNotification notification)
        {
            var handler = VoiceDeclineNotification;
            if (handler != null)
            {
                handler(this, new EventArgs<VoiceDeclineNotification>(notification));
            }
        }

        /// <summary>
        /// Notification when user's connection status changes
        /// </summary>
        /// <param name="notification"></param>
        private void OnInstantMessageNotification(InstantMessageNotification notification)
        {
            var handler = InstantMessageNotification;
            if (handler != null)
            {
                handler(this, new EventArgs<InstantMessageNotification>(notification));
            }
        }


        /// <summary>
        /// Notification when user's connection status changes
        /// </summary>
        /// <param name="notification"></param>
        private void OnInstantMessageResponse(InstantMessageResponse response)
        {
            var handler = InstantMessageResponse;
            if (handler != null)
            {
                handler(this, new EventArgs<InstantMessageResponse>(response));
            }
        }

        /// <summary>
        /// Notification when user's connection status changes
        /// </summary>
        /// <param name="ISocketInterface"></param>
        /// <param name="notification"></param>
        private void OnFriendshipChangeNotification(FriendshipChangeNotification notification)
        {
            var handler = FriendshipChangeNotification;
            if (handler != null)
            {
                handler(this, new EventArgs<FriendshipChangeNotification>(notification));
            }
        }

        private void OnFriendshipRemovedNotification(FriendshipRemovedNotification notification)
        {
            var handler = FriendshipRemovedNotification;
            if (handler != null)
            {
                handler(this, new EventArgs<FriendshipRemovedNotification>(notification));
            }
        }

        /// <summary>
        /// Notifies the client that a conversation has been read, up to a certain timestamp.
        /// </summary>
        /// <param name="ISocketInterface"></param>
        /// <param name="notification"></param>
        private void OnConversationReadNotification(ConversationReadNotification notification)
        {
            var handler = ConversationReadNotification;
            if (handler != null)
            {
                handler(this, new EventArgs<ConversationReadNotification>(notification));
            }
        }

        private void OnSuggestedFriendsNotification(FriendSuggestionNotification notification)
        {
            var handler = SuggestedFriendsNotification;
            if (handler != null)
            {
                handler(this, new EventArgs<FriendSuggestionNotification>(notification));
            }
        }

        /// <summary>
        /// Notifies the client of a change to their own user model.
        /// </summary>
        /// <param name="ISocketInterface"></param>
        /// <param name="notification"></param>
        private void OnUserChangeNotification(UserChangeNotification notification)
        {
            var handler = UserChangeNotification;
            if (handler != null)
            {
                handler(this, new EventArgs<UserChangeNotification>(notification));
            }
        }

        private void OnGroupChangeNotification(GroupChangeNotification notification)
        {
            var handler = GroupChangeNotification;
            if (handler != null)
            {
                handler(this, new EventArgs<GroupChangeNotification>(notification));
            }
        }

        private void OnConversationMessageResponse(ConversationMessageResponse response)
        {
            Guid guid;
            if (Guid.TryParseExact(response.ConversationID, "D", out guid))
            {
                var notification = new GroupMessageResponse { ClientID = response.ClientID, GroupID = guid, ClientName ="", GroupName = "", Status = response.Status};
                var handler = GroupMessageResponse;
                if (handler != null)
                {
                    handler(this, new EventArgs<GroupMessageResponse>(notification));
                }
            }
        }

        private void OnGroupMessageResponse(GroupMessageResponse notification)
        {
            var handler = GroupMessageResponse;
            if (handler != null)
            {
                handler(this, new EventArgs<GroupMessageResponse>(notification));
            }
        }

        //only throw the message event if it's a group
        private void OnConversationMessageNotification(ConversationMessageNotification response)
        {
            Guid guid;
            if (Guid.TryParseExact(response.ContactID, "D", out guid))
            {
                var handler = GroupMessageNotification;
                var clientID = response.ClientID == null ? Guid.Empty : Guid.Parse(response.ClientID);
                var serverID = response.ServerID == null ? Guid.Empty : Guid.Parse(response.ServerID);
                var timestamp = ConvertTimestamp(response.Timestamp);
                var notification = new GroupMessageNotification() { ClientMessageID = clientID, GroupID = guid, Message = response.Body, SenderID = response.SenderID, ServerMessageID = serverID, Timestamp = timestamp };
                if (handler != null)
                {
                    handler(this, new EventArgs<GroupMessageNotification>(notification));
                }
            }
        }

        private DateTime ConvertTimestamp(long timestamp)
        {
            return new DateTime(1970, 1, 1) + new TimeSpan(timestamp * 10000);
        }


        private void OnGroupMessageNotification(GroupMessageNotification notification)
        {
            var handler = GroupMessageNotification;
            if (handler != null)
            {
                handler(this, new EventArgs<GroupMessageNotification>(notification));
            }
        }

        private void OnGroupVoiceInvitationResponse(GroupVoiceInvitationResponse notification)
        {
            var handler = GroupVoiceInvitationResponse;
            if (handler != null)
            {
                handler(this, new EventArgs<GroupVoiceInvitationResponse>(notification));
            }
        }

        private void OnGroupVoiceInvitationNotification(GroupVoiceInvitationNotification notification)
        {
            var handler = GroupVoiceInvitationNotification;
            if (handler != null)
            {
                handler(this, new EventArgs<GroupVoiceInvitationNotification>(notification));
            }
        }
        private void OnGroupVoiceDeclineNotification(GroupVoiceDeclineNotification notification)
        {
            var handler = GroupVoiceDeclineNotification;
            if (handler != null)
            {
                handler(this, new EventArgs<GroupVoiceDeclineNotification>(notification));
            }
        }

        private void OnOfflineMessageNotification(OfflineMessageNotification notification)
        {
            var handler = OfflineMessageNotification;
            if (handler != null)
            {
                handler(this, new EventArgs<OfflineMessageNotification>(notification));
            }
        }

        private void OnOfflineMessageResponse(OfflineMessageResponse response)
        {
            var handler = OfflineMessageResponse;
            if (handler != null)
            {
                handler(this, new EventArgs<OfflineMessageResponse>(response));
            }
        }

        private void OnVoiceAcceptNotification(VoiceAcceptNotification notification)
        {
            var handler = VoiceAcceptNotification;
            if (handler != null)
            {
                handler(this, new EventArgs<VoiceAcceptNotification>(notification));
            }
        }

        private void OnGroupPreferenceNotification(GroupPreferenceNotification notification)
        {
            var handler = GroupPreferenceNotification;
            if (handler != null)
            {
                handler(this, new EventArgs<GroupPreferenceNotification>(notification));
            }
        }

        #endregion

        #region Requests

        public void SendGroupVoiceInvitation(Guid groupID, string inviteUrl)
        {
            var request = new GroupVoiceInvitationRequest() { GroupID = groupID, InviteUrl = inviteUrl };
            WebSocket.Send(new NotificationMessageWrapper<GroupVoiceInvitationRequest>(request).ToString());
        }

        public void SendGroupVoiceDecline(Guid groupID, string inviteUrl)
        {
            var request = new GroupVoiceDeclineRequest() { GroupID = groupID, InviteUrl = inviteUrl };
            WebSocket.Send(new NotificationMessageWrapper<GroupVoiceDeclineRequest>(request).ToString());
        }

        public void SendVoiceInvitation(int friendID, string inviteUrl)
        {
            var request = new VoiceInvitationRequest { FriendID = friendID, InviteUrl = inviteUrl };
            WebSocket.Send(new NotificationMessageWrapper<VoiceInvitationRequest>(request).ToString());
        }

        public void SendVoiceDecline(int friendID, string inviteUrl)
        {
            var request = new VoiceDeclineRequest { FriendID = friendID, InviteUrl = inviteUrl };
            WebSocket.Send(new NotificationMessageWrapper<VoiceDeclineRequest>(request).ToString());
        }

        public void SendInstantMessage(int friendID, string message, Guid clientID)
        {
            var request = new InstantMessageRequest { FriendID = friendID, Message = message, ClientID = clientID };
            WebSocket.Send(new NotificationMessageWrapper<InstantMessageRequest>(request).ToString());
        }

        public void SendConversationRead(Guid groupID, DateTime timestamp)
        {
            var request = new ConversationReadRequest { GroupID = groupID, FriendID = 0, Timestamp = timestamp };
            WebSocket.Send(new NotificationMessageWrapper<ConversationReadRequest>(request).ToString());
        }

        public void SendConversationRead(int friendID, DateTime timestamp)
        {
            var request = new ConversationReadRequest { GroupID = Guid.Empty, FriendID = friendID, Timestamp = timestamp };
            WebSocket.Send(new NotificationMessageWrapper<ConversationReadRequest>(request).ToString());
        }

        public void SendGroupMessage(Guid groupID, string message, Guid clientID)
        {
            //var request = new GroupMessageRequest { GroupID = groupID, Message = message, ClientID = clientID };
            var request = new ConversationMessageRequest { ClientID = clientID, ConversationID = groupID.ToString(), Message = message};
            WebSocket.Send(new NotificationMessageWrapper<ConversationMessageRequest>(request).ToString());
        }

        //Offline message requests are DEPRECATED. use Friends service COnversationHistory call instead
        //public void SendOfflineMessageRequest(Guid requestID, DateTime minDate, DateTime? maxDate = null)
        //{
        //    var request = new OfflineMessageRequest { RequestID = requestID, MinDate = minDate, MaxDate = maxDate };
        //    WebSocket.Send(new NotificationMessageWrapper<OfflineMessageRequest>(request).ToString());
        //}

        //public void SendOfflineMessageRequest(Guid requestID, DateTime minDate, Guid groupID, DateTime? maxDate = null)
        //{
        //    var request = new OfflineMessageRequest { GroupID = groupID, RequestID = requestID, MinDate = minDate, MaxDate = maxDate };
        //    WebSocket.Send(new NotificationMessageWrapper<OfflineMessageRequest>(request).ToString());
        //}

        //public void SendOfflineMessageRequest(Guid requestID, DateTime minDate, int friendID, DateTime? maxDate = null)
        //{
        //    var request = new OfflineMessageRequest { FriendID = friendID, RequestID = requestID, MinDate = minDate, MaxDate = maxDate };
        //    WebSocket.Send(new NotificationMessageWrapper<OfflineMessageRequest>(request).ToString());
        //}

        #endregion

        #region Base Implementation

        private void OnMessageReceived(object sender, WebSocketSharp.MessageEventArgs messageEventArgs)
        {
            if (!IsConnected)
            {
                return;
            }

            try
            {
                var wrapper = JObject.Parse(messageEventArgs.Data);
                var type = JsonConvert.DeserializeObject<int>(wrapper["TypeID"].ToString());
            
                IWebSocketContractDispatcher dispatcher = null;
                if (_contractDispatchers.TryGetValue(type, out dispatcher))
            {
                try
                {
                        dispatcher.Dispatch(wrapper["Body"].ToString());
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Dispatcher callback failed for message type: " + dispatcher.TypeName);
#if DEBUG
                    Debugger.Break();
#endif
                }

            }
        }
            catch (Exception ex)
            {
                Logger.Error(ex, "Error deserializing notification message data.", messageEventArgs.Data);
            }
        }

        private readonly Dictionary<int, IWebSocketContractDispatcher> _contractDispatchers = new Dictionary<int, IWebSocketContractDispatcher>();

        private void AddContractDispatcher<T>(Action<T> handler) where T : Contract<T>, new()
        {
            int messageType = Contract<T>.MessageType;
            IWebSocketContractDispatcher dispatcher = new WebsocketContractDispatcher<T>(handler);
            _contractDispatchers.Add(messageType, dispatcher);

        }

        public void Dispose()
        {
            try
            {
                if (WebSocket != null)
                {
                    WebSocket.OnClose -= ClientDisconnected;
                    WebSocket.OnOpen -= WebSocket_OnOpen;
                    WebSocket.OnMessage -= OnMessageReceived;
                    WebSocket.OnError -= WebSocket_OnError;
                    if (WebSocket.IsAlive)
                    {
                        WebSocket.Close();
                    }
                    WebSocket = null;
                }

                if (_handshakeThread != null)
                {
                    try
                    {
                        _stopHandshakeThread = true;
                        if (!_handshakeThread.Join(200))
                        {
                            Logger.Warn("Handshake thread failed to exit cleanly, aborting...");
                            _handshakeThread.Abort();
                        }
                    }
                    catch (Exception ex)
                    {
                        Logger.Error(ex, "Failed to abort handshake thread!");
                    }
                }

            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to cleanly dispose FriendsNotificationClient!");
            }
        }

        private volatile bool _stopHandshakeThread;
        private readonly TimeSpan _handshakeInterval = TimeSpan.FromSeconds(10);

        [DebuggerStepThrough]
        private void HandshakeThread()
        {
            var lastHandshake = DateTime.UtcNow.AddSeconds(5); // Start handshaking 5 seconds after connecting

            while (!_stopHandshakeThread)
            {
                try
                {
                    Thread.Sleep(100); // Every 100 milliseconds, to eliminate the abort

                    if (DateTime.UtcNow - lastHandshake < _handshakeInterval)
                    {
                        continue;
                    }

                    if (!IsConnected)
                    {
                        continue;
                    }

                    lastHandshake = DateTime.UtcNow;
                    DoHandshake();
                }
                catch (ThreadAbortException) // Do nothing
                {
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Handshake thread error");
                }
            }
        }

        private void DoHandshake()
        {
            try
            {
                WebSocket.Send(new NotificationMessageWrapper<Handshake>(new Handshake()).ToString());
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to send handshake! Disconnecting...");
                if (WebSocket.IsAlive)
                {
                    WebSocket.Close();                    
                }

                IsConnected = false;
                ConnectionState = ConnectionState.Disconnected;
            }
        }

        #endregion
    }

    [JsonObject]
    public class JsonMessageWrapper
    {
        [JsonProperty]
        public int Type { get; set; }

        [JsonProperty]
        public string Body { get; set; }
    }

    public interface IWebSocketContractDispatcher
    {
        void Dispatch(string message);
        string TypeName { get; }
    }

    public class WebsocketContractDispatcher<T> : IWebSocketContractDispatcher where T : Contract<T>, new()
    {
        private readonly Action<T> _callback;
        private readonly string _typeName;

        public string TypeName
        {
            get
            {
                return _typeName;
            }
        }

        public WebsocketContractDispatcher(Action<T> callback)
        {
            _callback = callback;
            _typeName = typeof(T).Name;
        }

        public void Dispatch(string message)
        {
            T contract = null;

            try
            {
                contract = JsonConvert.DeserializeObject<T>(message);
                //if (Contract<T>.IsSerialized != message.Header.IsSerialized)
                //{
                //    Logger.Debug("Contract message header serialization value does not match the contract definition.");
                //    return;
                //}

                //if (message.Header.IsSerialized)
                //{
                //    contract = Contract<T>.FromMessage(message, socketInterface.EncryptionAlgorithm);
                //}
                //else
                //{
                //    contract = new T();
                //    contract.ParseMessage(message);
                //}

                //if (!contract.Validate())
                //{
                //    Logger.Debug("Failed to validate contract of type '" + _typeName + "'");
                //    return;
                //}
            }
            catch (Exception ex)
            {
                Logger.Warn(ex, "Failed to create contract from message of type '" + _typeName + "'");
            }
            if (contract != null)
            {
                _callback(contract);
            }
        }
    }
}
